/* Parameterized Open-X
 *
 * modules:
 *
 * OpenXAssembly( 
 *
 *	arm_height,        // Passed on to OpenXArm
 *      arm_shoulder,
 *      arm_angle,
 *      arm_screw_radius,
 *
 *      bracket_taper,     // Passed on to OpenXBracket
 *      bracket_extension,
 *
 *      carraige_inset,    // Passed on to OpenXCarriage
 *      carraige_ribs )
 *
 *    This module renders the Open-X parts in assembled position along with
 *    bearings and smooth rods.  It is helpful for visualizing design changes.
 *
 *
 *  OpenXArm(height,shoulder,angle,length,screw_spacing)
 *      height   - decrease this to move the bearings upwards toward the carriage 
 *                 (default: 24.6685 mm)
 *      shoulder - decrease this to move the bearings forward towards the bar
 *                 (default: 1.5 mm)
 *      angle    - bearings are held at this angle from the horizontal 
 *                 (default: 45 degrees)
 *      length   - length of the arm in the x direction (default: 99.311716 mm)
 *      screw_spacing - distance between mounting screws (default: 24mm)
 *      screw_radius - radius of screw holes where bearings attach (default: 2.2 mm)
 *
 *      When the defaults are used, this module faithfully creates the arm as
 *      it appears in the PSU Open-X Re-design 
 *      (http://www.thingiverse.com/thing:21276)
 *
 *
 *  OpenXBracket(shoulder_radius,length,extension,slot,taper)
 *      shoulder_radius - outer radius of the bracket at the end that attaches
 *                 to the carriage (default: 5mm)
 *      length   - increase this to move the bearing farther away from the
 *                 the endof hte bracket attached to the carraige 
 *                 (default: 36.5)
 *      extension - increase this to move the pulley backward toward the bar 
 *                 (default: 2)
 *      slot     - set this to non-zero for a slot rather than a screw hole at
 *                 for mounting (default: 0)
 *      taper    - increase this to reduce the radius immediately adjacent to
 *                 the bearing (default: 1)
 *
 *    When invoked with as
 *     OpenXBracket( shoulder_radius=6.5,
 *                   length=38,
 *                   extension=3,
 *                   slot=10,
 *                   taper=0 );
 *    this module will faithly reproduce the part as it appears in the PSU 
 *    Open-X Re-design (http://www.thingiverse.com/thing:21276)
 *
 *
 *  OpenXCarriage(t,d,h,w,inset,ribs,nut_traps)
 *      t        - thickness (default: 12)
 *      d        - holediameter (default: 4.4)
 *      h        - height (default: 101)
 *      w        - width (default: 61)
 *      inset    - number of screw-head cavities to povide at top of carraige
 *                 (default: 1)
 *                 0 - none
 *                 1 - back right screw is inset, the back left is not 
 *                     (useful for an endstop flag)
 *                 2 - both screws holding the bearings with z-oriented axles
 *                     are inset
 *      nut_traps - simple or delux carraige (default: 1)
 *                 0 - thin carriage without nut traps, x-axis oriented
 *                     extruder mounting only
 *                 1 - nut traps on ribs, plus y-axis oriented option
 *      ribs     - whether or not to have ribs (and x-axis oriented extruder)
 *                 (default: 0)
 *                 0 - use a more open carraige without ribs (you'd better set
 *                     set nut_traps to 1 if you want any extruder mounting
 *                     screw holes at all)
 *                 1 - render ribs
 *
 *      When invoked OpenXCarriage(ribs=1,inset=2), this module will produce
 *      the OpenX Carriage mk2 as in http://www.thingiverse.com/thing:11519  
 * 
 *      When invoked OpenXCarriage(nut_traps=0,ribs=1,inset=2), it will produce
 *      the original OpenX Carriage as in http://www.thingiverse.com/thing:6278
 *                 
 */


OpenXAssembly();


module OpenXAssembly(
arm_height=24.6685, // ah=23.5,//22.8;
arm_shoulder=1.5, // ash=1.4,
arm_angle=45,
arm_screw_radius=2.1,
bracket_taper=1,
bracket_extension=2,
carriage_inset=1,
carriage_ribs=0,
)
{

	translate([-61/2-10-arm_shoulder,0,7-arm_height/2+5])
	rotate([0,90,0]) {
	OpenXArm(angle=arm_angle,height=arm_height,screw_radius=arm_screw_radius,shoulder=arm_shoulder);
	OpenXArmBearings(angle=arm_angle,height=arm_height);
	}

	b_angle=28.5;

	//translate([35.5,18+.25,-4])
	blen=36.5;
	brad=5;
	translate([35.5,18 -6+(blen/2-brad)*sin(b_angle)  ,-4 + 11-(+blen/2-brad)*cos(b_angle) ])
	rotate([90,b_angle,-90]) {
		OpenXBracket(taper=bracket_taper,extension=bracket_extension,shoulder_radius=brad,length=blen,slot=0);
		translate([0,-blen/2+6.5,5+bracket_extension+1])
		bearing624();
	}

	translate([35.5,-18 +6-(blen/2-brad)*sin(b_angle)  ,-4 + 11-(+blen/2-brad)*cos(b_angle) ])
	rotate([90,-b_angle,-90]) {
		OpenXBracket(taper=bracket_taper,extension=bracket_extension,shoulder_radius=brad,length=blen,slot=0);
		translate([0,-blen/2+6.5,5+bracket_extension+1])
		bearing624();
	}


	translate([0,0,12])
	rotate([180,0,180])
	OpenXCarriage(inset=carriage_inset,ribs=carriage_ribs,$fs=.1);
	//OpenXCarriage(nut_traps=0,ribs=1,inset=1);//,inset=carriage_inset,ribs=carriage_ribs,$fs=.1);
	//carriage();


	xbar_spacing=50;

	translate([-xbar_spacing/2,0,-4.5])
	rotate([90,0,0])
	cylinder(r=4,h=120,$fs=.1,center=true);

	translate([+xbar_spacing/2,0,-4.5])
	rotate([90,0,0])
	cylinder(r=4,h=120,$fs=.1,center=true);

	translate([-14.5,-45.5,-6])
	bearing624();
	translate([22,-34,6])
	rotate([0,90,0])
	bearing624(); 
	translate([-22-5,-34,6])
	rotate([0,90,0])
	bearing624(); 
	translate([22,34,6])
	rotate([0,90,0])
	bearing624(); 
	translate([-22-5,34,6])
	rotate([0,90,0])
	bearing624(); 
	translate([-14.5,45.5,-6])
	bearing624();


	module OpenXArmBearings(
	height=24.6685,
	length = 99.311716,
	angle=45,
	)
	{
		anglec=90-angle;
		translate([-height/2,0,0]) 
		translate([0,-length/2,0])
		//translate([height-10.500001/2-.5,17.04673/2-2,5-0.275])
		translate([height-10.500001/2-.5,17.04673/2-2,5-0.275])
		rotate([0, anglec, 0])
		translate([0,0,6.39948607536574-2.5])
		bearing624();

		translate([-height/2,0,0]) 
		translate([0,length/2,0])
		//translate([height-10.500001/2-.5,17.04673/2-2,5-0.275])
		translate([height-10.500001/2-.5,-17.04673/2+2,5-0.275])
		rotate([0, anglec, 0])
		translate([0,0,6.39948607536574-2.5])
		bearing624();
	}


	module bearing624($fs=.1) {
		difference() {
			cylinder(h=5,r=13/2);
			cylinder(h=5,r=4/2);
		}
	}

}



module OpenXBracket( 
	shoulder_radius=5,
	length=36.5,
	extension=2,
	slot=0,
	taper=1,
	$fs=.1)
{
	//translate([1.13,-6.37,0])
	translate([0,-length/2+6.5,0])
	difference()  
	{
		union() {
			translate([0,0,5])
			cylinder(h=extension,r1=6.5,r2=6.5-taper,$fs=.1);
			hull() {
				cylinder(h=5,r=6.5);
				translate([0 /*-0.130*/,length-6.5-shoulder_radius,0])
				cylinder(h=5,r=shoulder_radius);
			}
		}
		cylinder(h=5+extension,r=2.2);
		hull() {
			translate([0/*-0.130*/,length-6.5-shoulder_radius,0])
			cylinder(h=5,r=2.2);
			translate([0/*-0.130*/,length-6.5-shoulder_radius-slot,0])
			cylinder(h=5,r=2.2);
		}
	}
}

module OpenXArm(
	height = 24.6685,
	length = 99.311716,
	angle = 45,
	shoulder = 1.5,
	screw_spacing = 24,
	screw_radius = 2.2,
	$fs=.1
) 
{

	anglec = 90-angle;
	shoulder_length= length - (99.311716-39.9); //39.9;
	overlap = 0.1;

	translate([-height/2,0,0]) union() { mirror([0,1,0]) half(); half(); }

	module half() {
		difference() {
			//translate([0,-39.9/2-29.705858,0])
			translate([0,-length/2,0])
			union() {

				//cube([5,length/2,4]);

				translate([0,(length-shoulder_length)/2,0]) hull() {
					cube([10,shoulder_length/2,10+shoulder]);
					linear_extrude(height = 10)
					polygon([[10,-/*length*0.089966223*/ 8.9347], 
						 [10,shoulder_length/2],
						 [0,shoulder_length/2],
						 [0,0] ] );
				}

				// belt hook
				difference() {
					translate([10,length/2-5,0])
					cube([10,10/2,7.316625]);

					//translate([10,length/2-5,7.316625]), combining leads to badness...
					translate([0,length/2,0]) 
					translate([10,-5,7.316625])
					rotate([0,90,0])
					{
						linear_extrude(height=9) 
						polygon([[0,0],[3.316625,0],[0,5]]);
					}
				}

				intersection()
				{
					//translate([height-10.500001-overlap,0,0])
					translate([height-10.500001,0,0])
					cube([10.500001,15.04673+overlap,10.0]);

					translate([height-10.500001/2-.5,17.04673/2-2,5-0.275])
					rotate([0,anglec,0])
					difference() {
						//translate([0,0,-0.55])
						cube([17,17+overlap,6.39948607536574],center=true);
						// screw holes for attaching bearings
						translate([0,0,-10])
						cylinder(h=20,r=screw_radius);//,$fs=.1);
					}
					
			
				}


				difference() 
				{
					union() {
						linear_extrude(height=10)
						polygon([[height,17.04673],
							 [height-10.500001,17.04673],
							 [10,20.771158],
							 [10,29.705858]]);
						translate([height-10.500001,15.04673,0])
						cube([10.500001,2+overlap,10.0]);
					}

					translate([height-10.500001/2-.5,17.04673,5-0.275])
					rotate([0, anglec, 0])
					translate([0,0,6.39948607536574])
					cube([17,17,6.39948607536574],center=true);

				}

			}

			// screw holes for mounting to carriage
			translate([5,-screw_spacing/2,0]) cylinder(h=4,r=3.5);//,$fs=.1);
			translate([5,-screw_spacing/2,4]) cylinder(h=10+shoulder-4,r=2.2);//,$fs=.1);

		}
	}
}



module OpenXCarriage(
t = 12, //thickness
d = 4.4, //hole diameter
h = 101, //height
w = 61, //width
inset = 1,
ribs = 0,
nut_traps=1,
$fs=.1,
)
{

side_height=42;
side_thickness=10;
wall_thickness = 4;
wall_thickness_main=13;
corner_slope_outer=3;
corner_slope_inner=5;

undercarriage_thickness=4;

//flip for printing
translate([0,0,t/2]) rotate([0,180,0])
open_x_carriage();

//bump up for printing
//translate([0, 0, undercarriage_thickness/2])
//open_x_undercarriage();


//comment this out if you have prusa mendel's mendel_misc.inc or similar file
// For nut cavities, "height" is the max distance between two points on the hex.
module m4_nut_cavity(l) {
	hexagon(height=7.5,depth=l);
}

module hexagon(height, depth) {
	boxWidth=height/1.75;
		union(){
			box(boxWidth, height, depth);
			rotate([0,0,60]) box(boxWidth, height, depth);
			rotate([0,0,-60]) box(boxWidth, height, depth);
		}
}

module box(w,h,d) {
	scale ([w,h,d]) cube(1, true);
}

module corner(){
	shh=side_height/2;
	cso= corner_slope_outer;
	csi = corner_slope_inner;
	st = h/2-side_thickness;
	wt = wall_thickness;
	wtm = wall_thickness_main;

	if(nut_traps>0) {

		difference(){
		linear_extrude(height = t, center = true)
		polygon(
			points=[
				[0,0],             [0,-h/2],
				[shh-cso,-(h/2)],        [shh,-(h/2)+cso],
				[shh,-h/2+shh+cso], [w/2,-h/2+w/2],
				[w/2,0]
			]
		);
		linear_extrude(height = t+2, center = true)
		polygon(
			points=[
				[0,0],[0,-st],
				[15.5,- st],[17,-39],
				[17, -(w/2-csi)],[ w/2-wtm, -19],
				[w/2-wtm,0]
			]
		);
		}
	}
	else {
		difference(){
			linear_extrude(height = t, center = true)
			polygon(points=[[0,0],
			                [0,-h/2],
			                [18,-(h/2)],
			                [21,-(h/2)+3],
			                [21,-h/2+24],
			                [w/2,-h/2+30.5],
			                [w/2,0]]);
			linear_extrude(height = t+2, center = true)
			polygon(points=[[0,0],
			                [0,-40.5],
			                [15.5,-40.5],
			                [17,-39],
			                [17,-25.5],
			                [26.5,-19],
			                [26.5,0]]);
		}
	}
}

module rib(){
union(){
	if(nut_traps>0) {
		translate([0,25-3.5/2,0])cube([46,3.5,12], center=true);
		translate([0,25+3.5/2,0])cube([42,3.5,12], center=true);
		translate([0,25+3.5+6, t/4])cube([1.6,12, 6], center=true);
	}
	else {
		translate([0,25-3.5/2,t/2-2.5])cube([46,3.5,5], center=true);
		translate([0,25+3.5/2,t/2-2.5])cube([42,3.5,5], center=true);
		translate([0,25+3.5+6,t/2-2.5])cube([1.6,12,5], center=true);
	}
}}


module curvy_wall(){
	//translate([0,25+3.5+6-7/2, t/4])cube([(12+7)*2,12+7, 24], center=true);
	//scale([1,(25+3.5+6)/35,1])
	difference() {
	//translate([0,25+3.5+6-35/2, -t/2]) cylinder(h=24,r=35/2,center=true);
	translate([-42/2,-3.5/2+25-3.5/2,-12/2])cube([42,19+h/2-50.5,12]);
	translate([/*-46/2*/ 0,-3.5/2+25-3.5/2,-12/2])
		scale([1,(19+h/2-50.5)/35*2,1])
		cylinder(h=24,r=35/2);
	}
	

}


module mounting_holes() {
	//Wade's holes
	translate([0,-25,0]){
		cylinder(h=t+2, r=d/2, center=true);
		translate( [0,0,-4.4])  rotate([ 0, 0,0]) m4_nut_cavity(4.4);
		translate([0,0,t/2-2])cube([0.1,2*d,t]);
	}
	translate([0,25,0]){
		cylinder(h=t+2, r=d/2, center=true);
		translate( [0,0,-4.4])  rotate([ 0, 0,0]) m4_nut_cavity(4.4);
		translate([0,-2*d,t/2-2])cube([0.1,2*d,t]);
	}

	if(nut_traps>0) {
		translate([-25,0,0]){
			cylinder(h=t+2, r=d/2, center=true);
			translate( [0,0,-4.4])  rotate([ 0, 0,0]) m4_nut_cavity(4.4);
			translate([-2*d, 0, t/2-1])cube([2*d, 0.1, t]);
		}

		translate([25,0,0]){
			cylinder(h=t+2, r=d/2, center=true);
			translate( [0,0,-4.4])  rotate([ 0, 0,0]) m4_nut_cavity(4.4);
			translate([0,0,t/2-1])cube([2*d, 0.1, t]);
		}
	}
}

module open_x_carriage() {

difference(){
	
	//body
	union(){
		corner();
		mirror([0,1,0])corner();
		mirror([1,0,0])corner();
		mirror([0,1,0])mirror([1,0,0])corner();


		if(ribs>0) {
			//ribs
			rib();
			mirror([0,1,0])rib();
		}
		else {
			curvy_wall();
			mirror([0,1,0])curvy_wall();
		}
	}

	mounting_holes();

	//Vert holes
	translate([0,-45.5,0]){
		cylinder(h=t+2, r=d/2, center=true);
		translate([0,-2*d,t/2-1])cube([0.1,2*d,t]);
	}
	translate([0,45.5,0]){
		cylinder(h=t+2, r=d/2, center=true);
		translate([0,0,t/2-1])cube([0.1,2*d,t]);
	}
	translate([14.5,-45.5,0]){
		cylinder(h=t+2, r=d/2, center=true);
		translate([0,-2*d,t/2-1])cube([0.1,2*d,t]);
	}
	translate([14.5,45.5,0]){
		cylinder(h=t+2, r=d/2, center=true);
		translate([0,0,t/2-1])cube([0.1,2*d,t]);
	}
	translate([-14.5,-45.5,0])union(){
		if(inset>=2) {
			translate([0,0,-t/2-2])cylinder(h=t,r=d/2);
			translate([0,0,-t/2+2])cylinder(h=3,r1=d/2,r2=3.5);
			translate([0,0,-t/2+5])cylinder(h=t,r=3.5);
			translate([0,-2*d,t/2-1])cube([0.1,2*d,t]);
		}
		else 
			translate([0,0,-t/2])cylinder(h=t,r=d/2);
	}
	translate([-14.5,45.5,0])union(){
		if(inset>=1) {
			translate([0,0,-t/2-2])cylinder(h=t,r=d/2);
			translate([0,0,-t/2+2])cylinder(h=3,r1=d/2,r2=3.5);
			translate([0,0,-t/2+5])cylinder(h=t,r=3.5);
			translate([0,0,t/2-1])cube([0.1,2*d,t]);
		}
		else
			translate([0,0,-t/2])cylinder(h=t,r=d/2);
	}

	//Horiz holes
	translate([0,12,1]){
	translate( [0,0,0])  rotate([ 0, 90, 0])  m4_nut_cavity(w-16);
	rotate([0,90,0]) 
	cylinder(h=w+2, r=d/2, center=true);
	}

	translate([0,-12,1]){
	translate( [0,0,0])  rotate([ 0, 90, 0])  m4_nut_cavity(w-16);
	rotate([0,90,0])cylinder(h=w+2, r=d/2, center=true);
	}


	if( ribs==0 ) {
		translate([0,34,0])rotate([0,90,0]) m4_nut_cavity(61-16-11);
		translate([0,-34,0])rotate([0,90,0]) m4_nut_cavity(61-16-11);
	}

	translate([21,34,0])rotate([0,90,0])cylinder(h=t, r=d/2, center=true);
	translate([21,-34,0])rotate([0,90,0])cylinder(h=t, r=d/2, center=true);
	translate([-21,34,0])rotate([0,90,0])cylinder(h=t, r=d/2, center=true);
	translate([-21,-34,0])rotate([0,90,0])cylinder(h=t, r=d/2, center=true);	

} //difference
} //module


module open_x_undercarriage() {
	t = undercarriage_thickness;
	w=38;
	h=20;


	difference() {
		union() {
			box( w - h, h , t);
			translate( [-h/2.2, 0, 0] ) cylinder( h=t, r=h/2, center=true, $fn=8);
			translate( [h/2.2, 0, 0] ) cylinder( h=t, r=h/2, center=true, $fn=8);
		}
		//holes
		translate([ -13, 0, 0]) cylinder(h=t, r=2.2, center=true, $fn=8);

		scale( [ .5, .33, 1])
		translate(, [h/2, 0, 0])
		union() {
			box( w - h, h , t);
			translate( [-h/2.2, 0, 0] ) cylinder( h=t, r=h/2, center=true, $fn=12);
			translate( [h/2.2, 0, 0] ) cylinder( h=t, r=h/2, center=true, $fn=12);
		}
//		translate([ 12.5, 0, 0]) rotate ([0, 0, 20]) cylinder(h=t, r=2.2, center=true, $fn=8);

//		translate([ 10.5, 0, 0]) rotate ([0, 0, 20]) cylinder(h=t, r=2.2, center=true, $fn=8);
	}

}

}
