/*******************************************
 Open-face mold generator script
 Author: Jason Webb
 Website: http://jason-webb.info
********************************************/
// Model parameters
model_filename = "Sample inputs/supershape5_fixed.stl";	
model_scale = .7;				// Scale factor for the model
model_translate = [0,0,-22];		// Translate model relative to mold
model_rotate = [0,0,0];			// Rotate model relative to mold

// Mold parameters
mold_width = 65;		// Width of the mold
mold_height = 75;		// Height of the mold

// Wall parameters
wall_thickness = 1.5;	// Thickness of all walls
wall_height = 30;		// Height of the walls
walls = true;			// Walls or no walls

// Removable wall parameters
removable_wall = false;				// Do you want one wall to be removable?
removable_wall_thickness = 1;		// Thickness of the removable wall
wiggle_room = 0.25;				// Fit between wall and slot

// Base parameters
base_height = 1.5;		// Thickness of the base

generate_mold();

/*********************************************************
 Create a single, open-face mold surrounding an external STL file. 
**********************************************************/
module generate_mold() {
	difference() {
		union() {
			difference() {
				// Walls and base
				translate(v = [0,0,wall_height/2 + base_height/2])
					cube(size = [mold_width,mold_height,wall_height+base_height], center = true);
			
				// Hollow out the mold, leaving walls and a base
				translate(v = [0,0,25 + base_height])
					if(walls)
						cube(size = [mold_width - wall_thickness*2, mold_height - wall_thickness*2, 50], center = true);
					else
						cube(size = [mold_width, mold_height, 50], center = true);
			}
			
			// Scale, translate and import an external STL
			scale(v = [model_scale, model_scale, model_scale])
				translate(model_translate)
					rotate(model_rotate)
						import(model_filename);
		}

		// Create slot for removable wall
		if(removable_wall) {
			// Remove the old wall
			translate(v = [mold_width/2 - wall_thickness/2, 0, base_height + 25])
				cube(size = [wall_thickness, mold_height - wall_thickness*2, 50], center = true);

			// Create a notch for the removable wall
			translate(v = [mold_width/2 - wall_thickness, 0, base_height + 25])
				cube(size = [removable_wall_thickness + wiggle_room*2, mold_height - wall_thickness + wiggle_room*2, 50], center = true);
		}

		// Remove any geometry popping out from under the mold
		translate(v = [0,0,-50])
			cube(size = [mold_width, mold_height, 100], center = true);
	}

	// Add removable wall
	if(removable_wall)
		translate(v = [mold_width/2 + 5, 0, wall_height/2])
			cube(size = [removable_wall_thickness, mold_height - wall_thickness, wall_height], center = true);
}