/*-----------------------------------------------------------------------------------

 nozzle fan
 (c) 2014 by Stemer114 (stemer114@gmail.com)
 http://www.thingiverse.com/Stemer114/designs
 License: licensed under the 
          Creative Commons - Attribution - Share Alike license. 
          http://creativecommons.org/licenses/by-sa/3.0/

 for mendel90 x carriage mounting

 Credits: 
 
  Mendel90 reprap printer by nophead
 
  MCAD Library - Polyholes Copyright 2011 Nophead (of RepRap fame)
          It is licensed under the terms of Creative Commons Attribution 3.0 Unported.
          https://github.com/SolidCode/MCAD/blob/master/polyholes.scad

  pie.scad: This openSCAD library is part of the [dotscad](https://github.com/dotscad/dotscad)
  project.
  @copyright Chris Petersen, 2013
  @license http://creativecommons.org/licenses/LGPL/2.1/
  @license http://creativecommons.org/licenses/by-sa/3.0/
 
-----------------------------------------------------------------------------------
*/

//-----------------------------------------------------------------------------------
//libraries
//-----------------------------------------------------------------------------------
//polyholes lib is in local dir
use <MCAD/polyholes.scad>

//-----------------------------------------------------------------------------------
// View settings
// (when exporting single stls for printing, enable these one by one)
//-----------------------------------------------------------------------------------
show_debugging_cube = false;
show_fanduct = true;  
show_bracket = true;

//-----------------------------------------------------------------------------------
// printer/printing settings
// (some dimensions are tuned to printer settings)
//-----------------------------------------------------------------------------------
layer_h = 0.3;  //layer height when printing
nozzle_d = 0.4; //nozzle size (adjust for your printer)

//-----------------------------------------------------------------------------------
//global configuration settings
//-----------------------------------------------------------------------------------
de = 0.1; //epsilon param, so differences are scaled and do not become manifold

//-----------------------------------------------------------------------------------
//parametric settings (object dimensions)
//-----------------------------------------------------------------------------------
//main outlet (ring)
P0110 = 40;  //inner dia
P0111 = 60;  //outer dia
P0112 = 1.6; //wall thickness
P0113 = 10;  //duct height
P0115 = 1.6;   //duct outlet width (slot width) - not used anymore

//duct outlets configuration
P0116 = 4;   //radial length (=depth)
P0117 = 3;   //height
P0118 = 5;   //circumferential length (=width)
P0119 = 8;   //number of outlets across circumference

P0120 = 1.6; //base thickness
P0121 = 1.6; //lid thickness
P0122 = 3;   //end cap thickness (angle in degrees)
P0123 = 90;  //cutout angle in degress

//outlet connector at 9 o'clock
P0130 = 10;  //width x direction
P0131 = 30;  //width y direction
P0132 = 12;  //duct width (inside)
P0133 = 5;   //offset connector into outlet

//connecting duct 1 (nozzle outlet towards fan)
P0140 = 20;  //width x direction
P0141 = 35;  //length (y direction)
P0142 = 15;  //chamfer radius of northwest corner (hopefully for better air flow)

//fan housing
P0150 = 50;  //width x
P0151 = 50;  //width y
P0152 = 15;  //mounting offset
P0153 = 5;   //mounting bore offset from north edge
P0155 = P0113;  //height 1
P01552 = P0155+5;  //height 2 (for fan only)
P0156 = 45;  //fan cutout diameter
P0157 = 5;  //fan mounting hole offset from edge
P0158 = 3.2;  //fan mounting hole dia 
P0159 = 3.2;  //carriage mounting hole dia
P01591 = 6;   //slot hole length
//offset for fan housing relative to carriage mounting bore (middle/left)
P0160 = 40;  //x offset
P0161 = 53;  //y offset
//internal duct cutouts
P0165 = 12;  //width (x direction)
P0166 = 9;   //offset from edge
//partitioning wall in fan cutout
P0167 = 2;                  //width
P0168 = P0155 - P0120 - 2;  //height
echo("fan cutout partitioning wall width, height: ", P0167, P0168);

//connection duct 2 (duct 1 to fan housing)
P0170 = P0150;  //lower width (x)
P0171 = P0140;  //upper width (x)
P0172 = P0161-P0153-P0141+P0131/2;     //length (y direction) needs to be calculated from duct offsets
P0173 = P0160+P0150/2-P0111/2-P0130+P0133-P0140;     //left offset of upper side, needs to be calculated as well

//mounting bracket
P0180 = 50;  //mounting holes offset
P0181 = P0150/2-P0157;  //offset from left to middle hole
P0182 = 3.2;  //hole dia
P01821 = 6;   //slot hole length
P0183 = 8;  //width (y)
P0184 = 3;  //small thickness
P0185 = 15; //large thickness
P0186 = P0157;  //hole offset from edge

//-----------------------------------------------------------------------------------
// main rendering
//-----------------------------------------------------------------------------------

assembly();

//-----------------------------------------------------------------------------------
// modules
//-----------------------------------------------------------------------------------

module assembly()
{
    union()
    {
        if (show_debugging_cube) {
            translate([-50, 0, 0])
                % cube(10);
        }

        if (show_fanduct) {
            fanduct();
        }  //end if

        if (show_bracket) {
            translate([-P0160, -P0161, P0113+5])
            bracket();
        }  //end if

        }  //end union
}  //end module


module fanduct()
{
    difference()
    {
        union()
        {
            outlet();

            translate([-P0111/2-P0130+P0133-P0140, -P0141+P0131/2, 0])
            connecting_duct1();

            translate([-P0160-P0150/2, -P0161+P0153, 0])
                connecting_duct2();

            translate([-P0160, -P0161, 0])
                fan_housing();


        }  //union 1

        union()
        {
 
        }  //union 2
    }  //difference
}

module outlet() {
    difference()
    {
        union()
        {
            //rotate([0, 0, 180+45])
                outlet_base();

        }  //union 1

        union()
        {

        }  //union 2

    }  //difference

}

module outlet_base() {

    difference()
    {
        union()
        {
            //base ring with angle cutout
            rotate([0, 0, 180+45])
                ring_sector(angle=P0123, od=P0111, id=P0110, h=P0113);

            //connector body
            translate([-P0130-P0111/2+P0133, -P0131/2, 0])
                cube([P0130, P0131, P0113]);

        }

        union() {

            //inner cutout
            translate([0, 0, P0120])
                rotate([0, 0, -P0122])
                rotate([0, 0, 180+45])
                ring_sector(angle=P0123+2*P0122, od=P0111-2*P0112, id=P0110+2*P0112, h=P0113-P0120-P0121);

            //outlet cutout
            translate([0, 0, -de])
                rotate([0, 0, P0123+P0122])
                //rotate([0, 0, -P0122])
                rotate([0, 0, 180+45])
                {
                    /*
                       ring_sector(angle=90+2*P0122, od=P0110+2*P0112+2*P0115, id=P0110+2*P0112, h=P0120+2*de);
                     */

                    //multiple outlet cutouts
                    //translate([P0110/2-de, 0, -de])
                    //translate([0, -P0118/2, 0]) cube([P0116+de, P0118, P0117+de]);
                    for ( i = [0 : P0119-1] )
                    {
                        rotate( (0.5*(360-P0123-2*P0122)/P0119) + i * (360-P0123-2*P0122) / P0119, [0, 0, 1])
                            translate([P0110/2-2*de, 0, 0])
                            translate([0, -P0118/2, 0]) cube([P0116+2*de, P0118, P0117+de]);
                    }
                }
                


            //connector inside cutouts (2 pieces to make bridges not too long)
            translate([-P0130-P0111/2+P0133-de, 0, P0120])
                //translate([0, 0, 0])
            {
                translate([0, -P0131/2+P0112, 0])
                    cube([P0130+2*de, P0132, P0113-P0120-P0121]);
                translate([0, -P0132+P0131/2-P0112, 0])
                    cube([P0130+2*de, P0132, P0113-P0120-P0121]);
            }

        }  //union 2
    }

}

//duct 1 connect nozzle outlet with duct 2
module connecting_duct1() {
    difference()
    {
        union()
        {
            cube([P0140, P0141, P0113]);

        }  //union 1

        union()
        {
            //main duct cutout
            translate([P0112, -de, P0120])
            {
                //base cube
                //cube([P0140-2*P0112, P0141-P0112+de, P0113-P0120-P0121]);
                //chamfer upper corner
                hull()
                {
                    cube([de, de, P0113-P0120-P0121]);
                    translate([P0140-2*P0112, 0, 0]) cube([de, de, P0113-P0120-P0121]);
                    translate([P0142/2, P0141-P0112+de-P0142/2, 0]) polyhole(d=P0142, h=P0113-P0120-P0121);
                    translate([P0140-2*P0112, P0141-P0112+de, 0]) cube([de, de, P0113-P0120-P0121]);
                }
            }

            //connector cutout towards nozzle
            translate([P0140-P0112-de, P0141-P0131+P0112, P0120])
                cube([P0112+2*de, P0131-2*P0112, P0113-P0120-P0121]);



        }  //union 2

    }  //difference

}


//duct 2 - connecting duct 1 with
//fan housing
module connecting_duct2() {
    difference()
    {
        union()
        {
            //base body
            hull() {
                cube([P0170, de, P0155]);
                translate([P0173, P0172, 0])
                    cube([P0171, de, P0155]);
            }

        }  //union 1

        union()
        {
            //internal cutout left
            hull() {
                translate([P0112, -de, P0120])
                    cube([(P0170-4*P0112)/2, de, P0155-P0120-P0121]);
                translate([P0173+P0112, P0172+de, P0120])
                    cube([(P0171-4*P0112)/2, de, P0155-P0120-P0121]);
            }
            //internal cutout right
            hull() {
                translate([3*P0112+(P0170-4*P0112)/2, -de, P0120])
                    cube([(P0170-4*P0112)/2, de, P0155-P0120-P0121]);
                translate([P0173+3*P0112+(P0171-4*P0112)/2, P0172+de, P0120])
                    cube([(P0171-4*P0112)/2, de, P0155-P0120-P0121]);
            }



        }  //union 2

    }  //difference

}



//fan housing with mounting holes for mounting bracket
module fan_housing() {
    //translate the fan housing so the reference point
    //(the central mounting bore)
    //is at the origin
    translate([-P0150/2, -P0151-P0152+P0153, 0])
    difference()
    {
        union()
        {
            //base body
            cube([P0150, P0151+P0152, P0155]);
            //additional height 2 for fan cutout only
            translate([0, 0, 0])
                cube([P0150, P0151, P01552]);

        }  //union 1

        union()
        {
            //fan cutout
            translate([P0150/2, P0151/2, P0120])
                difference()
                {
                    //cutout base cylinder
                    polyhole(h=P01552-P0120+de, d=P0156);
                    //minus partition walls
                    //center
                    translate([0, 0, P0168/2-de])cube([P0167, P0156+4*de, P0168+2*de], center=true);
                    //left (half length)
                    translate([-P0156/4, -P0156/2-2*de, P0168/2-de])cube([P0167, P0156+4*de, P0168+2*de], center=true);
                    //right
                    translate([P0156/4, -P0156/2-2*de, P0168/2-de])cube([P0167, P0156+4*de, P0168+2*de], center=true);
                }

            //mounting holes
            translate([P0157, P0157, -de]) polyhole(h=P01552+2*de, d=P0158);
            translate([P0150-P0157, P0157, -de]) polyhole(h=P01552+2*de, d=P0158);
            translate([P0157, P0151-P0157, -de]) polyhole(h=P01552+2*de, d=P0158);
            translate([P0150-P0157, P0151-P0157, -de]) polyhole(h=P01552+2*de, d=P0158);

            //carriage mounting holes
            //center
            translate([P0150/2, P0151+P0152-P0153, -de]) 
                hull() {
                    polyhole(h=P0155+2*de, d=P0159);
                    translate([0, -P01591, 0]) polyhole(h=P0155+2*de, d=P0159);
                }
            //east side
            translate([P0150-P0157, P0151+P0152-P0153, -de]) 
                hull() {
                    polyhole(h=P0155+2*de, d=P0159);
                    translate([0, -P01591, 0]) polyhole(h=P0155+2*de, d=P0159);
                }

            //duct cutouts
            //left
            translate([P0166, P0151+P0152+de-P0152-P0151/4, P0120])
                cube([P0165, P0152+P0151/4+de, P0155-P0120-P0121]);
            //right
            translate([P0150-P0166-P0165, P0151+P0152+de-P0152-P0151/4, P0120])
                cube([P0165, P0152+P0151/2+de, P0155-P0120-P0121]);


        }  //union 2

    }  //difference

}

//bracket (for mounting to x-carriage bracket)
module bracket() {
    //align with center of left hole at origin
    translate([-P0186, -P0183/2, 0])
    difference()
    {
        union()
        {
            //upper plate (thinner)
            translate([0, 0, P0185-P0184])
            cube([P0180+2*P0186+P01821, P0183, P0184]);
            //lower plate (shorter and thicker)
            cube([P0181+2*P0186+P01821, P0183, P0185]);

        }  //union 1

        union()
        {
            //mounting holes
            //left
            translate([P0186, P0183/2, -de]) 
                hull() {
                polyhole(h=P0185+2*de, d=P0182);
                translate([P01821, 0, 0]) polyhole(h=P0185+2*de, d=P0182);
                }
            //middle
            translate([P0186+P0181, P0183/2, -de]) 
                hull() {
                polyhole(h=P0185+2*de, d=P0182);
                translate([P01821, 0, 0]) polyhole(h=P0185+2*de, d=P0182);
                }
            //right
            translate([P0186+P0180, P0183/2, -de]) 
                hull() {
                polyhole(h=P0185+2*de, d=P0182);
                translate([P01821, 0, 0]) polyhole(h=P0185+2*de, d=P0182);
                }




        }  //union 2

    }  //difference

}



//utility functions

//a ring with heigth h, outside diameter od and inside diameter id
module ring(
        h=1,
        od = 10,
        id = 5
        ) 
{
    difference() {
        cylinder(h=h, r=od/2);
        translate([0, 0, -de])
            cylinder(h=h+2*de, r=id/2);
    }
}

/* square tube
   wx, wy: width in x,y direction
   h: height/length in z direction
   t: wall thickness
 */
module squaretube(
        h=10,
        wx=10,
        wy=10,
        t=2
        )
{

    translate([0, 0, h/2])
    difference() {
        cube([wx, wy, h], center=true);
        translate([0, 0, 0])
            cube([wx-2*t, wy-2*t, h+2*de], center=true);
    }
}

/* round tube
   od: outside diameter
   t:  wall thickness
   h: height/length in z direction
*/
module roundtube(
        h=10,
        od=10,
        t=2
        )
{
    ring(h, od, od-2*t);
}

module ring_sector(
        angle=90,
        od=100,
        id=80,
        h=10
        )
{

    difference()
    {
        //the full ring
        ring(h=h, od=od, id=id);
        translate([0, 0, -de])
            pie(radius=od/2+de, angle=angle, height=h+2*de);
    }
}

/**
 * pie.scad
 *
 * Use this module to generate a pie- or pizza- slice shape, which is particularly useful
 * in combination with `difference()` and `intersection()` to render shapes that extend a
 * certain number of degrees around or within a circle.
 *
 * This openSCAD library is part of the [dotscad](https://github.com/dotscad/dotscad)
 * project.
 *
 * @copyright Chris Petersen, 2013
 * @license http://creativecommons.org/licenses/LGPL/2.1/
 * @license http://creativecommons.org/licenses/by-sa/3.0/
 *
 * @see http://www.thingiverse.com/thing:109467
 * @source https://github.com/dotscad/dotscad/blob/master/pie.scad
 *
 * @param float radius Radius of the pie
 * @param float angle Angle (size) of the pie to slice
 * @param float height Height (thickness) of the pie
 * @param float spin Angle to spin the slice on the Z axis
 */
module pie(radius, angle, height, spin=0) {
    // Negative angles shift direction of rotation
    clockwise = (angle < 0) ? true : false;
    // Support angles < 0 and > 360
    normalized_angle = abs((angle % 360 != 0) ? angle % 360 : angle % 360 + 360);
    // Select rotation direction
    rotation = clockwise ? [0, 180 - normalized_angle] : [180, normalized_angle];
    // Render
    if (angle != 0) {
        rotate([0,0,spin]) linear_extrude(height=height)
            difference() {
                circle(radius);
                if (normalized_angle < 180) {
                    union() for(a = rotation)
                        rotate(a) translate([-radius, 0, 0]) square(radius * 2);
                }
                else if (normalized_angle != 360) {
                    intersection_for(a = rotation)
                        rotate(a) translate([-radius, 0, 0]) square(radius * 2);
                }
            }
    }
}



