/*
 The Open Academic Robot Kit
 http://www.oarkit.org
 info@oarkit.org
 Copyright (C) 2014 Raymond Sheh <raymond@oarkit.org>
 Version 2014-07-16 BETA
 
 NOTE: This program is experimental. Use at your own risk. Please visit http://www.oarkit.org for updates. 
 
 This program for the Arduino Pro Micro 5V (or compatible) reads 16 values from 
 an nRF24 radio tranceiver, performs a computation to mix the values to one or more 
 servos with specify a scaling and offset, and transmits the resulting commands to 
 Dynamixel AX-12 series servos attached to the Tx line of the microcontroller's USART. 
 
 This program was developed for controlling a robot manipulator using a master-slave system. 
 The master arm has potentiometers fitted to each joint. Each potentiometer divides the 5V 
 supply; the resulting voltage corresponds to the angle of the potentiometer. Each potentiometer 
 feeds this voltage into a pin on the Arduino. This program converts that voltage to a normalised 
 angle relative to a zero point and sends it to the robot. 
 
 For an example of a robot that can be controlled using this program, please visit 
 http://www.oarkit.org and follow the links for the master-slave controller. 
 
 Note that the settings and comments in this program are for the 5DOF Arm attached to the 
 Six Wheeled Wonder robot and controlled using the Master-Slave arm and joysticks. However, 
 this program may be easily modified for arbitrary robot configurations by changing the 
 values in the mixing table and other settings at the start of the program. 
 
 This program relies on the ArbotiX AX12 library available from https://code.google.com/p/arbotix/ 
 and the nRF24 library available from http://maniacbug.github.com/RF24 .
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 version 2 as published by the Free Software Foundation.
 */
 
#include <SPI.h>
#include "nRF24L01.h"
#include "RF24.h"

#include <ax12.h>

RF24 radio(9,10); 
const uint64_t pipes[2] = { 0xF0F0F0F0E1LL, 0xF0F0F0F0D2LL }; 

#define NUM_INPUTS 16
#define NUM_MOTORS 15

AX12 motor[NUM_MOTORS]; 



// Motor index is a convenience index we use that starts from 0 and 
// consecutively refers to each motor, specific to this program. 
// Note that this is NOT the input index (that we're getting from the sender) - 
// that relationship is established in the mixing table. 
// This allows us to swap motor IDs or input IDs around without needing 
// to change things in more than one place. 
int motorIndexToAX12ID[] = {
13, // 0: Arm Base tilt 1
14, // 1: Arm Base tilt 2
15, // 2: Arm Elbow
17, // 3: Arm Wrist tilt
12, // 4: Arm Wrist yaw
18, // 5: Arm Wrist roll
6,  // 6: Back axel (plus arm slew)
8,  // 7: Front axel
7,  // 8: Front suspension
4,  // 9: Back Right wheel
3,  // 10: Center Right wheel
10, // 11: Front Right wheel
5,  // 12: Back Left wheel
2,  // 13: Center Left wheel
9  // 14: Front Left wheel
}; 

// Motors in either position mode or continuous mode. 
// Subsequent parameters will refer to either position or velocity 
// based on this table. 
boolean motorIsContinuous[] = {
false, false, false, false, false, false, // Arm
false, false, false, // Axles and suspension
true, true, true, true, true, true // Wheels
}; 

// Zero offset. 
// Positions are in centidegrees (cd) zero centered (so full ccw is -1800 and full cw is 1800). 
// Velocities are in millis (thousandths of full speed), -1000 to 1000. 
int motorOffset[] = {
0, 0, 450, -450, 0, 0, // Arm - note that two of the servos are 45 degrees out so the dead spots end up in sensible places. 
0, 0, 0,               // Axels and suspensions
0, 0, 0, 0, 0, 0       // Wheels
}; 

// Motor low and high limits. 
// Positions are in cd zero centered. 
// Velocities are in millis, -1000 to 1000. 
int motorLowLimit[] = {
-900, -900, -1500, -1500, -1050, -1800, // Arm
-200, -200, // Axels
-300, // Suspension
-1000, -1000, -1000, -1000, -1000, -1000 // Wheels
}; 

int motorHighLimit[] = {
900, 900, 1350, 1350, 900, 1800, // Arm
200, 200, // Axles
300, // Suspension
1000, 1000, 1000, 1000, 1000, 1000 // Wheels
}; 

// Motor velocity to use for motors that are in position mode, in millis (0 = no power, 1000 = full speed). 
int motorVel[] = {
200, 200, 200, 200, 200, 200, // Arm
200, 200, 200, // Axels and suspension
0, 0, 0, 0, 0, 0 // Wheels (ignored as they're already in velocity mode)
}; 

// A safe position for all the motors to be in. Note that there is no guarantee of the 
// order in which this is achieved! Also note that the unit is cd zero centered
// (so full ccw is -1800 and full cw is 1800). 
int motorSafePosition[] = {
0, 0, 0, 0, 0, 0, // Arm
0, 0, 0, // Axels and suspension
0, 0, 0, 0, 0, 0 // Wheels (ignored)
}; 

// Mixing table that, for each input, gives the amount to be added 
// to each motor index command. Note: 
// - Both inputs and motor outputs for position mode are normalised to one cd (full ccw is -1800, center is 0, full cw is 1800). 
// - Both inputs and motor outputs for velocity mode are normalised to one milli (full speed ccw is -1000, stopped is 0, full speed cw is 1000). 
// - These proportions are in millis (ie. for 1:1, put 1000). 
// - You can reverse directions by using a negative value here. 
// - You can also have several inputs go to the same motor (where the result is added). 
// - You can also have several motors come from the same input (where the results are replicated). 
// - Values will clip if necessary. 
int inputToMotorIndexMixer[NUM_INPUTS][NUM_MOTORS] = 
  //                                    Motor index                                            Control input
{ //   0     1     2     3     4     5     6     7     8     9    10    11    12    13    14
  {    0,    0,    0,    0,    0,    0, 1000,-1000,    0,    0,    0,    0,    0,    0,    0}, // 0: Steering (tractor, also base slew) - front and back oppose. 
  {-1000, 1000,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0}, // 1: Base joint - two opposing motors
  {    0,    0,-1000,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0}, // 2: Elbow
  {    0,    0,    0,-1000,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0}, // 3: Wrist tilt
  {    0,    0,    0,    0, 1000,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0}, // 4: Wrist yaw
  {    0,    0,    0,    0,    0, 1000,    0,    0,    0,    0,    0,    0,    0,    0,    0}, // 5: Wrist roll
  {    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0}, // 6: (unused)
  {    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0}, // 7: (unused)
  {    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0}, // 8: (unused)
  {    0,    0,    0,    0,    0,    0,    0,    0,    0,-1000,-1000,-1000, 1000, 1000, 1000}, // 9: Throttle
  {    0,    0,    0,    0,    0,    0,    0,    0,    0,-1000,-1000,-1000,-1000,-1000,-1000}, // 10: Steering (skid)
  {    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0}, // 11: Safety (<512 = off, >512 = on) - note we handle this specially. 
  {    0,    0,    0,    0,    0,    0,    0,    0, 1000,    0,    0,    0,    0,    0,    0}, // 12: Suspension
  {    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0}, // 13: (unused)
  {    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0}, // 14: (unused)
  {    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0}  // 15: (unused)
}; 

int safetyControlInputIndex = 11; // Safety switch index. 
boolean driveMotorsOn = false; // This is set based on the safety control input. 

int watchdogLimit = 100; 
int watchdog = 0; 


void setup(void)
{

  // Set up nRF24 radio module. 
  radio.begin(); 
  radio.setRetries(15,15); 
  radio.setPayloadSize(32); 
  radio.setDataRate(RF24_250KBPS); 
  
  radio.openWritingPipe(pipes[1]); 
  radio.openReadingPipe(1,pipes[0]); 
  radio.startListening(); 
  
  // Initialise AX-12 motors. 
  for (int i = 0; i < NUM_MOTORS; i ++)
  {
    motor[i] = AX12(); 
  }

  AX12::init (1000000); 
  
  // Set up motor objects, set rotation mode and ensure they're off. 
  for (int i = 0; i < NUM_MOTORS; i ++)
  {
    motor[i].id = motorIndexToAX12ID[i]; 
    motor[i].setEndlessTurnMode(motorIsContinuous[i]); 
    motorsOff(); 
  }
}

// Update the output values for all of the motors based on the 
// current input and the mixing table. 
void computeMotorPositions(int * inInput, int * outMotorAX12)
{
  // Position inputs are in cd, zero centered (-1800 to 1800). 
  // Velocity inputs are in milli from -1000 to 1000. 
  for (int i = 0; i < NUM_MOTORS; i ++)
  {
    int outTemp = motorOffset[i]; 
    
    for (int j = 0; j < NUM_INPUTS; j ++)
    {
      outTemp += (int)((long)(inInput[j])*(long)(inputToMotorIndexMixer[j][i])/1000L); 
    }
    
    outTemp = min(motorHighLimit[i], max(motorLowLimit[i], outTemp)); 

    if (motorIsContinuous[i])
    {
      outMotorAX12[i] = milliToAX12Vel(outTemp); 
    }
    else
    {
      outMotorAX12[i] = cdToAX12Pos(outTemp); 
    }
  }
  

  
}

void loop(void)
{
  static boolean gotsomethinglast = false; // For watchdog. 
  static int latestInput[NUM_INPUTS]; 
  static int motorOutputAX12[NUM_MOTORS]; 
  
  if (radio.available())
  {
    gotsomethinglast = true; 
    watchdog = 0; 

    bool done = false; 
    while (!done)
    {
      done = radio.read(&latestInput, NUM_INPUTS*sizeof(int)); 
    }
    
    // Check safety. 
    driveMotorsOn = (latestInput[safetyControlInputIndex] > 512); 
    
    computeMotorPositions(latestInput, motorOutputAX12); 
    
    for (int i = 0; i < NUM_MOTORS; i ++)
    {
      if (motorIsContinuous[i])
      {
        if (driveMotorsOn)
        {
          motor[i].setVel(motorOutputAX12[i]); 
        }
        else
        {
          motor[i].setVel(0); 
        }
      }
      else
      {
        motor[i].setVel(milliToAX12Vel(motorVel[i])); 
        motor[i].setPos(motorOutputAX12[i]); 
      }
    }
  }
  else
  {
    gotsomethinglast = false; 
    watchdog ++; 
    if (watchdog > watchdogLimit)
    {
      motorsOff(); 
    }
  }

}


void motorsOff()
{
  for (int i = 0; i < NUM_MOTORS; i ++)
  {
    // Switch off motors. 
    motor[i].setVel(0); 
  }
}


// Calculate the AX-12 position (Robotis protocol) given an angle 
// in centidegrees (10'th of a degree). 
int cdToAX12Pos(int inCD)
{
  // CD is from -1800 to 1800 for a full circle. 
  // AX12 is 0 to 1024 according to the following: 
  // AX12      CD
  //    0   -1500
  //  512       0
  // 1024    1500
  
  // Clip to AX12 limits. 
  if (inCD > 1500) inCD = 1500; 
  if (inCD < -1500) inCD = -1500; 
  return (int)(((long)(inCD + 1500)*1024l)/3000l);
}


// Input is millis -1000 to 1000 for max negative to max positive velocity. 
// Output is Robotis protocol, 0-1023 for positive velocity, 1024-2047 for negative velocity. 
int milliToAX12Vel(int inMilli)
{
  long retval; 
  if (inMilli >= 0)
  {
    retval = 128L*(long)inMilli/125L; 
    if (retval > 1023) retval = 1023; 
    if (retval < 0) retval = 0; 
  }
  else
  {
    retval = -128L*(long)inMilli/125L + 1024L; 
    if (retval > 2047) retval = 2047; 
    if (retval < 1024) retval = 1024; 
  }
  return (int)retval; 
}


