/*
 The Open Academic Robot Kit
 http://www.oarkit.org
 info@oarkit.org
 Copyright (C) 2014 Raymond Sheh <raymond@oarkit.org>
 Version 2014-07-16 BETA
 
 NOTE: This program is experimental. Use at your own risk. Please visit http://www.oarkit.org for updates. 
 
 This program for the Arduino Mega 2560 (or compatible) reads analog values from 
 the 16 analog inputs, performs a computation to specify a scaling and offset, and 
 transmits the resulting values through an nRF24 radio tranceiver. 
 
 This program was developed for controlling a robot manipulator using a master-slave system. 
 The master arm has potentiometers fitted to each joint. Each potentiometer divides the 5V 
 supply; the resulting voltage corresponds to the angle of the potentiometer. Each potentiometer 
 feeds this voltage into a pin on the Arduino. This program converts that voltage to a normalised 
 angle relative to a zero point and sends it to the robot. 
 
 For an example of a robot that can be controlled using this program, please visit 
 http://www.oarkit.org and follow the links for the master-slave controller. 
 
 This program relies on the nRF24 library available from http://maniacbug.github.com/RF24 .
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 version 2 as published by the Free Software Foundation.
 */



/**
 * @file printf.h
 *
 * Setup necessary to direct stdout to the Arduino Serial library, which
 * enables 'printf'
 */
#ifndef __PRINTF_H__
#define __PRINTF_H__

#ifdef ARDUINO

int serial_putc( char c, FILE * ) 
{
  Serial.write( c );

  return c;
} 

void printf_begin(void)
{
  fdevopen( &serial_putc, 0 );
}

#else
#error This example is only for use on Arduino.
#endif // ARDUINO

#endif // __PRINTF_H__

/**
 * Transmitting node - Arduino Mega 2560. 
 * Reads values from all 16 analog inputs and transmits them as centidegrees (cd), centered at zero (-1800 to 1800). 
 */

#include <SPI.h>
#include "nRF24L01.h"
#include "RF24.h"

RF24 radio(48,49); 

const uint64_t pipes[2] = { 
  0xF0F0F0F0E1LL, 0xF0F0F0F0D2LL }; 


void setup(void)
{
  Serial.begin(57600); 
  Serial.println("Transmitter."); 
  printf_begin(); 
  radio.begin(); 
  radio.setRetries(15, 15); 
  radio.setPayloadSize(32); 
  radio.setDataRate(RF24_250KBPS); 

  radio.openWritingPipe(pipes[0]); 
  radio.openReadingPipe(1,pipes[1]); 


  radio.printDetails(); 
}

byte analogPins[] = {
  A0, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, A11, A12, A13, A14, A15}; 
#define NUM_ANALOG_PINS 16

// The conversion from analog to centidegrees - scalar and offset. 
// Note the scalar is in millis (the real number x 1000). This 
// allows us to have better precision while staying with integer math. 
// The idea is: 
// outCD[i] = (analogToCDMilliScalar[i]*(inAnalog[i] - analogToCDOffset[i]))/1000
// (Note that the actual calculation is slightly different to ensure that we don't run into 
// integer overflow.)
int analogToCDOffset[] = {
  487, 525, 330, 340, 512, 512, 0, 0, 0, 511, 514, 0, 502, 0, 0, 0}; 
int analogToCDMilliScalar[] = {
  3000, 2571, 2600, -2636, 2667, 2600, 0, 0, 0, 10000, 10000, 1000, 2000, 0, 0, 0}; 



void loop(void)
{

  int i; 

  int outVal[NUM_ANALOG_PINS]; 

  for (i = 0; i < NUM_ANALOG_PINS; i ++)
  {
    outVal[i] = analogToCD(analogRead(analogPins[i]), i); 
  }

  radio.stopListening(); 

  bool ok = radio.write(&outVal, NUM_ANALOG_PINS*sizeof(int)); 

  radio.startListening(); 

  if (ok)
  {
    printf("Sent: "); 
  }
  else 
  {
    printf("Send failed: "); 
  }


  for (i = 0; i < NUM_ANALOG_PINS; i ++)
  {
    printf("%d,\t", outVal[i]); 
  }
  printf("\n");   

  delay(10); 
}


int analogToCD(int inAnalog, int i)
{
  return (int)((long)analogToCDMilliScalar[i]*((long)inAnalog - (long)analogToCDOffset[i])/1000L); 
}

