// ------------------------------------
// global sizes we need
// ------------------------------------

include <triangles.scad>

build_label = "inosion - v1.3";

// helpers
module move(x=0,y=0,z=0,rx=0,ry=0,rz=0) { translate([x,y,z])rotate([rx,ry,rz]) children(); }
 
// outer dimensions of the box, minus any locks and screw tabs
boxWalls  = 8;
baseDepth = 8;
boxLength = 162;
boxWidth  = 118;
boxHeight = 35;

// inside wall thickness of the "tray"
// a honeycomb lattice is at the bottom, 
// with a gap, and the drive sits on top of that gap/ridge
innerShelf=3;

subtrayHeight= baseDepth - 2.5; // makes the honeycomb 2.5mm thick
subtrayHeightOffset = baseDepth - subtrayHeight;
subtrayXYOffset=innerShelf + boxWalls;
subtrayLength=boxLength - subtrayXYOffset - innerShelf;
subtrayWidth=boxWidth - ((innerShelf + boxWalls) * 2);


// the trapezoidal locks
lockBottom  = 12;
lockTop     = 15;
lockHeight  = 4.5;
lockDepth   = boxWalls;

// where do we position these locks ? 
lockPos1Center = 40;
lockPos2Center = boxLength - 40;
lockPos1 = 40 - (lockTop / 2);
lockPos2 = boxLength - 40 - (lockTop / 2);

// screwtabs
screwTabSize      = 12;
screwTabThickness = 4;

// allowance (recess) for a screwhead
screwHeadRadius     = 4.5; // 9mm
screwHeadShaftRadius    = 3.5/2;
screwHoleRadius   = 1.1;
screwHoleDepth    = 5;
screwMaterial     = 2.5; // this is "how thick is a piece" that the screw holds down.
                       // caters for depth of a screw. Bolts are approx 6mm long

// where we need to join pieces together, we need to "over" or
// underprint a part, in mm, that value is... 
// this figure is 1/2'ved  (so .3mm around)
join_mm = 0.6;
pocket_join_mm = 1.2; // they needed a little more room

// radius for rounded edges
cornerRound = 2;

// --------------------------------
// honeycomb lattice
// --------------------------------
// number of rows and columns,
// beware that some hexagonal cells are clipped
// at rectangular box boundaries, so 
// the total number of cells will be
// smaller than rows * columns
rows          = 8;
columns       = 11;
inner_walls   = 3; // thickness between honeycomb cells

// cell step is hole size between
// opposite hexagon walls plus inner wall thickness
cell_step     = 15;

module honeycomb_row(cells, cell_step, inner_walls, height) {
    move( z = -height/2 )
    linear_extrude(height=height) {
        for (j = [0 : cells - 1]) {
            move(x = j * cell_step, rz = 30)
            circle(r=(cell_step - inner_walls)/sqrt(3),$fn=6);
        }
    }
}


module full_honeycomb(rows, columns, cell_step, inner_walls, height) {
    for (i = [0 : rows - 1]) {
        move( x = (i % 2) * cell_step / 2
             ,y = cell_step * i * sqrt(3) / 2)
          honeycomb_row(columns, cell_step, inner_walls, height);
    }
}

module clipped_honeycomb(rows, columns, cell_step, inner_walls, height) {
    intersection() {
        // cube([(columns - 1) * cell_step, (rows - 1) * cell_step * sqrt(3) / 2, height]);
        cube([subtrayLength, subtrayWidth, height]);
        move (x=-1.5, y=2) {
          full_honeycomb(rows, columns, cell_step, inner_walls, 3 * height);
        }
    }
}


// --------------------------------
// locking trapezoids
// --------------------------------

// Locking piece is a Trapezoid
// so we define, top length, bottom length, height and depth
// they have a rectangle bottom and top.


module trapezoid(width_base, width_top,thickness, height) {

  move(rx=90, , y=thickness) linear_extrude(height = thickness) polygon(points=[[0,0],[width_base,0],[width_base-(width_base-width_top)/2,height],[(width_base-width_top)/2,height]], paths=[[0,1,2,3]]); 
  
}


// this is a cool shape (rotate a trapezoid in a circle)
module trapezoid_90_angle(width_base, width_top,height) {
  rotate_extrude(angle = 90, convexity = 10, $fn=100) 
  move(rz=90, y=(width_top - width_base ) /2, x=height+0.0000001) polygon(points=[[0,0],[width_base,0],[width_base-(width_base-width_top)/2,height],[(width_base-width_top)/2,height]], paths=[[0,1,2,3]]); 
}

//!trapezoid_90_angle(5.00001,12,6);
// --------------------------------
// a simple vent we will use around the "outside" (means less plastic used)
// --------------------------------
module vertBars(number, width=3, height=20, depth = boxWalls, distanceApart=6 ) {
  for (i = [0 : number -1]) {
    move( x = i * distanceApart ) { 
      cube([width, depth, height]);
    }
  }
}


module tray_base() {
  difference() {
     
      // full box
     union() {
       move(x=cornerRound) cube([boxLength - cornerRound - cornerRound,boxWidth,boxHeight]);
       
       hull() {
         move( x = cornerRound, y = cornerRound) cylinder(r=cornerRound,h=boxHeight, $fn=20);
         move( x = cornerRound, y = boxWidth - cornerRound) cylinder(r=cornerRound,h=boxHeight, $fn=20);
       }
       
       hull() {
         move( x = boxLength - cornerRound, y = cornerRound) cylinder(r=cornerRound,h=boxHeight, $fn=20);
         move( x = boxLength - cornerRound, y = boxWidth - cornerRound) cylinder(r=cornerRound,h=boxHeight, $fn=20);
       }
     }
      
     // drive cavity
     move(x=boxWalls,y=boxWalls,z=boxWalls) {
       cube([boxLength-boxWalls,boxWidth-(2*boxWalls),boxHeight-boxWalls]);
     };
     
     // inner sub tray
     move(x=subtrayXYOffset,y=subtrayXYOffset,z=subtrayHeightOffset) {
       cube([subtrayLength,subtrayWidth,subtrayHeight]);
     };
     
     // end door hinge recess (7mm x 6mm)
     recessBX = 6;
     recessBY = 7;
     move(x=boxLength - recessBY, y=boxWidth - boxWalls, z=baseDepth)
         cube([ recessBY , recessBX, boxHeight - baseDepth]);
     
     // hole for screw for end door
     screwHole=4;
     move(x=boxLength - (recessBY/2) , y=boxWidth - boxWalls + (recessBX/2) )
       union() {
         cylinder(r=screwHole/2, h=baseDepth);
         cylinder(r=3, h=5);
       }


     // end door latch recess
     recessAX = 3;
     recessAY = 7;
     move(x=boxLength - recessAY, y=boxWalls - recessAX, z=baseDepth)
         cube([ recessAY , recessAX, boxHeight - baseDepth - 4]);
     
  }
  
  move(x = boxLength - 3, y = boxWalls - 3, z= boxHeight - 10) Right_Angled_Triangle(a=2, b=3, height=6, centerXYZ=[false,false,false]);

};

module inside_vent_track(length=boxLength - boxWalls - boxWalls) {
  
     ventSize=12;
     innerHeight=5;
     depth=6;
     move(rz=270,ry=90, y=depth,z=ventSize) trapezoid(ventSize, innerHeight, length, depth);  
}


module tray_base_no_interlock() {
    difference() {
        
        tray_base(); // this is a shape with no cut-aways
        
        // add (remove) the honeycomb to bottom
        translate([subtrayXYOffset,subtrayXYOffset,0]) {
          clipped_honeycomb(rows, columns, cell_step, inner_walls, subtrayHeightOffset);
        };
        
        // make a rear inset on the back
        rearInset = 2; // mm
        move( y=2,z=2 ) {
          cube([  rearInset, boxWidth - (2 * rearInset), boxHeight - (2 * rearInset)]);
        };
        
        // hole for the sata cables
        sataBoxLength = 60; //length of the box
        sataBoxHeight = 9; 
        sataBoxOffset = 48; // how far along the Y do we need to move it
        sataBoxStart = sataBoxLength + sataBoxOffset; // where on the Y did does it start ?
        translate([0, sataBoxOffset, boxWalls]) {
            cube([  boxWalls, sataBoxLength, sataBoxHeight]);
        };

        // side cooling slot
        slotHeight = boxHeight - 15;
        move(rz=180, z=slotHeight, y=6 + 3.5, x=6 + 3.5) trapezoid_90_angle(5.0001, 12, 6);
        color("red") move(  x=boxWalls, y = 4, z=slotHeight ) inside_vent_track(boxLength - boxWalls - boxWalls + 1);    
        move(rz=270 ,x=4, y=110, z=slotHeight) inside_vent_track(101);


        // screw holes either side for a rear sata mount
        move(ry=90,z=(boxWalls + (sataBoxHeight/2)), y=sataBoxStart + 5, x=boxWalls/2) cylinder(r=screwHoleRadius,h=boxWalls, center=true, $fn=20); 
        move(ry=90,z=(boxWalls + (sataBoxHeight/2)), y=sataBoxOffset - 5, x=boxWalls/2) cylinder(r=screwHoleRadius,h=boxWalls, center=true, $fn=20); 
        
        // make a rear grill
        move( x=boxWalls,y=12,z=boxWalls, rz=90) {
           vertBars(4, height=21);
        }
        move(ry=90,z=5, y=(37), x=boxWalls/2) cylinder(r=screwHoleRadius,h=boxWalls, center=true, $fn=20); 
        move(ry=90,z=5, y=(8), x=boxWalls/2) cylinder(r=screwHoleRadius,h=boxWalls, center=true, $fn=20); 
        move(ry=90,z=boxHeight - 5, y=(37), x=boxWalls/2) cylinder(r=screwHoleRadius,h=boxWalls, center=true, $fn=20); 
        move(ry=90,z=boxHeight - 5, y=(8), x=boxWalls/2) cylinder(r=screwHoleRadius,h=boxWalls, center=true, $fn=20); 

        // screw holes for around the grill

                
        // add label / version
        move(x= 50, y = boxWidth - 7, z=boxHeight - 1.5) linear_extrude(height = 1.5) text(build_label, size=6);
    }
    
};


// -------------------------
// a place where the locking tabs slip into
// -------------------------
cavityX=lockTop + pocket_join_mm;
cavityY=lockDepth;
cavityZ=lockHeight + pocket_join_mm;

module pocket_cavity() {

  union() {
    move (x=( cavityX - ( lockBottom + pocket_join_mm) )/ 2) 
    trapezoid( lockBottom + pocket_join_mm
             , lockTop + pocket_join_mm
             , lockDepth  // don't need to make the depth bigger
             , lockHeight + pocket_join_mm); 
    move (y = - lockDepth) cube([cavityX, cavityY, cavityZ]);
  }
      
}

// add the trapezoid locks
module caddy_with_pockets() {


  difference() {
    union() {
      tray_base_no_interlock();
        
      // add the trapezoid locks on top .. one side

      //color("red") move (y=boxWidth, z=0) cube([lockPos2Center,5,5]);
      move(x=lockPos1Center - (lockBottom / 2), y= boxWidth-boxWalls, z= boxHeight) {
        trapezoid(lockBottom,lockTop,lockDepth,lockHeight);
      }
      move(x=lockPos2Center - (lockBottom / 2), y= boxWidth-boxWalls, z= boxHeight) {
        trapezoid(lockBottom,lockTop,lockDepth,lockHeight);
      }
        
      // add the trapezoid locks on top .. other side
      move(x=lockPos1Center - (lockBottom / 2), z= boxHeight) {
        trapezoid(lockBottom,lockTop,lockDepth,lockHeight);
      }
      move(x=lockPos2Center - (lockBottom / 2), z= boxHeight) {
        trapezoid(lockBottom,lockTop,lockDepth,lockHeight);
      }
      
      // we need to add some "solid" for the bottom pocket's on the back
      // we will them remove an amount of that (little bigger than the "volume"
      // of the trapezoid locks
      
      // declare a large enough cube (in place)
      
      // 3mm padding around lock    cavity
      lockPad = 3.5 ;
      lockBoxX = lockTop    + join_mm + (2 * lockPad);
      lockBoxY = lockDepth  + join_mm + lockPad;
      lockBoxZ = baseDepth;
      
      // move it to central position
      move( x=lockPos1Center - (lockBoxX/2), y=boxWidth - lockBoxY - lockDepth) {
        color("red") cube([lockBoxX,lockBoxY,lockBoxZ]);
      }
      
      // 2nd box
      move(x=lockPos2Center - (lockBoxX / 2), y=boxWidth - lockBoxY - lockDepth) {
        color("blue") cube([lockBoxX,lockBoxY,lockBoxZ]);
      }

    }
    
    // subtract the trapezoid pockets underneath
    move(x=lockPos1Center - (cavityX / 2),y=boxWidth-lockDepth) { 
      pocket_cavity();
    };
    
    // subtract the trapezoid pockets underneath
    move(x=lockPos2Center - (cavityX / 2),y=boxWidth-lockDepth) { 
      pocket_cavity();
    }
    
    
    topPockBot = lockBottom + pocket_join_mm;
    topPockTop = lockTop + pocket_join_mm;
    topPockDepth = lockDepth; // don't need to make the depth bigger
    topPockHeight = lockHeight + pocket_join_mm;
    
    // subtract trapezoid pockets on bottom .. other side
    move(x= lockPos1Center - (topPockBot / 2)) {
      trapezoid( topPockBot, topPockTop, topPockDepth, topPockHeight);
    };
    move (x = lockPos2Center - (topPockBot / 2)) {
      trapezoid( topPockBot, topPockTop, topPockDepth, topPockHeight);
    };
    
  }
};


// ---------------------------------
// add a screw tab in the center
// ---------------------------------

module screwtab_top_rounded(size, thickness) { 
  hull() {
    move( rx=90, y=thickness, z=size - cornerRound, x= cornerRound) cylinder(r=cornerRound,h=thickness, $fn=20);
    move( rx=90, y=thickness, z=size - cornerRound, x=size - cornerRound) cylinder(r=cornerRound,h=thickness, $fn=20);
  }
}
module screwtab_generator(type, size, thickness) {
  if (type == "tab") {
      
      difference() {
        union() {
          screwtab_top_rounded(size, thickness);
          cube([ size , thickness, size - 2]);
        }
        screwHeadDepth = thickness - screwMaterial;
        move(x=size/2 , rx=90, z=size/2, y=screwHeadDepth/2 ) cylinder(r=screwHeadRadius,h=screwHeadDepth, center=true, $fn=20); 
        move(x=size/2 , rx=90, z=size/2, y=screwHeadDepth + (screwMaterial / 2) ) cylinder(r=screwHeadShaftRadius,h=screwMaterial+1, center=true, $fn=20); 
      }
  } else if (type == "pocket") {
    union() {
      cube([ size , thickness, size - 2]);  
      move(x=size/2 , rx=90, z=size/2, y=thickness + (screwHoleDepth/2) ) cylinder(r=screwHoleRadius,h=screwHoleDepth, center=true, $fn=20); 
      screwtab_top_rounded(size, thickness);
    }
  }
}

module hdd_side_screw_pockets() {
  headDepth = boxWalls-screwMaterial;
  move(y=headDepth/2 - .1) {
    move(rx=90) cylinder(r=screwHeadRadius,h=headDepth, center=true, $fn=20); 
    move(rx=90, y=(boxWalls/2) + (headDepth/2) - .5) cylinder(r=screwHeadShaftRadius,h=boxWalls, center=true, $fn=20); 
  }
}

module caddy_with_screwtabs() {
  difference() {
    union() {
      caddy_with_pockets();
      move( x=(boxLength/2) - (screwTabSize/2), z=boxHeight ) 
        screwtab_generator("tab", screwTabSize, screwTabThickness);
    }
    move( x=(boxLength/2) - ((screwTabSize + join_mm  )/2) ) 
      screwtab_generator("pocket", screwTabSize + join_mm , screwTabThickness + join_mm );
    
    // make HDD mount pockets
    move(x= boxWalls + 28.10,z=baseDepth + 6.35) hdd_side_screw_pockets();
    move(x= boxWalls + 28.10 + 100.6, z=baseDepth + 6.35) hdd_side_screw_pockets();

  }
}

caddy_with_screwtabs();

// ---------------------------------
// Below here is JUST print supports
// add some vert bars for print support
// ---------------------------------

module _pocket_support(length=lockDepth - 1) {
  cube([2.5, length, 1]);
  move (x=.75) cube([.7, length, lockHeight + pocket_join_mm]);
}

module _screwtab_support() {
  union() {
    cube([.7, screwTabThickness - 1, screwTabSize + 2]);
    move (x=-1) cube([3, screwTabThickness - 1, 2]);
  }
}

move(rz=90, x= boxWalls, z= baseDepth, y=58) vertBars(number=5, width=0.7, height=9, depth = boxWalls - 2, distanceApart=10 );

move (x=lockPos1Center + 2 ) _pocket_support();
move (x=lockPos1Center - 4 ) _pocket_support();
move (x=lockPos2Center + 2 ) _pocket_support();
move (x=lockPos2Center - 4 ) _pocket_support();

move (x=lockPos1Center + 2, y = boxWidth - lockDepth - lockDepth + 1   ) _pocket_support(lockDepth + lockDepth - 1);
move (x=lockPos1Center - 4, y = boxWidth - lockDepth - lockDepth + 1   ) _pocket_support(lockDepth + lockDepth - 1);
move (x=lockPos2Center + 2, y = boxWidth - lockDepth - lockDepth + 1   ) _pocket_support(lockDepth + lockDepth - 1);
move (x=lockPos2Center - 4, y = boxWidth - lockDepth - lockDepth + 1   ) _pocket_support(lockDepth + lockDepth - 1);

move (x=boxLength / 2 - 4) _screwtab_support();
move (x=boxLength / 2 + 3) _screwtab_support();

//move (y=boxWidth) import ("HDD-Caddy-V1.2.stl",convexity=3);

