/*   Test-Firmware for the SyriCtrl Board v1.1
 *   as can be downloaded from http://www.thingiverse.com/thing:1820146
 *    
 *   by Marco Radukic, Bielefeld, Germany, 03. Nov. 2016
 *   Version 1.0, Rev. 0
 *   
 *   Licensed under Creative Commons - Attribution - Share Alike
 *   Use at your own risk. For research purpose only.
 *   
 *   Description: 
 *   When executed on a SyriCtrl Board v1.1 with interface-buttons/led connected to it,
 *   this code will light up the led constantly. When the "pause"-button is pressed, the led will 
 *   begin to blink with a 1 Hz frequenzy. In this "paused"-mode, a connected stepper can be operated
 *   through the "reverse" and "forward"-buttons. The state of the program (paused or operational) 
 *   will be communicated through serial USB @ 9600 bps.
 */

#include <EEPROM.h>

/*Global Variables & Const*******************************************************************/
// pinouts
const byte dir = 4;         //'direction' pinout
const byte stp = 2;         //'step' pinout
const byte enb = 3;         //'enable' pinout
const byte bRev = 5;        //'reverse' button pin
const byte bPause = 6;      //'pause' button pin
const byte bFor = 7;        //'rforward' button pin
const byte ledPin = 8;      //'Indicator LED' pinout 

//motor setup
const long spMM = 3200;     //steps per mm including microsteps, default is 1.8° stepper and 
                            //1/16 microstepping w/ M6 leadscrew (1mm pitch per rotation)
const boolean invS = false; //Invert stepper (direction of travel) 

//globals for button handling
boolean reading = HIGH;
boolean lastPauseState = HIGH;
boolean pauseState = HIGH;
unsigned long lastDebounceTime = 0;

//globals for stepper operation
boolean pauseOp = false;
unsigned long steppermicros = 0;
long ffspeed = 0;

/*Classes************************************************************************************/
class MyLED {
  byte ledpin;
  long currentmicros;
  int frequenzy;                      
  boolean state;

public:
  MyLED (byte pin){
    ledpin=pin;
    state=HIGH;
    pinMode(ledpin, OUTPUT);
    digitalWrite(ledpin, state);
    
    currentmicros=micros();
  }

  //blink in multiples of 0.1 Hz
  void blink01Hz (int freq){          
    frequenzy=freq;
    if (frequenzy==0){
      digitalWrite(ledpin, LOW);
      return;     
    } else {
      if ((micros()-currentmicros)>=10000000/frequenzy){
        currentmicros=micros();
        digitalWrite(ledpin, state=!state);
      }
    }   
  }

  //Turn LED constantly ON 
  void ledON (){                     
    state=HIGH;
    digitalWrite(ledpin, state);
  }

  //Turn LED constantly OFF
  void ledOFF (){                    
    state=LOW;
    digitalWrite(ledpin, state);
  }
};

MyLED led1(ledPin);

/*Setup*************************************************************************************/
void setup() {
  pinMode(bRev, INPUT_PULLUP);
  pinMode(bFor, INPUT_PULLUP);
  pinMode(bPause, INPUT_PULLUP);

  pinMode(dir, OUTPUT);
  pinMode(stp, OUTPUT);
  pinMode(enb, OUTPUT);

  digitalWrite(dir, invS);
  digitalWrite(stp, LOW);
  digitalWrite(enb, HIGH);

  Serial.begin(9600);
  while(!Serial){;}
  Serial.println("Hello! This is a Test-Firmware for the SyriCtrl Board v1.1");
  
  steppermicros=micros();
}

/*Loop*************************************************************************************/
void loop() {
  //Pause-Button handling w/ 50 ms debounce
  reading=digitalRead(bPause);
  if(reading!=lastPauseState)
    lastDebounceTime=millis(); 
  if ((millis()-lastDebounceTime)>50){
    if(reading!=pauseState){
      pauseState=reading;
      if(pauseState==LOW){
        pauseOp=!pauseOp;
        EEPROM.write(0, pauseOp);
        if(pauseOp==true)
          Serial.println("Paused.");
        else
          Serial.println("Operational.");
      }
    }
  }
  lastPauseState=reading;
  pauseOp=EEPROM.read(0);

  //Switch-Case operator for Pause-function
  switch(pauseOp){
    //Normal Operation
    case false:
      //normal operation will happen here, including serial and stuff.
      led1.ledON();    
      break;
    
    //Paused Operation
    case true:
      //Button-Handling: Fast For/Rev     
      if (pauseOp==true){                                                    
        if (digitalRead(bFor)==LOW && digitalRead(bRev)==LOW)
          ffspeed=0;
        else if (digitalRead(bFor)==LOW)                                                
          ffspeed=1000;
        else if (digitalRead(bRev)==LOW)                                                          
          ffspeed=-1000;
        else 
          ffspeed=0;  
      }
      
      //Stepper-Handling
      if (ffspeed==0){
        digitalWrite(enb, HIGH);
      } else {
        digitalWrite(enb, LOW);
        if (ffspeed<0){
          digitalWrite(dir, !invS);
        } else {
          digitalWrite(dir, invS);
        }
        ffspeed=abs(ffspeed);
        if ((micros()-steppermicros)>(1000000000L/spMM/ffspeed)){
          steppermicros=micros();
          digitalWrite(stp, HIGH);
          delayMicroseconds(4);
          digitalWrite(stp, LOW);
        }
      }
      break;
  }
  
  //LED-Handling
  if (pauseOp==true)
    led1.blink01Hz(10);
}
