// Configurable Odroid XU4 Case
// http://www.thingiverse.com/thing:2655875
// Printbus, November 2017
//
// OVERVIEW
// Configurable Odroid XU4 Case provides an open frame case for use with a Hardkernel Odroid XU4 board.
//
// This is a highly modified adaptation of version 1e of the XU4 case openSCAD script originally 
// provided by user hominoid on the Hardkernel/Odroid support forum at www.forum.odroid.com
// See the original Split Air Flow Vertical or Horizontal Case thread at URL
// https://forum.odroid.com/viewtopic.php?f=97&t=26373
//
// Four fundamental case configurations are supported in this adaptation:
//  - no case fan
//  - fan mount on case top plate
//  - fan mount is on "outside" of a case end
//  - fan mount is on "inside" of fan on a case end
// The inside mount and outside mount configurations provide "split airflow" cooling of the XU4,
// with some airflow passing both below and above the XU4. 
// The outside fan mount configuration is the original hominoid design.  
// Various minor aspects of the case are independently configurable. 
//
// LICENSING
// Public domain (in accordance with hominoid's original license)
// Warranty - The design is provided as-is, without warranty of any kind.
// Liability - There is no recourse for damages or losses incurred through the use of this design.
// Although provided as public domain, attribution to the originator(s) is appreciated
//
// SUMMARY OF REVISIONS FROM THE v1e HOMINOID SCRIPT
// - Leverages the v2 XU4 model from file Odroid_XU4_Model_v2.scad https://www.thingiverse.com/thing:2655812
//    - Utilization of XU4 physical parameters from new standalone parameters file issued with the v2 model
//    - Added transparency/opacity of the overall XU4 model when viewed on case bottom or assembled case
//    - Parameter control of XU4 model options (heatsink type & perch height, emmc & SD card presence)
// - Added top level geometry section to simplify selection of desired openSCAD outputs
// - Added test plate option for testing/refinement of hardware fitment before printing full case
// - Added case configuration with "inside" fan mounting where fan mounts flush to case end 
//     perhaps for using fans without iso pins or for mounting an external fan adapter to the case.
// - Added case configuration with no case fan 
//     perhaps for a low-height case where the short OEM heatsink and fan provides adequate cooling
// - Added case configuration with fan mounted on case top plate
//     perhaps for use with the Odroid "passive" heatsink
// - Added support for fans other than 10mm thick, like 7mm or 20mm also typically available in 40mm fan size
// - Case size is now automatically adjusted to reflect design options and parameters
// - Expanded heatsink mounting options
//    - No holes on case bottom, for XU4 with OEM-type spring-clip mounted heatsinks
//    - Standoffs under XU4, for minimizing PCB stress with various through-case hardware schemes
// - Expanded surface options for heatsink and XU4 mounting screw holes
//    - Improved sizing of countersink for flat head screws
//    - Added option for fully recessed screw heads
//    - Added option for fully recessed nut traps
// - Wiring channel provision under XU4 PCB is now an option selectable to either side of board
// - Added more support struts under XU4 for configuration with no airflow ducting
// - Added option for mounting tabs on case bottom
// - Added support for case top vent grid in lieu of full heatsink cutout for optional use
// - Added support ribs across long width of case top
// - Increased flexibility for case top and case bottom standoffs (integrated vs. printed accessory vs. user supplied)
//    - Determination of holes for threading vs clearance holes is parameter based
//    - MCAD library polyhole method now used for forming holes intended for direct screw threading without tapping
// - Determination of printed accessories is now based on case option parameters (unused accessories are not printed)
// - Many square corners migrated to rounded corners for appearance and to minimize printer "ringing" on sharp turns
// - Transitioned most hardcoded values to parameters for understandability and ease of revision/tailorability
// - Improved geometry mesh to minimize "ghosting" in openSCAD model view and to ensure manifold STL results
// - $fn parameter for cylinders now set on case by case basis for more optimal relationship to cylinder diameter
// - General openSCAD scripting structure, expanded use of {} to aid with integrity of future revisions
// - Added various echo message outputs to openSCAD console pane
// - Extensive commenting added throughout
// - Added provision for using separate parameter data files for users with multiple case configurations
//  
// TAILORING NOTES AND SUGGESTIONS
// - Generally, define fan mount type first, then case options, then other design/dimension parameters
// - If user-installed threaded standoff is used with case top, use clearance holes on both case top & bottom XU4 mounting
// - Check openSCAD console pane after preview for fit conflicts or other warnings as changes are made
// - Keep in mind printer layer height limitations when striving for vertical height refinement
//
// SLICING AND PRINTING
// The openSCAD results should be manifold and not require repair before slicing.
// Adding supports in slicing may be necessary for recessed screw holes and nut traps.
// All test prints were with 3 perimeters and 2 solid top & bottom layers; 3 solid top layers may provide better finish.
// Infill is mostly limited to layers in the case top and case bottom flats.  Test prints used 30% infill. 
// If used, measure height of printed heatsink spacers and sand/file to the desired thickness
// If gold heatsink adapters are used, ensure clearance mounting holes don't bind *at all* on M3 hardware.
// The gold heatsink adapters will preferrably print a bit snug and require some sanding/filing to fit.
// If standoffs were chosen for heatsink mounting holes, check print for burrs etc. at the upper hole where clearance is zero. 

// OPENSCAD INFORMATION
// The design has been developed with openSCAD version 2015.03
//
// Due to comments and long equations, viewing this source file in widescreen monitor is recommended.
// Various messages are written to the openSCAD console, so have that open. 
//
// For OpenSCAD documentation, see 
// http://en.wikibooks.org/wiki/OpenSCAD_User_Manual/The_OpenSCAD_Language
//
// For the helpful OpenSCAD hyperlink cheatsheet, see
// http://www.openscad.org/cheatsheet/
//
// Summary of how some esoteric openSCAD functions are used -
//   union: Result is the simple sum of the subelements
//   difference: First solids are defined, and subsequent elements are removed from the solids
//   intersection: Result is the smaller volume all elements have in common
//   hull: Creates a convex shape formed by the worst case extremes of all subelements
//   linear_extrude: extends a 2D polygon in a third dimension
//   str: form a text string containing argument values
//   echo: display string contents on openSCAD console pane
//
// openSCAD use of ?: conditional operator is the same as in C/C++
//   result = condition?expression1:expression2
//   if condition tests as true (i.e., nonzero), result is set to expression1, otherwise result set to expression2
//
// openSCAD use of for loop
//   for ( variable = [ start:increment:end]) {}
//     perform the bracketed action using a variable with the value start and incrementing to the value end
//   for ( variable = [ start:end ]) {} 
//     similar, but increment will default to value of 1
//   for ( variable = [ start,end ]) {}
//     only two passes, one with variable set to value start, one with variable set to value end
//
// openSCAD scripts are interpreted, not compiled.  
// Programmers take note: while openSCAD syntax looks like C, it isn't really C code.
// You cannot, for example, modify a parameter within an if statement

////////// REVISION HISTORY ///////////////////////////////////////////////////
// YYMMDD date code
// 171119 - initial publish to Thingiverse

////////// INCLUDE AND USE  ///////////////////////////////////////////////////
// NOTE: openSCAD version 2015.03 or newer required for text support used in make_test_plate()
//       openSCAD version 2014.03 or newer required for MCAD library used for polyhole()
//
include <Odroid_XU4_Parameters.txt>    // from https://www.thingiverse.com/thing:2655812
// Odroid_XU4_Parameters.txt captures key dimension info for the XU4
//
// EXTERNAL FUNCTIONS USED
use <MCAD/polyholes.scad>
//  Requires openSCAD MCAD library to be installed (openSCAD 2014.03 or later)
//  polyhole( h,d ) provides improved size control on small holes intended for threading
//    h: height of the hole
//    d: hole diameter
//    Applies false centering; polyhole intentionally varies $fn with size of hole
use <Odroid_XU4_Model_v2.scad>        // from https://www.thingiverse.com/thing:2655812
//  odroid_xu4( emmc_type, sd_type, hs_type, perch_t, opacity, a_type ) provides an XU4 3D model, with options 
//    emmc_type: 0=no emmc module, 1=show emmc module
//    sd_type:   0=no microSD card, 1=show microSD card
//    hs_type:   0=no heatsink, 1=stock blue, 2=stock with fan, 3=passive 40x40x25mm,
//               4=gdstime gold w/gds hardware, 5=gdstime gold w/custom adapters, 6=flat 40mm plate
//    perch_t:   Thickness of any shim or perch to be added between ICs and heatsink
//    opacity:   transparency factor to be applied to model colors
//    a_type:    adapter type for the gold heatsink: 0=clearance hole, 1=nut trap
//  view_fan( fan_size, fan_w, fan_ho, fan_hid, fan_wall ) generates a representative fan block 
//    fan_size: edge-to-edge size of the fan frame, typically 40mm, 80mm, etc. 
//    fan_w: width or thickness of the fan frame, typically 10mm, 20mm, etc. 
//    fan_hid: fan corner mounting hole inside diameter
//    fan_ho: corner mounting hole offsets from fan edges
//    fan_wall: amount of frame left at sides, nominally 1mm on a 40mm fan
//  rounded_plate ( size, radius ) generates a flat plate with rounded corners
//    size: array of [x, y, z] dimensions
//    radius: radius to be applied to the corners
//  heatsink_adapter_gold( type, opacity ) provides elements for mounting a gdstime heatsink in rotated orientation
//    type: 0=clearace through-hole, 1=include an M3 nut trap
//    opacity: transparency factor to be applied to colors   

//////////////////////////////////////////////////////////////////////////////////////
// TOP LEVEL GEOMETRY
// Edit this section to uncomment the ONE result desired; all others need to be commented out !!
// For local debug throughout the file, all options here must be commented out (i.e, disabled)
//
 show_assembly ( xu4_opacity=0.6 ); // non-printable view of the assembled case with transparent XU4 installed
// show_bottom ( xu4_opacity=0.6 );   // non-printable view of transparent xu4 above just case bottom
//
// make_test_plate ();  // prepare test plate for checking hole sizes before printing case
// make_bottom ();      // prepare just the case bottom for printing
// make_top ();         // prepare just the case top for printing
// make_case ();        // prepare case top and bottom for printing
// make_accessories (); // prepare any required case accessories for printing
// make_all ();         // prepare all necessary case components and accessories for printing
//
// end top level geometry

//////////////// PARAMETERS  ///////////////////////////////////////////////////
// All dimensions are millimeter 
// Parameters are grouped by physical component dimensions, case options, design parameters, hardware sizing,
//   followed by non-tailorable parameters and calculated parameters
// In openSCAD, all parameters are constants.  There are no true variables.  
//   UPPERCASE labels are arbitrarily used here to generally identify hardcoded constants.  
//   Lowercase labels are arbitrarily used here to generally identify something calculated.
// For parameter adjustments here to have any effect, comment out all parameter over-ride includes
//   at the end of the list of tailorable parameters
// ERRATA: Not all parameters are used in the four basic case types, but most cannot be ignored/deleted
//   since they are still used in initial calculations and parameter limit checking.  
//
// **COMPONENT REFERENCE DIMENSION PARAMETERS**
// This section contains dimensions from component measurements or data sheets
//
// XU4 board and heatsink assembly
//   See file Odroid_XU4_Parameters.txt file for other key XU4 dimension parameters that may be used here
//   The following parameters define specifics for THIS utilization of the XU4
//   Any other parameters prefixed with XU4_ used in this file are defined in the Odroid_XU4_Parameters.txt file
// XU4_HS_TYPE: 0=no heatsink, 1=stock without fan, 2=stock with fan, 3=passive blue 40x40x25mm,
//              4=gdstime gold w/gds hardware, 5=gdstime gold w/custom adapters, 6=flat 40mm plate
XU4_HS_TYPE = 5;        // see the above comment
XU4_EMMC_TYPE = 1;      // 0=no emmc module, 1=yes show emmc module present
XU4_USD_TYPE = 1;       // 0=no microSD card, 1=yes show microSD card present
XU4_PERCH_T = 3.18;     // 0=no perch, >0=raise the heatsink up that amount in XU4 model
XU4_HS_ADAPT_TYPE = 1;  // 0=M3 clearance hole, 1=M3 nut trap (applies to gold heatsink adapters) 
//
// Case mounted fan - generic 40mm fan
EXT_FAN_SIZE = 40;      // nominal size 40mm x 40mm (use CASE_FAN_FIT to adjust fit clearance, not this)
EXT_FAN_HID = 4.0;      // fan corner mounting hole inside diameter (use case parameters to set hole size, not this)
EXT_FAN_MHO = 4.0;      // corner mounting hole offsets from fan edges
EXT_FAN_WALL = 1.0;     // amount of frame left at thin sides of fan body
// EXT_FAN_T: include any isolation shims, finger guards used, etc. that would also need to fit in fan cutout, if used  
EXT_FAN_T = 10;         // nominal thickness like 7mm, 10mm or 20mm (use CASE_FAN_FIT to adjust fit clearance, not this)
EXT_FAN_CSCREW_OD = 5;  // Fan Case Screw OD, should case screws be used with outside fan mount (sizes vary; measure YOURS)

// **CASE CONFIGURATION OPTIONS PARAMETERS** //
// Case top-level configuration options
// CASE_FMOUNT_TYPE: Selects the four fundamental case configurations
//   0=no case fan; case height driven by CASE_NOFAN_HEIGHT
//   1=fan mount on case top; case height driven by CASE_NOFAN_HEIGHT
//   2=fan mount outside of fan on case end; case height is derived from EXT_FAN_SIZE
//   3=fan mount inside of fan on case end; case ehight is derived from EXT_FAN_SIZE
CASE_FMOUNT_TYPE = 2;    // see the above comment
CASE_FMOUNT_HOLE_TYPE=0; // fan mount hole: 0=iso pin clearance, 1=case screw clear, 2=M3 screw clear, 3=for threading M3
CASE_FAN_HW_SHOW = 1;    // 0=show no hardware, 1=show iso pins, 2=show Keystone 720 bumper, 3=show printed foot
//CASE_DUCT_TYPE: this parameter is ignored if CASE_FMOUNT_TYPE doesn't involve a fan mount on case end
CASE_DUCT_TYPE = 1;      // 0=no ducting, 1=include ducting on case top and bottom
CASE_WCHAN_TYPE = 1;     // 0=no wire channel, 1=wire channel under USB3.0 side, 2=wire channel under RJ45 side
// CASE_BFOOT_TYPE: still needs to be set to 2 or 3 to force accessory print even if foot will actually be used on fan instead
CASE_BFOOT_TYPE = 0;     // bottom feet: 0=none, 1=Keystone 720, 2=printed accessory (clear hole), 3=printed accessory (nut trap)
CASE_MTAB_TYPE = 0;      // 0=no case mounting tabs, 1=mounting tabs on case bottom
CASE_BSO_TYPE = 1;       // 0=provide no case bottom standoffs, 1=integrated into case, 2=printed accessory only
CASE_EMMC_CUTOUT = 1;    // 0=no cutout, 1=finger access in case bottom
CASE_HS_MOUNT_TYPE = 1;  // 0=no case HS mount (PCB spring clip), 1=thru-case w/stanchion only, 2=thru-case with standoff
//CASE_HS_CUTOUT_TYPE: This parameter is ignored if CASE_FMOUNT_TYPE specifies fan mounted on case top
CASE_HS_CUTOUT_TYPE = 1; // 0=no cutout, 1=provide cutout sized to CASE_HS_CUTOUT_SIZE, 2=grid pattern
//CASE_XU4_THOLE_TYPE, CASE_XU4_BHOLE_TYPE and CASE_HS_HOLE_TYPE:
//  These specify the type of hole and screw head provision for 
//  case top XU4 holes, case bottom XU4 holes, and case bottom heatsink mounting holes respectively
//    0=clear hole w/no head provision
//    1=clear w/flat head countersink
//    2=clear w/full head recess
//    3=clear hole w/nut trap
//    4=thread thru-hole
//    5=thread hole to a specified depth
CASE_XU4_THOLE_TYPE = 0; // see the above comment; if set to 5 parameter CASE_XU4_THOLE_TD sets thread depth
CASE_XU4_BHOLE_TYPE = 0; // see the above comment; if set to 5 parameter CASE_XUF_BHOLE_TD sets thread depth
CASE_HS_HOLE_TYPE = 0;   // see the above comment' if set to 5 parameter CASE_HS_HOLE_TD sets thread depth
CASE_UHOLDER_TYPE = 1;   // 0=no UART holder, 1=UART on RJ45 side, 2=UART on USB3 side, 3=printed accessory only
CASE_BHOLDER_TYPE = 2;   // 0=no RTC battery holder, 1=holder on RJ45 side, 2=holder on USB3 side, 3=printed accessory only
CASE_TSO_TYPE = 1;       // 0=provide no case top standoffs, 1=integrated into case, 2=printed accessory only
CASE_TSO_USER_H = 24;    // 0=no user-installed standoffs involved, >0=assume user will install this height standoff
CASE_TSO_USER_TYPE = 1;  // top side user installed standoffs: 0=user provided, 1=printed accessory
//
// **TAILORABLE DESIGN PARAMETERS**
// These define details associated with the higher level configuration options defined above
// Case sizing
CASE_XU4_INSET = 2;      // amount of inset from case edge to xu4 edge (i.e., case oversize from PCB)
CASE_BH = 8;             // Overall case bottom height, including standoffs. Affects airflow split above/below XU4 PCB
CASE_FAN_ADJ = 1;        // additional spacing to be provided between XU4 and fan, i.e., for boot switch clearance
CASE_NOFAN_HEIGHT = 42;  // case overall height for no fan or fan mounted on case top plate set to clear passive HS on XU4
//
// Parameters common to both case bottom and case top
CASE_FMOUNT_THICK = 4;   // thickness of the case fan mount; 4 works for (most?) iso pins and for threading into 
CASE_FAN_FIT = 0.3;      // fit tolerance to be added around all faces of the fan
CASE_XU4_HFN = 18;       // $fn to be applied to XU4 mounting/case assembly screw holes
//
// Other Case bottom parameters
// CASE_FAN_BRECESS: 0=fan fits within case bottom, +value recesses fan into a cutout, -value raises fan mount up
//CASE_FAN_BRECESS = 0;   // fan body is flush with case bottom plate; no fan cutout necessary for outside mount
CASE_FAN_BRECESS = EXT_FAN_WALL;  // recess fan mount so inside of fan side wall lines up with case bottom plate
CASE_BSO_RISER_HEIGHT = 2; // height of the riser shoulders applied at base of standoffs on case bottom
CASE_BSO_OD = 7.0;       // outer diameter of the case bottom standoffs, if integrated into the case bottom
CASE_BSO_FN = 6;         // $fn value to be applied to printed case bottom standoffs:  6=hex, 24+=round
CASE_XU4_BHOLE_TD = 6;   // case bottom XU4 mounting hole limited depth for threading when specified by CASE_XU4_BHOLE_TYPE
CASE_HS_HFN = 18;        // $fn to be applied to through-case heatsink mounting screws
CASE_HS_STANCHION_H = 3; // height of the support stanchions around heatsink holes, if present
CASE_HS_HOLE_TD = 6;     // case bottom heatsink mounting hole limited depth for threading when specified by CASE_HS_HOLE_TYPE 
CASE_MTAB_HID = 4.4;     // hole ID on case mounting tabs set for about #4 wood screw clearance
CASE_MTAB_WIDTH = 8;     // mounting tab width set to a bit larger than head on #4 wood screw
//
// Other Case top parameters
// CASE_FAN_TRECESS: 0=fan fits within case top, +value recesses fan into a cutout, -value increases XU4/heatsink clearance
//CASE_FAN_TRECESS = 0;      // fan body is flush with case top plate; no fan cutout necessary for outside mount 
CASE_FAN_TRECESS=EXT_FAN_WALL; // recess fan mount so inside of fan side wall lines up with case top plate 
//CASE_FAN_TRECESS = -1;     // example value for increasing case height to provide more passive heatsink clearance
//CASE_FAN_TRECESS = -8;     // example value for increasing case height to provide gold heatsink clearance under top plate
// CASE_HS_CUTOUT_SIZE: More than 44 will cause one of the case top riser shoulders to protrude into cutout
CASE_HS_CUTOUT_SIZE = 44;  // xy size of case top cutout above XU4 heatsink
CASE_TSO_RISER_HEIGHT = 2; // height of the riser shoulders applied at base of standoffs on case top
CASE_TSO_OD = 7.0;         // outer diameter of the case top standoffs, if integrated into the case top
CASE_TSO_FN = 6;           // $fn value to be applied to printed case top standoffs:  6=hex, 24+=round
CASE_XU4_THOLE_TD = 6;     // case top XU4 mounting hole limited depth for threading, when selected by CASE_XU4_THOLE_TYPE
//
// User-installed top standoff parameters, used for showing case in preview and preparing as printed accessory
TSO_USER_OD = CASE_TSO_OD; // outer diameter of the user installed top standoffs set here to match rest of top standoffs
TSO_USER_FN = 6;        // $fn value to be applied to user installed case top standoffs:  6=hex, 24+=round
TSO_USER_HOLE_TYPE = 2; // user standoff: 0=clear thru-hole, 1=thread thru-hole, 2=limited depth hole for threading each end
TSO_USER_HOLE_TD = 6;   // user standoff: thread to this depth each end when specified by TSO_USER_HOLE_TYPE
//
// User-installed heatsink spacer for XU4 
HS_SPACER_TYPE = 1;     // 0=no spacer, 1=print spacer as accessory
HS_SPACER_SIZE = 8;     // width and depth of the spacer 
HS_SPACER_HEIGHT = XU4_PERCH_T+IC_SOC_H;   // height of the spacer set here to SOC chip height and any perch used
HS_SPACER_WT = 1.5;     // wall thickness of the spacer
//
// UART holder parameters (global since they are shared between uart_holder and uart_strap modules)
UHOLDER_POST_HSP=21;    // UART holder mounting post hole spacing, center to center
UHOLDER_POST_OD=6;      // UART holder mounting post outer diameter
//
// User-installed foot, used for showing case in preview and preparing as printed accessory
FOOT_H = 8;             // foot height
FOOT_OD2 = 9;           // foot outside diameter at narrow end of cone forming foot
//    
// **HARDWARE/FASTENER SIZING PARAMETERS**
/*==================================================== 
The following table provides standard hardware dimension reference data
These are a reference point only; adjustment may be necessary due to nozzle size, etc.
Also note that MCAD polyhole does better at actual diameters than circle/cylinder

SIZE         THREAD  CLEAR   NUT AFD  NUT OD   NUT     HEAD     HEAD    WASHER  FLAT    
             HOLE    HOLE    WRENCH   @fn=6    HEIGHT  OD       HEIGHT  HEIGHT  OD      
-----------  ------  ------  -------  -------  ------  ------   ------  ------  ----- 
M2 x 0.4     1.75mm  2.20mm  4.0mm    4.62mm   1.6mm   4.0mm    2.0mm   0.3mm   5.5mm
M2.5 x 0.45  2.20mm  2.75mm  5.0mm    5.77mm   2.0mm   5.0mm    2.5mm   0.3mm   6.0mm
M3 x 0.5     2.70mm  3.30mm  5.5mm    6.35mm   2.4mm   6.0mm    3.0mm   0.5mm   7.0mm 
M4 x 0.7     3.50mm  4.40mm  7.0mm    8.08mm   3.2mm   8.0mm    4.0mm   0.8mm   9.0mm
M5 x 0.8     4.50mm  5.50mm  8.0mm    9.24mm   4.7mm   10.0mm   5.0mm   1.0mm   10.0mm
M6 x 1.0     5.50mm  6.60mm  10.0mm   11.55mm  5.2mm   12.0mm   6.0mm   1.6mm   12.0mm
M8 x 1.25    7.20mm  8.80mm  13.0mm   15.01mm  6.8mm   16.0mm   8.0mm   2.0mm   17.0mm
#2-56        1.85mm  2.44mm  4.76mm   5.50mm   1.59mm  4.60mm   2.18mm  0.91mm  6.35mm
#3-56        2.26mm  2.79mm  4.76mm   5.50mm   1.59mm  5.28mm   2.51mm  0.91mm  7.94mm
#4-40        2.44mm  3.26mm  6.35mm   7.33mm   2.38mm  5.97mm   2.85mm  1.14mm  9.53mm
#6-32        2.95mm  3.80mm  7.94mm   9.17mm   2.78mm  7.37mm   3.51mm  1.14mm  11.11mm
#8-32        3.66mm  4.50mm  8.73mm   10.08mm  3.18mm  8.74mm   4.17mm  1.14mm  12.7mm
#10-24       4.09mm  5.11mm  9.53mm   11.00mm  3.18mm  10.13mm  4.83mm  1.14mm  14.29mm
#10-32       4.31mm  5.11mm  9.53mm   11.00mm  3.18mm  10.13mm  4.83mm  1.14mm  14.29mm
1/4-20       5.56MM  6.76MM  11.11mm  12.83mm  4.76mm  13.03mm  6.35mm  1.80mm  18.65
Notes:       1,2     2,3     4        5        6       7,8      8,9     8,10    8,11
----------------------------------------------------------------------------------------
Note  1: Thread hole is for tap or self thread of machine screw in soft material
Note  2: Hole dimensions are from littlemachineshop.com Tap Drill - 50% Thread column data
Note  3: Clearance hole data is littlemachineshop.com Clearance Drill - Standard Fit column data
Note  4: Nut Across Flat Diameter (AFD) metric data is ISO, inch is boltdepot.com US Nut Size table
Note  5: Nut round OD is the openSCAD circle diameter required to achieve a nut size at $fn=6 
         Nut OD is calculated as = (NUT AFD)/cos(30)
Note  6: Nut height is for standard hex nut; jam nuts are less, lock nuts are more
         Metric data is ISO 4032, SAE data from boltdepot.com US Nut Size tables
Note  7: Head diameter varies with the head style; value shown is max across all except truss head
Note  8: Data from http://www.numberfactory.com/nf_metric.html or http://www.numberfactory.com/nf_inch.html 
Note  9: Head height varies with head style; value shown is max across all styles
Note 10: Standard washer height or thickness
Note 11: Flat washer outer diameter
====================================================*/
// Tailor fastener sizing and fit adjustments here
// MCAD library polyhole will be used for screw thread holes - more accurate small diameter than hole by cylinder method
// Printer variables include PLA vs. ABS, extrusion calibration, inside vs. outside perimeters printed first, etc. 
// Values provided were tested with Simplify3D as slicer, PLA on Prusa i3 type printer
// First define values standardized for your setup based on printing the available test plate
// ERRATA: changes to M3_HID_CLEAR, M3_NUT_AFD_CLEAR, and M3_NUT_RD require replication in model file heatsink_adapter_gold()
M3_HID_THREADED = 2.7 + 0.2;   // Hole inside diameter for M3 threading without tapping (polyhole method used)
M3_HID_CLEAR = 3.3 + 0.3;      // Hole inside diameter for M3 clearance (cylinder method used)
M3_HEAD_OD_CLEAR = 6.0 + 0.2;  // Hole outside diameter for worst case M3 head recess
M3_HEAD_RD = 3.0;              // Recess depth for worst case M3 head
M3_NUT_AFD_CLEAR = 5.5 + 0.3;  // Across-Flats-Diameter to be used for M3 nut traps
M3_NUT_RD = 2.4 + 0.2;         // Recess depth to be used for M3 nut traps
//
// Most holes throughout the case will leverage those parameters directly 
// Apply these standardized values to some holes that might warrant hole-specific user adjustment
BHOLDER_MTAB_HID = M3_HID_THREADED;      // battery holder mounting tab hole inside diameter set to threading M3 (polyhole used)
USTRAP_HID = M3_HID_CLEAR;               // UART strap hole inside diameter set to M3 clearance hole (cylinder used)
UHOLDER_POST_HID = M3_HID_THREADED;      // UART strap mounting post inside diameter set for threading M3 (polyhole used)
UHOLDER_MTAB_HID = M3_HID_THREADED;      // UART holder mounting tab hole inside diameter set to threading M3 (polyhole used)
// Other fastener details
CASE_FMOUNT_ISO_HID_CLEAR = 4.0+0.4;     // Fan mount clearance hole ID for use with iso pins
CASE_FMOUNT_CSCREW_HID_CLEAR = EXT_FAN_CSCREW_OD + 0.4; // Fan mount clearance hole ID for use with fan case screws
// end user-tailorable parameters
//
// PARAMETER IMPORT FILES (OPTIONAL)
// Standalone files containing just the above parameters can be included here. 
// Any parameter values in a standalone file will over-ride the values provided above
// This makes it easy to archive various case configurations while only having one "active" scad file that creates the case
// The "last" parameter definition wins, so comment out all but the parameters file you want to use.
// If you want edit the files in openSCAD, change the files to *.scad and edit the entries here to match.  
// include <sampleA_parameters.txt>   // XU4 with OEM heatsink & fan, no case fan
// include <sampleB_parameters.txt>   // XU4 with passive heatsink, top side fan mount with 7mm fan
// include <sampleC_parameters.txt>   // XU4 with gdstime gold heatsink & 3.18mm perch, outside mount 10mm fan
// include <sampleD_parameters.txt>   // XU4 with gdstime gold heatsink & 3.18mm perch, inside mount 20mm fan
//
// **NON-TAILORABLE PARAMETERS**
// Parameters beyond this point are normally not altered in basic tailoring
// The design is easy to break; tread gently
//
// Factors applied to ensure geometry mesh
MF = 0.02;         // Mesh overlap factor is the amount of overlap on geometries for proper mesh
MSA = MF;          // Mesh single adjustment factor (translate ends; size adjustment on single ended mesh)
MDA = 2*MF;        // Mesh double adjustment (size adjustment on double ended mesh like boring holes)
//
// The design isn't set up to handle changes to these parameters  
CASE_THICK = 2.0;  // thickness of both case bottom and case top; various logic will break if this changes from 2
CASE_BWT = 2;      // Case bottom wall thickness for ducts and XU4 support struts (design requires this as 2)
CASE_TWT = 2;      // Case top wall thickness for ducts and support ribs (design likely requires this left as 2) 
CASE_RIB_HEIGHT = 1.0;   // height of support ribs added to case top and case bottom (1 works best with integrated UART holder)

// **PARAMETER VALIDITY CHECKS**
// Prompt user with parameter warning and error messages to openSCAD console panel
// Note: validity checks performed may not catch all possible problems
//
// Check for invalid configuration with stock heatsink and fan
if ( XU4_HS_TYPE == 2 ) {             // xu4 with stock heatsink & fan
  if ( CASE_FMOUNT_TYPE )             // some form of case fan
    echo ("CAUTION: Configuration contains both stock heatsink fan and case fan");
  if ( CASE_HS_CUTOUT_TYPE == 0 )     // no airflow cutout in case top
    echo ("WARNING: No case cutout provided above stock heatsink fan");
}  // end if XU4_HS_TYPE is stock with fan
//
// check for UART holder and RTC holder conflict
if ( CASE_UHOLDER_TYPE == 1)          // uart holder on RJ45 side
  if ( CASE_BHOLDER_TYPE == 1)        // battery on RJ45 side
    echo ("ERROR: UART holder and RTC battery holder both defined for RJ45 side");
if ( CASE_UHOLDER_TYPE == 2)          // UART holder on USB3 side
  if ( CASE_BHOLDER_TYPE == 2)        // battery holder on USB3 side
    echo ("ERROR: UART holder and RTC battery holder both defined for USB3 side");
//  
// check clearance between boot select switch lever and fan or fan mount
sw_clearance = CASE_XU4_INSET+CASE_FAN_ADJ-SW_BOOT_LEVER_EXT;
if ( CASE_FMOUNT_TYPE > 1 ) {        // only check switch clearance if there's an end-mounted fan
  if ( sw_clearance < 0 )
    echo ( str("WARNING: ",sw_clearance,"mm conflict between fan and boot switch; increase CASE_XU4_INSET or CASE_FAN_ADJ"));
  else
    echo ( str("Clearance between fan body and boot select switch lever: ",sw_clearance,"mm"));
}  // end if CASE_FMOUNT_TYPE involves a fan mounted on case end
// 
// check material available for width of recessed screw heads or nut traps at corners
if ( CASE_XU4_INSET == 0 ) {         // case is no bigger than PCB
  if ( CASE_XU4_BHOLE_TYPE == 2 )    // recessed head on case bottom holes
    echo ("WARNING: Minimal case wall for case bottom recessed screw heads; increase CASE_XU4_INSET");
  if ( CASE_XU4_BHOLE_TYPE == 3 )    // nut trap on case bottom holes
    echo ("WARNING: Minimal case wall for case bottom nut traps; increase CASE_XU4_INSET");
}  // end if CASE_XU4_INSET is zero
//
// check material available for depth of recessed screw heads or nut traps
if ( CASE_BSO_RISER_HEIGHT == 0 ) {  // no riser shoulder on floor of case bottom
  if ( CASE_XU4_BHOLE_TYPE == 2 )    // recessed screw heads on case bottom XU4 mounting holes
     echo ("WARNING: Bottom side recessed screw heads should be used with standoff risers (CASE_BSO_RISER_HEIGHT)");
  if ( CASE_XU4_BHOLE_TYPE == 3 )    // recessed nut traps on case bottom XU4 mounting holes
     echo ("WARNING: Bottom side nut traps should be used with standoff risers (CASE_BSO_RISER_HEIGHT)");
}  // end if CASE_BSO_RISER_HEIGHT is no riser shoulder
if ( CASE_TSO_RISER_HEIGHT == 0 ) {  // no riser shoulder on floor of case top
  if ( CASE_XU4_THOLE_TYPE == 2 )    // recessed screw heads on case top XU4 mounting holes
     echo ("WARNING: Top side recessed screw heads should be used with standoff risers (CASE_TSO_RISER_HEIGHT)");
  if ( CASE_XU4_THOLE_TYPE == 3 )    // recessed nut traps on case top XU4 mounting holes
     echo ("WARNING: Top side nut traps should be used with standoff risers (CASE_BSO_RISER_HEIGHT)");
}  // end if CASE_TSO_RISER_HEIGHT is no riser shoulder
//
// check material available for specified thread depths
if ( CASE_XU4_BHOLE_TYPE == 5 )        // on case bottom, apply limited depth hole for threading
  if ( CASE_XU4_BHOLE_TD > CASE_BH )
    echo ("WARNING: Bottom XU4 hole thread depth exceeds case bottom height (CASE_XU4_BHOLE_TD vs. CASE_BH)");
if ( CASE_HS_HOLE_TYPE == 5 )        // on case bottom, apply limited depth hole for threading
  if ( CASE_HS_HOLE_TD > CASE_BH )
    echo ("WARNING: Bottom heatsink hole thread depth exceeds case bottom height (CASE_HS_HOLE_TD vs. CASE_BH)");
// Test for case top thread depth occurs after top height has been calculated
//
if ( CASE_FAN_BRECESS > CASE_THICK )  // fan mounting holes will recess into case bottom plate
  echo ( str("WARNING: CASE_FAN_BRECESS of ",CASE_FAN_BRECESS,"mm should not exceed CASE_THICK of ",CASE_THICK,"mm"));
if ( CASE_FAN_TRECESS > CASE_THICK )  // fan mounting holes will recess into case top plate
  echo ( str("WARNING: CASE_FAN_TRECESS of ",CASE_FAN_TRECESS,"mm should not exceed CASE_THICK of ",CASE_THICK,"mm"));
//
// gap between a heatsink spring clip (in case it is used) and case bottom heatsink mount stanchion (if applied)
stanchion_gap = CASE_BH-CASE_THICK-HS_STOCK_PIN_PROTRUSION-CASE_HS_STANCHION_H;
if ( CASE_HS_MOUNT_TYPE == 1 )      // stanchion used for heatsink mount 
  if ( stanchion_gap < -1 )
    echo ( str("CAUTION: CASE_HS_STANCHION_H of ",CASE_HS_STANCHION_H,
               "mm may lead to stanchion intererence with heatsink spring clips, if used"));
//
// Type of screw head provisions under feet  
if ( CASE_BFOOT_TYPE != 0 ) {                     // any form of foot installed on bottom
  if ( CASE_XU4_BHOLE_TYPE != 0 )                 // any form of countersink, recess, or nut trap on case bottom holes
    echo ("WARNING: Feet being installed over unnecessary countersink or recess on case bottom XU4 holes");
  if ( CASE_MTAB_TYPE != 0 )
    echo ("WARNING: Feet and mounting tabs on case bottom are normally mutually exclusive options");
}  // end if CASE_BFOOT_TYPE is any form of foot installed on bottom
//
// end parameter validity checks
  
// **CALCULATED PARAMETERS AND ECHO MESSAGES**
echo ( str("Airflow gap below XU4 PCB with ",CASE_BH,"mm bottom height: ",CASE_BH-CASE_THICK,"mm"));
//
case_depth = XU4_PCB_DEPTH+2*CASE_XU4_INSET;       
echo ( str("Case depth with ",CASE_XU4_INSET,"mm CASE_XU4_INSET: ",case_depth,"mm"));
//
// use indirect way to calculate case_width since can't alter a parameter within an openSCAD if{}
case_width_nofan = CASE_XU4_INSET!=0?
    (XU4_PCB_WIDTH+2*CASE_XU4_INSET):XU4_PCB_WIDTH;
case_width_fan_outside = CASE_XU4_INSET!=0?
    (XU4_PCB_WIDTH+2*CASE_XU4_INSET+CASE_FAN_ADJ+2*CASE_FAN_FIT+EXT_FAN_T+CASE_FMOUNT_THICK):
    (XU4_PCB_WIDTH+CASE_FAN_ADJ+2*CASE_FAN_FIT+EXT_FAN_T+CASE_FMOUNT_THICK);
case_width_fan_inside = CASE_XU4_INSET!=0?
    (XU4_PCB_WIDTH+2*CASE_XU4_INSET+CASE_FAN_ADJ+CASE_FMOUNT_THICK):
    (XU4_PCB_WIDTH+CASE_FAN_ADJ+CASE_FMOUNT_THICK);
case_width = (CASE_FMOUNT_TYPE<2)?case_width_nofan:
             (CASE_FMOUNT_TYPE==2)?case_width_fan_outside:case_width_fan_inside;
if ( CASE_FMOUNT_TYPE < 2)    // no end-mounted fan provided in case
  echo ( str("Case width with ",CASE_XU4_INSET,"mm CASE_XU4_INSET and no fan at case end: ",case_width,"mm"));
if ( CASE_FMOUNT_TYPE == 2)    // inside mounted fan 
  echo ( str("Case width with ",CASE_XU4_INSET,"mm CASE_XU4_INSET and ",EXT_FAN_T,"mm inside mount fan: ",case_width,"mm"));
if ( CASE_FMOUNT_TYPE == 3)    // outside mounted fan 
  echo ( str("Case width with ",CASE_XU4_INSET,"mm CASE_XU4_INSET and outside mounted fan: ",case_width,"mm"));
//
case_height_w_fan = EXT_FAN_SIZE+2*CASE_FAN_FIT+2*CASE_THICK-CASE_FAN_BRECESS-CASE_FAN_TRECESS; // height is driven by fan
case_height = (CASE_FMOUNT_TYPE < 2)?CASE_NOFAN_HEIGHT:case_height_w_fan; // height is fixed if no end mounted fan
echo ( str("Case overall height: ",case_height,"mm"));
case_th = case_height - (CASE_BH + XU4_PCB_THICK); // case top height is whatever is left after case bottom and XU4 PCB
echo ( str("Case top height: ",case_th,"mm"));
//if ( CASE_TSO_USER_H > (case_th - CASE_TSO_RISER_HEIGHT - CASE_THICK ))  // user standoff is too tall
//  echo(str("ERROR: User-installed standoff height of ",CASE_TSO_USER_H,"mm is too tall for case top height of ",case_th,"mm"));
//
case_cr = CASE_XU4_INSET>0?XU4_PCB_MHO+CASE_XU4_INSET:XU4_PCB_MHO;  // case corner radius follows inset for appearance
echo ( str("Case corner radius with ",CASE_XU4_INSET,"mm CASE_XU4_INSET: ",case_cr,"mm"));
//
foot_od1 = case_cr>FOOT_OD2/2?2*case_cr:FOOT_OD2;                 // wider OD of foot set to wider of FOOT_OD2 or 2*case_cr
//
// case top duct heights are the amount they rise *above* CASE_THICK
case_tduct_bootsel_h = case_th - CASE_THICK - SW_BOOT_H - 1;     // bring duct down to 1mm from boot select switch body
case_tduct_i2sl_h = case_th - CASE_THICK - CN_I2S_H - 1;         // bring duct down to 1mm from i2s connector body
//
// limit case bottom standoff riser OD to larger of 11mm or 2*case corner radius
case_bso_riser_od = case_cr>5.5?2*5.5:2*case_cr;
// top riser is limited to smaller of 9mm OD or 2*case corner radius to minimize overlap with heatsink cutout
case_tso_riser_od = CASE_HS_CUTOUT_TYPE==0?case_bso_riser_od:case_cr>4.5?2*4.5:2*case_cr;
case_bso_height = CASE_BH - CASE_THICK - CASE_BSO_RISER_HEIGHT;  // bottom standoff height above risers
echo ( str("Case bottom standoff height: ",case_bso_height,"mm"));
//
case_tso_height = case_th - CASE_THICK - CASE_TSO_RISER_HEIGHT - CASE_TSO_USER_H;  // top standoff height above risers
if ( case_tso_height >= 0 ) {             // value of CASE_TSO_USER_H is acceptable                 
  echo ( str("Case top standoff height: ",case_tso_height,"mm (reflects user-provided standoff of ",CASE_TSO_USER_H,"mm)"));
  if ( CASE_XU4_THOLE_TYPE ==5 )          // on case top, apply limited depth hole for threading
    if ( CASE_XU4_THOLE_TD > (case_tso_height + CASE_TSO_RISER_HEIGHT + CASE_THICK ))
      echo ("WARNING: Case top XU4 hole thread depth exceeds available material height");
}  // end if case_tso_height > 0
else                                      // can't fit CASE_TSO_USER_H in the specified case height
  echo(str("ERROR: User-installed standoff height of ",CASE_TSO_USER_H,"mm is too tall for case top height of ",case_th,"mm"));
//
xu4_pcb_origin = [ CASE_XU4_INSET+XU4_PCB_WIDTH/2, CASE_XU4_INSET+XU4_PCB_DEPTH/2, CASE_BH ];
//
case_mtab_mho = CASE_MTAB_WIDTH/2 + 2;   // case mounting hole offset from case edge provides add'l head clearance
echo ( str("Case mounting tab hole spacing is x: ",case_width-2*case_cr-CASE_MTAB_WIDTH,
          "mm, y: ",case_depth+2*case_mtab_mho,"mm"));
//                                 
// xu4_ldi: Lower duct inset from the xu4 lower (emmc side) edge
// The duct on the opposite edge is effectively forced to be no farther than 50mm from the lower board edge.
// The case design assumes the upper edge of the fan cutout aligns with the outside of that upper duct. 
// The lower duct, starting at xu4_ldi from the board edge, is set to that the outside of the
// lower duct is aligned with the lower edge of the fan cutout.
// The width of the fan mount block is sized to match the fan cutout, so xu4_ldi also then sets the 
// inset for the fan mount from the board edge.  
xu4_ldi = (50+CASE_BWT)-(EXT_FAN_SIZE+2*CASE_FAN_FIT);
//echo ( str("Lower duct inset from XU4 edge is ",xu4_ldi,"mm"));    // uncomment for debug use
//
// fan mount heights
case_fan_omount_bh = 2*EXT_FAN_MHO-CASE_FAN_BRECESS;  // height of fan mount on case bottom when fan is in a cutout
case_fan_omount_th = 2*EXT_FAN_MHO-CASE_FAN_TRECESS;  // height of fan mount on case top when fan is in a cutout
case_fan_imount_bh = EXT_FAN_SIZE/2+CASE_FAN_FIT-CASE_FAN_BRECESS; // bottom mount height set to half the fan for inside mount
case_fan_imount_th = EXT_FAN_SIZE/2+CASE_FAN_FIT-CASE_FAN_TRECESS; // top mount height set to half of the fan for inside mount
// For inside mount, the fan mount serves as some of the airflow ducting - hence the mounts being set to half the fan size
// Note that case_fan_omount_bh and th will be used to govern inside mount heights when no airflow ducting is included
//
// airflow space above XU4 & heatsink
// algorithms here assume XU4_PERCH_T could be *anything* and ultimately drive the heatsink height 
xu4_hstype0_h = CN_USB3_H;                                    // USB3 connector defaults to highest with no XU4 heatsink
xu4_hstype1_h = (IC_SOC_H+XU4_PERCH_T+HS_STOCK_H)>CN_USB3_H?  
                (IC_SOC_H+XU4_PERCH_T+HS_STOCK_H):CN_USB3_H;  // go with higher of heatsink stack-up vs. USB3
xu4_hstype2_h = (IC_SOC_H+XU4_PERCH_T+HS_STOCK_H+HS_STOCK_FAN_H)>CN_USB3_H?
                (IC_SOC_H+XU4_PERCH_T+HS_STOCK_H+HS_STOCK_FAN_H):CN_USB3_H; // go with higher of heatsink stack-up vs. USB3
xu4_hstype3_h = IC_SOC_H+XU4_PERCH_T+HS_PASSIVE_H;            // passive HS defaults to highest 
xu4_hstype4_h = IC_SOC_H+XU4_PERCH_T+HS_GOLD_H;               // gold HS defaults to highest
xu4_hstype5_h = IC_SOC_H+XU4_PERCH_T+HS_GOLD_H;               // gold HS defaults to highest
xu4_hstype6_h = (IC_SOC_H+XU4_PERCH_T+HS_PLATE_H)>CN_USB3_H?
                (IC_SOC_H+XU4_PERCH_T+HS_PLATE_H):CN_USB3_H;  // go with higher of heatsink stack-up vs. USB3
case_top_material = CASE_FMOUNT_TYPE==1?CASE_FMOUNT_THICK:CASE_THICK; // top plate fan mount has more material depth
// clearance is the gap between the applicable XU4 component height and material thickness of the top plate 
if ( XU4_HS_TYPE == 0 )  // no heatsink
  echo ( str("Airflow clearance above XU4 highest component is ",case_th-case_top_material-xu4_hstype0_h,"mm"));
if ( XU4_HS_TYPE == 1 )  // stock low profile heatsink without fan
  echo ( str("Airflow clearance above XU4 highest component is ",case_th-case_top_material-xu4_hstype1_h,"mm"));
if ( XU4_HS_TYPE == 2 )  // stock low profile heatsink and fan
  echo ( str("Airflow clearance above XU4 highest component is ",case_th-case_top_material-xu4_hstype2_h,"mm"));
if ( XU4_HS_TYPE == 3 )  // passive heatsink
  echo ( str("Airflow clearance above XU4 highest component is ",case_th-case_top_material-xu4_hstype3_h,"mm"));
if ( XU4_HS_TYPE == 4 )  // gold heatsink
  echo ( str("Airflow clearance above XU4 highest component is ",case_th-case_top_material-xu4_hstype4_h,"mm"));
if ( XU4_HS_TYPE == 5 )  // rotated gold heatsink
  echo ( str("Airflow clearance above XU4 highest component is ",case_th-case_top_material-xu4_hstype5_h,"mm"));
if ( XU4_HS_TYPE == 6 )  // rotated gold heatsink
  echo ( str("Airflow clearance above XU4 highest component is ",case_th-case_top_material-xu4_hstype6_h,"mm"));
//
// ERRATA: Missing calculations for suggested screw lengths, but mounting options makes that complex
//
// end calculated parameters

// end PARAMETERS

//----------------------------------------------------------
// case_top ();                   // uncomment for local debug if all in top level geometry is disabled
module case_top() {
  // case_top defines the case top design
  // The design orientation is "as printed" to simplify z-axis data  
  // The result is flipped upside down for installation on a case bottom. 

  difference() {

    // form the solids
    union () { 
        
      // start with a solid plate      
      rounded_plate ( size=[ case_width, case_depth, CASE_THICK ], radius=case_cr );
        
      // add fan mounting provision, if specified
      if ( CASE_FMOUNT_TYPE == 1 ) {         // fan mounts flat on case top
        // add some material depth to where the fan mounting hardware goes through case top; bore holes later
        for ( x=[CASE_XU4_INSET+XU4_HS_CENTER_X-EXT_FAN_SIZE/2+EXT_FAN_MHO,
                 CASE_XU4_INSET+XU4_HS_CENTER_X+EXT_FAN_SIZE/2-EXT_FAN_MHO ] )
          for ( y=[case_depth-(CASE_XU4_INSET+XU4_HS_CENTER_Y-EXT_FAN_SIZE/2+EXT_FAN_MHO),
                   case_depth-(CASE_XU4_INSET+XU4_HS_CENTER_Y+EXT_FAN_SIZE/2-EXT_FAN_MHO) ] )
            translate ([ x, y, CASE_THICK-MSA ])
              cylinder ( d=CASE_TSO_OD, h=CASE_FMOUNT_THICK-CASE_THICK+MSA, $fn=24 );
      }
      if ( CASE_FMOUNT_TYPE > 1 ) {          // some form of fan mount on case end is involved
        intersection () {                    // intersection will be used to trim the mount to the desired height
          // fan mount flips to the opposite sidewall to keep it positioned above the case bottom fan mount
          translate ([ case_width, 
                      case_depth-(CASE_XU4_INSET+xu4_ldi)-EXT_FAN_SIZE-2*CASE_FAN_FIT, 
                      CASE_THICK+2*CASE_FAN_FIT+EXT_FAN_SIZE-CASE_FAN_TRECESS-CASE_FAN_BRECESS-MSA ]) {
            rotate ([ -90, 0, 90 ]) {        // get the fan mount vertical
              fan_mount_plate ();
            }  // end rotate the plate to vertical orientation 
          }  // end translate
          
          // trim down the fan mount to the specified mount height using intersection
          if ( CASE_FMOUNT_TYPE == 2 ) {     // fan mount is placed outside the fan     
            translate ([ case_width-CASE_FMOUNT_THICK-MSA, -MSA, -MSA ]) 
              cube ([ CASE_FMOUNT_THICK+MDA, case_depth+MDA, CASE_THICK+case_fan_omount_th+MDA ]);
          }  // end if CASE_FMOUNT_TYPE is outside mount
          if ( CASE_FMOUNT_TYPE == 3 ) {     // fan mount is placed inside the fan     
            if ( CASE_DUCT_TYPE == 1 ) {     // include airflow ducting
              // go with tall fan mount so fan mount serves as part of the ducting   
              translate ([ case_width-CASE_FMOUNT_THICK-MSA, -MSA, -MSA ]) 
                cube ([ CASE_FMOUNT_THICK+MDA, case_depth+MDA, CASE_THICK+case_fan_imount_th+MSA ]);
            }  // end if CASE_DUCT_TYPE is include airflow ducting
            if ( CASE_DUCT_TYPE == 0 ) {     // no airflow ducting
              // go with shorter fan mount since fan mount doesn't have to serve as ducting  
              translate ([ case_width-CASE_FMOUNT_THICK-MSA, -MSA, -MSA ]) 
                cube ([ CASE_FMOUNT_THICK+MDA, case_depth+MDA, CASE_THICK+case_fan_omount_th+MSA ]);
            }  // end if CASE_DUCT_TYPE is include airflow ducting
          }  // end if CASE_FMOUNT_TYPE is inside mount
        }  // end intersection   
      }  // end if CASE_FMOUNT_TYPE is some form of fan mount on case end

      // add low wall under fan if necessary to fill gap caused by CASE_FAN_TRECESS < 0
      if ( CASE_FMOUNT_TYPE == 2 ) {        // fan mount is placed outside the fan     
        if ( CASE_FAN_TRECESS < 0 ) {       // mount has been raised up above case top plate
          // fill the gap under the exposed fan face
          translate ([ case_width-CASE_FMOUNT_THICK-2*CASE_FAN_FIT-EXT_FAN_T,
                       case_depth-(CASE_XU4_INSET+xu4_ldi)-EXT_FAN_SIZE-2*CASE_FAN_FIT, CASE_THICK-MSA])
            cube ([ CASE_BWT, EXT_FAN_SIZE + 2*CASE_FAN_FIT, -CASE_FAN_TRECESS+MSA ]);
          // fill the gap under the i2s connector side of fan
          translate ([ case_width-CASE_FMOUNT_THICK-2*CASE_FAN_FIT-EXT_FAN_T,
                       case_depth-(CASE_XU4_INSET+xu4_ldi)-EXT_FAN_SIZE-2*CASE_FAN_FIT, CASE_THICK-MSA])
            cube ([ EXT_FAN_T + 2*CASE_FAN_FIT, CASE_BWT, -CASE_FAN_TRECESS+MSA ]);
          // fill the gap under the boot select switch side of fan
          translate ([ case_width-CASE_FMOUNT_THICK-2*CASE_FAN_FIT-EXT_FAN_T,
                       case_depth-(CASE_XU4_INSET+xu4_ldi)-CASE_TWT, CASE_THICK-MSA])
            cube ([ EXT_FAN_T + 2*CASE_FAN_FIT, CASE_BWT, -CASE_FAN_TRECESS+MSA ]);
        }  // end if CASE_FAN_TRECESS < 0
      }  // end if CASE_FMOUNT_TYPE is outside mount
          
      // add riser cones at XU4 mounting standoffs if specified
      if ( CASE_TSO_RISER_HEIGHT > 0 ) {
        // XU4 PCB mounting holes
        for ( x=[ XU4_PCB_MHO, XU4_PCB_WIDTH-XU4_PCB_MHO ]) {
          for ( y=[ XU4_PCB_MHO, XU4_PCB_DEPTH-XU4_PCB_MHO ]) {
            translate ([ x+CASE_XU4_INSET, y+CASE_XU4_INSET, CASE_THICK-MSA]) 
              cylinder ( d1=case_tso_riser_od, d2=CASE_TSO_OD, h=CASE_TSO_RISER_HEIGHT+MSA, $fn=24 );
          }  // end for loop on y
        }  // end for loop on x
      }  // end if CASE_BSO_RISER_HEIGHT is there to deal with 
        
      // add top standoffs, if specified
      if ( CASE_TSO_TYPE == 1 ) {             // integrate case top standoffs in case top design
        // XU4 PCB mounting holes
        for ( x=[ XU4_PCB_MHO, XU4_PCB_WIDTH-XU4_PCB_MHO ]) {
          for ( y=[ XU4_PCB_MHO, XU4_PCB_DEPTH-XU4_PCB_MHO ]) {
            translate ([ x+CASE_XU4_INSET, y+CASE_XU4_INSET, CASE_THICK+CASE_TSO_RISER_HEIGHT-MSA]) 
              cylinder ( d=CASE_TSO_OD, h=case_tso_height+MSA, $fn=CASE_TSO_FN );  // user specified hex vs round
          }  // end for loop on y
        }  // end for loop on x
      }  // end if CASE_BSO_TYPE says standoffs are integrated into case bottom
 
      // Add case top reinforcement ribs from corner post to corner post on both sides
      // Height of the ribs are limted to 1mm or issues can arise with sinking UART holder into top plate      
      translate ([ CASE_XU4_INSET + XU4_PCB_MHO, CASE_XU4_INSET, CASE_THICK - MSA ])
        rounded_wall ( length=XU4_PCB_WIDTH - 2*XU4_PCB_MHO, thick=CASE_TWT, height=CASE_RIB_HEIGHT+MSA );
      translate ([ CASE_XU4_INSET + XU4_PCB_MHO, CASE_XU4_INSET + XU4_PCB_DEPTH - CASE_TWT, CASE_THICK - MSA ])
        rounded_wall ( length=XU4_PCB_WIDTH - 2*XU4_PCB_MHO, thick=CASE_TWT, height=CASE_RIB_HEIGHT+MSA );
 
      // Add UART holder, if specified 
      if ( CASE_UHOLDER_TYPE == 1 ) {      // place UART on RJ45 side
        translate ([ CASE_XU4_INSET+17.5, case_depth, CASE_THICK-1 ])   // -1 to sink it into the case a bit
          rotate ([ 0, 0, 180 ]) 
            uart_holder( mtab=0 );         // UART holder without mounting tabs
      }  // end if CASE_UHOLDER_TYPE is place UART on RJ45 side
      if ( CASE_UHOLDER_TYPE == 2 ) {      // place UART on USB3 side
        translate ([ CASE_XU4_INSET+17.5, 0, CASE_THICK-1 ])             // -1 to sink it into the case a bit
          uart_holder( mtab=0 );           // UART holder without mounting tabs
      }  // end if CASE_UHOLDER_TYPE is place UART on USB3 side
      
      // add rtc battery holder, if specified
      if ( CASE_BHOLDER_TYPE == 1 ) {      // place battery holder on RJ45 side
        translate ([ CASE_XU4_INSET+17, case_depth-(CASE_XU4_INSET+17), 0 ]) {  
          batt_holder( mtab=0 );           // battery holder without mounting tabs
        }  // end translate
      }  // end if CASE_BHOLDER_TYPE is place battery holder on RJ45 side
      if ( CASE_BHOLDER_TYPE == 2 ) {      // place battery holder on USB3 side
        translate ([ CASE_XU4_INSET+17, CASE_XU4_INSET+17, 0 ]) {   
          batt_holder( mtab=0 );           // battery holder without mounting tabs
        }  // end translate
      }// end if CASE_BHOLDER_TYPE is place battery holder on SUB3 side

    }  // end union on forming the solids for the case bottom
    
    // remove cutout for fan recess, if specified and necessary
    if ( CASE_FMOUNT_TYPE == 2 ) {         // outside mount fan option
      if ( CASE_FAN_TRECESS > 0 ) {       // fan cutout is necessary
        // Position top fan cutout on revserse sidewall from bottom  
        translate ([ 2*CASE_XU4_INSET+XU4_PCB_WIDTH+CASE_FAN_ADJ,
                    case_depth-(CASE_XU4_INSET+xu4_ldi)-(EXT_FAN_SIZE+2*CASE_FAN_FIT), -MSA ]) 
          cube ([ EXT_FAN_T+2*CASE_FAN_FIT, EXT_FAN_SIZE+2*CASE_FAN_FIT, CASE_THICK+MDA ]);
      }  // end if cutout is necessary
    }  // end if CASE_FMOUNT_TYPE is case fan with outside fan mount
    
    // remove cutout and mounting holes for fan mounted flat on case top, if specified
    if ( CASE_FMOUNT_TYPE == 1 ) {         // fan mounts flat on case top
      // bore the large hole for fan airflow
      translate ([ CASE_XU4_INSET+XU4_HS_CENTER_X, case_depth-(CASE_XU4_INSET+XU4_HS_CENTER_Y), -MSA ])
        cylinder ( d=EXT_FAN_SIZE-2*EXT_FAN_WALL, h=CASE_THICK+MDA, $fn=180 );
      // bore the four mounting holes in the case top
      for ( x=[CASE_XU4_INSET+XU4_HS_CENTER_X-EXT_FAN_SIZE/2+EXT_FAN_MHO,
               CASE_XU4_INSET+XU4_HS_CENTER_X+EXT_FAN_SIZE/2-EXT_FAN_MHO ] ) {
        for ( y=[case_depth-(CASE_XU4_INSET+XU4_HS_CENTER_Y-EXT_FAN_SIZE/2+EXT_FAN_MHO),
                   case_depth-(CASE_XU4_INSET+XU4_HS_CENTER_Y+EXT_FAN_SIZE/2-EXT_FAN_MHO) ] ) {
          if ( CASE_FMOUNT_HOLE_TYPE == 0 ) {   // clearance hole for iso pin
            translate ([ x, y, -MSA ])
              cylinder ( d=CASE_FMOUNT_ISO_HID_CLEAR, h=CASE_FMOUNT_THICK+MDA, $fn=18 );
          }  // end if CASE_FMOUNT_HOLE_TYPE is clearance hole for iso pin
          if ( CASE_FMOUNT_HOLE_TYPE == 1 ) {   // clearance hole for case screw
            translate ([ x, y, -MSA ])
              cylinder ( d=CASE_FMOUNT_CSCREW_HID_CLEAR, h=CASE_FMOUNT_THICK+MDA, $fn=18 );
          }  // end if CASE_FMOUNT_HOLE_TYPE is clearance hole for case screw
          if ( CASE_FMOUNT_HOLE_TYPE == 2 ) {   // clearance hole for M3 screw
            translate ([ x, y, -MSA ])
              cylinder ( d=M3_HID_CLEAR, h=CASE_FMOUNT_THICK+MDA, $fn=18 );
          }  // end if CASE_FMOUNT_HOLE_TYPE is clearance hole for M3 screw
          if ( CASE_FMOUNT_HOLE_TYPE == 3 ) {   // hole for threading M3, use polyhole method
            translate ([ x, y, -MSA ])
              polyhole( d=M3_HID_THREADED, h=CASE_FMOUNT_THICK+MDA );
          }  // end if CASE_FMOUNT_HOLE_TYPE is hole for threading
        }  // end for loop on y
      }  // end for loop on x
      // form a rounded channel for fan wires to pass through case top
      // to keep the channel short, place it where fan wall is thinnest
      wire_hid = 4;  // inside diameter for a wire access hole set to accomodate 4mm sleeving OD on Noctua fan
      // form the round end
      translate ([ CASE_XU4_INSET+XU4_HS_CENTER_X-EXT_FAN_SIZE/2-wire_hid/2,
                   case_depth - (CASE_XU4_INSET+XU4_HS_CENTER_Y),
                   -MSA ])
        cylinder ( d=wire_hid, h=CASE_THICK+MDA, $fn=18 );
      // notch out material under fan edge
      translate ([ CASE_XU4_INSET+XU4_HS_CENTER_X-EXT_FAN_SIZE/2-wire_hid/2,
                   case_depth - (CASE_XU4_INSET+XU4_HS_CENTER_Y)-wire_hid/2,
                   -MSA ])
        cube ([ wire_hid, wire_hid, CASE_THICK+MDA ], center=false );
    }  // end if CASE_FMOUNT_TYPE is fan mounts on case top
    
    // remove heatsink cutout, if specified
    // any desired grid pattern is added back in later
    if ( CASE_FMOUNT_TYPE != 1 ) {         // can't do a heatsink cutout if fan mounted on case top
      if ( CASE_HS_CUTOUT_TYPE != 0 ) {    // requires a cutout
        translate ([ CASE_XU4_INSET+XU4_HS_CENTER_X-CASE_HS_CUTOUT_SIZE/2, 
                     case_depth-(CASE_XU4_INSET+XU4_HS_CENTER_Y+CASE_HS_CUTOUT_SIZE/2), 
                     -MSA ])
          // cutout is rounded a bit to minimize corner riser shoulder encroaching into cutout void
          rounded_plate ( size=[ CASE_HS_CUTOUT_SIZE, CASE_HS_CUTOUT_SIZE, CASE_THICK+MDA ], radius=2 );
      }  // end if XU4_HEATSINK_CUTOUT_TYPE requires a cutout
    }  // end if CASE_FMOUNT_TYPE is anything but fan mounted on case top
    
    // bore the XU4 mounting/case assembly screw holes
    for ( x=[ XU4_PCB_MHO, XU4_PCB_WIDTH-XU4_PCB_MHO ] ) {
      for ( y=[ XU4_PCB_MHO, XU4_PCB_DEPTH-XU4_PCB_MHO ] ) {
        if ( CASE_XU4_THOLE_TYPE == 5 ) {     // limited depth hole for threading M3
          translate ([ CASE_XU4_INSET+x, CASE_XU4_INSET+y, 
                      CASE_THICK+CASE_TSO_RISER_HEIGHT+case_tso_height-CASE_XU4_THOLE_TD ])
            polyhole( d=M3_HID_THREADED, h=CASE_XU4_THOLE_TD+MSA );
        }  // end if CASE_XU4_THOLE_TYPE is limited depth hole for threading M3
        else {                                // some form of through-hole is needed
          translate ([ CASE_XU4_INSET+x, CASE_XU4_INSET+y, -MSA ])
            form_screw_thru_hole ( type=CASE_XU4_THOLE_TYPE, 
                                   oh=CASE_THICK+CASE_TSO_RISER_HEIGHT+case_tso_height+MDA, 
                                   hfn=CASE_XU4_HFN );
        }  // end else some form of through-hole is needed
      }  // end for loop on y
    }  // end for loop on x
    
  }  // end difference
  
  // add top duct, if specified
  if ( CASE_FMOUNT_TYPE > 1 ) {   // case involves some form of fan mount on case end
    // prepare some algorithm shortcuts
    duct_start_x = CASE_XU4_INSET+XU4_HS_CENTER_X+CASE_HS_CUTOUT_SIZE/2;
    duct_end_x = (CASE_FMOUNT_TYPE==2)?(case_width-CASE_FMOUNT_THICK-EXT_FAN_T-2*CASE_FAN_FIT)
                                    :(case_width-CASE_FMOUNT_THICK);
    if ( CASE_DUCT_TYPE == 1 ) {   // include ducting
      // duct on the i2s connector side
      translate ([ duct_start_x, case_depth-(CASE_XU4_INSET+xu4_ldi)-EXT_FAN_SIZE-2*CASE_FAN_FIT, CASE_THICK-MSA ])
        cube ([ duct_end_x - duct_start_x + MSA, CASE_TWT, case_tduct_i2sl_h ]);
      // Add a short stub of additional duct in front of the i2s connector.  
      // Only take the stub to the same height as the bootsel side.
      // For anything higher, it would be the only part of the case top being printed at the very end
      //   and unless the user applies proper thermal management techniques, 
      //   print quality on that last bit would suffer.
      //   This would especially be true for small or zero values of CASE_FAN_ADJ
      translate ([ (duct_start_x + 6.5), case_depth-(CASE_XU4_INSET+xu4_ldi)-EXT_FAN_SIZE-2*CASE_FAN_FIT, 
                    CASE_THICK+case_tduct_i2sl_h-MSA ])
        cube ([ duct_end_x - (duct_start_x + 6.5)+ MSA, CASE_TWT, case_tduct_bootsel_h-case_tduct_i2sl_h+MSA ]);
      
      // duct on the boot select switch side
      translate ([ duct_start_x, case_depth-(CASE_XU4_INSET+xu4_ldi)-CASE_TWT, CASE_THICK-MSA ])
        cube ([ duct_end_x - duct_start_x + MSA, CASE_TWT, case_tduct_bootsel_h ]);
    }  // end if CASE_DUCT_TYPE is include ducting on case bottom & top
  }  // end if CASE_FMOUNT_TYPE involves some form of fan mount on case end
  
  // add vent plate for heatsink cutout here, if specified.
  // other style vent plates could be added here
  if ( CASE_FMOUNT_TYPE != 1 ) {     // no heatsink cutout to fill if case top has a fan mount on it
    if ( CASE_HS_CUTOUT_TYPE == 2 ) {  // square grid pattern
      translate ([ CASE_XU4_INSET+XU4_HS_CENTER_X, case_depth-(CASE_XU4_INSET+XU4_HS_CENTER_Y), 0 ])
        grid_vent_plate ();
    }  // end if CASE_HS_CUTOUT_TYPE is grid vent plate
  }  // end if CASE_FMOUNT_TYPE is anything but fan mount on case top
     
}  // end module case_top

//----------------------------------------------------------
//case_bottom ();                 // uncomment for local debug if all in top level geometry is disabled
module case_bottom() {
  // case_bottom prepares the case bottom design
  // design is built flat-side down, so it is already a printable orientation 
    
  SHOULDER_H = 2.5;           // height of clearance shoulder at top of upper heatsink standoff
  SHOULDER_ROTATION = 55;     // angle of rotation for the clearance shoulder at top of upper heatsink standoff
      CASE_RIB_HEIGHT = 1.0;           // height of support ribs added to case top and case bottom
    
  difference() {

    // form the solids
    union () { 
        
      // start with a solid plate      
      rounded_plate ( size=[ case_width, case_depth, CASE_THICK ], radius=case_cr );
        
      // add case bottom fan mounting block, if fan mount is specified
      if ( CASE_FMOUNT_TYPE > 1 ) {            // some form of mount at case end is involved
        intersection () {                     // intersection will be used to trim the mount to the desired height
          translate ([ case_width-CASE_FMOUNT_THICK, CASE_XU4_INSET+xu4_ldi, CASE_THICK-MSA ]) {
            rotate ([ 90, 0, 90 ]) {          // get the fan mount vertical
              fan_mount_plate ();
            }  // end rotate the plate to vertical orientation
          }  // end translate
          
          // trim down the fan mount to the specified mount height using intersection
          if ( CASE_FMOUNT_TYPE == 2 ) {      // fan mount is placed outside the fan     
            translate ([ case_width-CASE_FMOUNT_THICK-MSA, -MSA, -MSA ]) 
              cube ([ CASE_FMOUNT_THICK+MDA, case_depth+MDA, CASE_THICK+case_fan_omount_bh+MDA ]);
            if ( CASE_DUCT_TYPE == 1 ) {      // airflow ducting is specified
              if ( CASE_FAN_BRECESS < 0 ) {   // mount has been raised up above case bottom plate
                translate ([ case_width-CASE_FMOUNT_THICK-2*CASE_FAN_FIT-EXT_FAN_T,
                             CASE_XU4_INSET+xu4_ldi, CASE_THICK-MSA +30])
                  cube ([ CASE_BWT, EXT_FAN_SIZE + 2*CASE_FAN_FIT, -CASE_FAN_BRECESS+MSA ]);
              }  // end if CASE_FAN_BRECESS < 0
            }  // end if CASE_DUCT_TYPE is include ducting
          }  // end if CASE_FMOUNT_TYPE is outside mount
          if ( CASE_FMOUNT_TYPE == 3 ) {      // fan mount is placed inside the fan     
            if ( CASE_DUCT_TYPE == 1 ) {      // include airflow ducting
              // go with tall fan mount so fan mount serves as part of the ducting   
              translate ([ case_width-CASE_FMOUNT_THICK-MSA, -MSA, -MSA ]) 
                cube ([ CASE_FMOUNT_THICK+MDA, case_depth+MDA, CASE_THICK+case_fan_imount_bh+MSA ]);
            }  // end if CASE_DUCT_TYPE is include airflow ducting
            if ( CASE_DUCT_TYPE == 0 ) {     // no airflow ducting
              // go with shorter fan mount since fan mount doesn't have to serve as ducting  
              translate ([ case_width-CASE_FMOUNT_THICK-MSA, -MSA, -MSA ]) 
                cube ([ CASE_FMOUNT_THICK+MDA, case_depth+MDA, CASE_THICK+case_fan_omount_bh+MSA ]);
            }  // end if CASE_DUCT_TYPE is include airflow ducting
          }  // end if CASE_FMOUNT_TYPE is inside mount
        }  // end intersection
      }  // end if CASE_FMOUNT_TYPE is other than none

      // add low wall under fan if necessary to fill gap caused by CASE_FAN_BRECESS < 0
      if ( CASE_FMOUNT_TYPE == 2 ) {       // fan mount is placed outside the fan     
        if ( CASE_FAN_BRECESS < 0 ) {      // mount has been raised up above case bottom plate
          // add an air dam to fill the gap under the fan size
          translate ([ case_width-CASE_FMOUNT_THICK-2*CASE_FAN_FIT-EXT_FAN_T,
                       CASE_XU4_INSET+xu4_ldi, CASE_THICK-MSA])
            cube ([ CASE_BWT, EXT_FAN_SIZE + 2*CASE_FAN_FIT, -CASE_FAN_BRECESS+MSA ]);
          // add a short wall for appearance sake at boot sel switch side of fan
          translate ([ case_width-CASE_FMOUNT_THICK-2*CASE_FAN_FIT-EXT_FAN_T,
                       (CASE_XU4_INSET+xu4_ldi), CASE_THICK-MSA])
            cube ([ EXT_FAN_T + 2*CASE_FAN_FIT, CASE_BWT, -CASE_FAN_BRECESS+MSA ]);
          // add a short wall for appearance sake at i2s side of fan
          translate ([ case_width-CASE_FMOUNT_THICK-2*CASE_FAN_FIT-EXT_FAN_T,
                       CASE_XU4_INSET+50, CASE_THICK-MSA])
            cube ([ EXT_FAN_T + 2*CASE_FAN_FIT, CASE_BWT, -CASE_FAN_BRECESS+MSA ]);
        }  // end if CASE_FAN_BRECESS < 0
      }  // end if CASE_FMOUNT_TYPE is outside mount
          
      // add stanchions for through-case heatsink mounting if specified
      if ( CASE_HS_MOUNT_TYPE == 1 ) {    // through-case mounting without standoff spacers
        // upper heatsink mounting hole by the USB3.0 connector
        translate ([ XU4_HS_MH2_OX+CASE_XU4_INSET, XU4_HS_MH2_OY+CASE_XU4_INSET, CASE_THICK-MSA ])
          cylinder ( d1=20, d2=CASE_BSO_OD, h=CASE_HS_STANCHION_H+MSA, $fn=36 );
        // lower heatsink mounting hole by the boot select switch
        if ( CASE_FMOUNT_TYPE == 2 ) {   // outside mount 
          // intersect lower stanchion with cube ending at fan cutout location to "lopp off" excess in case there is no cutout
          //   there will be no cutout if CASE_FAN_BRECESS is 0, but we don't want stanchion extending into area for fan
          intersection () {   
            translate ([ XU4_HS_MH1_OX+CASE_XU4_INSET, XU4_HS_MH1_OY+CASE_XU4_INSET, CASE_THICK-MSA ]) 
              cylinder ( d1=22, d2=CASE_BSO_OD, h=CASE_HS_STANCHION_H+MSA, $fn=36 );
            cube ([ XU4_PCB_WIDTH+2*CASE_XU4_INSET+CASE_FAN_ADJ, case_depth, CASE_BH ]);
          }  // end intersection with case volume
        }  // end if CASE_FMOUNT_TYPE is outside mount
        else {   // no fan mount inside mount 
          // instersect lower stanchion with a cube sized to case volume to "lopp off" excess stanchion 
          intersection () {   
            translate ([ XU4_HS_MH1_OX+CASE_XU4_INSET, XU4_HS_MH1_OY+CASE_XU4_INSET, CASE_THICK-MSA ]) 
              cylinder ( d1=22, d2=CASE_BSO_OD, h=CASE_HS_STANCHION_H+MSA, $fn=36 );
            cube ([ case_width, case_depth, CASE_BH ]);
          }  // end intersection with case volume
        }  // end else CASE_FMOUNT_TYPE involves no fan mount on case end
      }  // end if CASE_HS_MOUNT_TYPE is through-case without standoff spacers 
        
      // add riser cones at mounting standoffs if specified
      if ( CASE_BSO_RISER_HEIGHT > 0 ) {
        // XU4 PCB mounting holes
        for ( x=[ XU4_PCB_MHO, XU4_PCB_WIDTH-XU4_PCB_MHO ]) {
          for ( y=[ XU4_PCB_MHO, XU4_PCB_DEPTH-XU4_PCB_MHO ]) {
            translate ([ x+CASE_XU4_INSET, y+CASE_XU4_INSET, CASE_THICK-MSA]) 
              cylinder ( d1=case_bso_riser_od, d2=CASE_BSO_OD, h=CASE_BSO_RISER_HEIGHT+MSA, $fn=24 );
          }  // end for loop on y
        }  // end for loop on x
        // heatsink mounting holes 
        if ( CASE_HS_MOUNT_TYPE == 2 ) {  // heatsink mount is through case with standoff
          // upper heatsink mounting hole
          translate ([ XU4_HS_MH2_OX+CASE_XU4_INSET, XU4_HS_MH2_OY+CASE_XU4_INSET, CASE_THICK-MSA ])
            rotate ([ 0, 0, SHOULDER_ROTATION ])  // rotate simply to keep facets on standoff segments lined up 
              cylinder ( d1=case_bso_riser_od, d2=CASE_BSO_OD, h=CASE_BSO_RISER_HEIGHT+MSA, $fn=24 );
          // lower heatsink mounting hole     
          translate ([ XU4_HS_MH1_OX+CASE_XU4_INSET, XU4_HS_MH1_OY+CASE_XU4_INSET, CASE_THICK-MSA ]) 
            cylinder ( d1=case_bso_riser_od, d2=CASE_BSO_OD, h=CASE_BSO_RISER_HEIGHT+MSA, $fn=24 );
        }  // end if CASE_HS_MOUNT_TYPE is through case with standoff
      }  // end if CASE_BSO_RISER_HEIGHT is there to deal with 
        
      // add bottom standoffs if specified
      if ( CASE_BSO_TYPE == 1 ) {     // include case bottom standoffs in case bottom design
        // XU4 PCB mounting holes
        for ( x=[ XU4_PCB_MHO, XU4_PCB_WIDTH-XU4_PCB_MHO ]) {
          for ( y=[ XU4_PCB_MHO, XU4_PCB_DEPTH-XU4_PCB_MHO ]) {
            translate ([ x+CASE_XU4_INSET, y+CASE_XU4_INSET, CASE_THICK+CASE_BSO_RISER_HEIGHT-MSA]) 
              cylinder ( d=CASE_BSO_OD, h=case_bso_height+MSA, $fn=CASE_BSO_FN );  // user specified hex vs round
          }  // end for loop on y
        }  // end for loop on x
        // heatsink mounting holes
        if ( CASE_HS_MOUNT_TYPE == 2 ) {  // heatsink mount is through case with standoff
          // lower heatsink mounting hole     
          translate ([ XU4_HS_MH1_OX+CASE_XU4_INSET, XU4_HS_MH1_OY+CASE_XU4_INSET, CASE_THICK+CASE_BSO_RISER_HEIGHT-MSA ]) 
            cylinder ( d=CASE_BSO_OD, h=case_bso_height+MSA, $fn=CASE_BSO_FN );
          // upper heatsink mounting hole
          // requires cutaway of the standoff top to clear surface mount components close to hole on rear of XU4
          translate ([ XU4_HS_MH2_OX+CASE_XU4_INSET, XU4_HS_MH2_OY+CASE_XU4_INSET, CASE_THICK+CASE_BSO_RISER_HEIGHT-MSA ]) {
            rotate ([ 0, 0, SHOULDER_ROTATION ]) {   // rotate cutaway to clear SMD components 
              // start with the full OD portion up to the cutaway
              cylinder ( d=CASE_BSO_OD, h=case_bso_height-SHOULDER_H, $fn=CASE_BSO_FN ); // rotate cyl too so facets line up
              // use hull to form the sum of a cutaway standoff and a thinwall standoff
              hull () {
                // form the balance of the standoff, cut in half so only one side remains
                difference () {
                  // start with the rest of the standoff height
                  translate ([ 0, 0, case_bso_height-SHOULDER_H ])
                    cylinder ( d=CASE_BSO_OD, h=SHOULDER_H, $fn=CASE_BSO_FN );
                  // cutaway half of the standoff at a depth of SHOULDER_H
                  translate ([ 0, -(CASE_BSO_OD/2+MSA), case_bso_height-SHOULDER_H ])
                    cube ([ CASE_BSO_OD/2 + MSA, CASE_BSO_OD, SHOULDER_H+MSA ]);
                } // end difference
                // form a thin-wall cylinder
                cylinder ( d=4.4, h=case_bso_height, $fn=CASE_BSO_FN ); 
              }  // end hull
            }  // end rotate
          }  // end translate
        }  // end if CASE_HS_MOUNT_TYPE is through case with standoff 
      }  // end if CASE_BSO_TYPE says standoffs are integrated into case bottom
 
      // add bottom side airflow ducting, if specified
      if ( CASE_FMOUNT_TYPE > 1 ) {   // case involves fan mounted on case end
        if (CASE_DUCT_TYPE == 1) {    // include ducting provisions
          // add duct wall on emmc side, routing the duct to the inside of the emmc
          // round off the starting end
          translate ([ CASE_XU4_INSET+21, CASE_XU4_INSET+19.5+CASE_BWT/2, CASE_THICK-MSA ])
            cylinder ( d=CASE_BWT, h=CASE_BH-CASE_THICK+MSA, $fn=18 );
          translate ([ CASE_XU4_INSET+21, CASE_XU4_INSET+19.5, CASE_THICK-MSA ]) 
            cube ([ 39, CASE_BWT, CASE_BH-CASE_THICK+MSA ]);
          // jog over 8.5mm
          // hull is used instead of a simple segment rotate to prevent mesh discontinuities at segment joints
          hull () {
            translate ([ CASE_XU4_INSET+21+39, CASE_XU4_INSET+19.5, CASE_THICK-MSA ])
              cube ([ MSA, CASE_BWT, CASE_BH-CASE_THICK+MSA ]); 
            translate ([ CASE_XU4_INSET+21+39+8.5, CASE_XU4_INSET+xu4_ldi, CASE_THICK-MSA ])
              cube ([ MSA, CASE_BWT, CASE_BH-CASE_THICK+MSA ]); 
          }  // end hull to form a jog
          // add last segment, ending at PCB if no fan mount. If fan mount, extend it by fan adjust
          if ( CASE_FMOUNT_TYPE < 2 ) {       // no fan mount at case end, so end duct at PCB
            // add segment that takes us just a round-wall radius from the end of the XU4 PCB
            translate ([ CASE_XU4_INSET+21+39+8.5, CASE_XU4_INSET+xu4_ldi, CASE_THICK-MSA ]) 
              cube ([ (CASE_XU4_INSET+XU4_PCB_WIDTH)-(CASE_XU4_INSET+21+39+8.5)-1, CASE_BWT, 
                    CASE_BH-CASE_THICK+MSA ]);
            // round off the fan end
            translate ([ CASE_XU4_INSET+XU4_PCB_WIDTH-1, CASE_XU4_INSET+xu4_ldi+CASE_BWT/2, CASE_THICK-MSA ])
              cylinder ( d=CASE_BWT, h=CASE_BH-CASE_THICK+MSA, $fn=18 );
          } // end if CASE_FMOUNT_TYPE is no fan mount at case end
          else {                             // extend duct to beyond fan adjust
            translate ([ CASE_XU4_INSET+21+39+8.5, CASE_XU4_INSET+xu4_ldi, CASE_THICK-MSA ]) 
              cube ([ (2*CASE_XU4_INSET+XU4_PCB_WIDTH+CASE_FAN_ADJ)-(CASE_XU4_INSET+21+39+8.5)+MSA, CASE_BWT, 
                      CASE_BH-CASE_THICK+MSA ]);
          }  // end else extend duct to beyond fan adjust
          //       
          // add duct wall on USB3.0 side 
          if ( CASE_WCHAN_TYPE == 1 ) {  // wire channel is on USB3.0 side, so shorten duct for wire access around corner post 
            // Having a duct wall at y=50 from board edge is critical to fan alignment with the duct
            translate ([ CASE_XU4_INSET+30, CASE_XU4_INSET+50, CASE_THICK-MSA ])
              rounded_wall ( length=(CASE_XU4_INSET+XU4_PCB_WIDTH)-(CASE_XU4_INSET+30)-9, thick=CASE_BWT, 
                              height=CASE_BH-CASE_THICK+MSA ); 
          }  // end if CASE_WCHAN_TYPE is wire channel on USB3.0 side
          else {                         // no wire channel conflict, so extend duct wall into fan mount
            translate ([ CASE_XU4_INSET+30, CASE_XU4_INSET+50, CASE_THICK-MSA ])
              // rounded wall length is adjusted by +1 to compensate for round end
              rounded_wall ( length=(2*CASE_XU4_INSET+XU4_PCB_WIDTH+CASE_FAN_ADJ)-(CASE_XU4_INSET+30)+1+MSA, 
                           thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA ); 
            // force a good mesh between duct and XU4 mounting post and minimize acute angles at mesh
            translate ([ CASE_XU4_INSET+XU4_PCB_WIDTH-XU4_PCB_MHO-2,CASE_XU4_INSET+50, CASE_THICK-MSA ])
              cube ([ 4, 4, CASE_BH-CASE_THICK+MSA ]);
          }  // end else
        }  // end if CASE_DUCT_TYPE is include ducting
      }  // end if CASE_FMOUNT_TYPE is some form of fan mount on case end
      
      // add case bottom reinforcement ribs, ending them flush with the end of the XU4 PCB
      // if emmc cutout is present, we'll just cut out the rib with it later rather than shorten it here
      for (y=[ 15:5:50 ]) {            // start at y=15, space every 5mm through y=50
        translate ([ CASE_XU4_INSET+20, CASE_XU4_INSET+y, CASE_THICK-MSA ])
          rounded_wall ( length=(CASE_XU4_INSET+XU4_PCB_WIDTH)-(CASE_XU4_INSET+20), 
                         thick=CASE_BWT, height=CASE_RIB_HEIGHT+MSA );
      }  // end for loop on y
      
      // add some support struts under the XU4
      if ( CASE_HS_MOUNT_TYPE != 2 ) {     // must be PCB spring clip mount or through case w/stanchion
        // add supports adjacent to heatsink mounting holes
        translate ([ CASE_XU4_INSET+76, CASE_XU4_INSET+25, CASE_THICK-MSA ]) 
          rounded_wall ( length=(CASE_XU4_INSET+XU4_PCB_WIDTH)-(CASE_XU4_INSET+76), 
                         thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA );
        translate ([ CASE_XU4_INSET+20, CASE_XU4_INSET+35, CASE_THICK-MSA ]) 
          rounded_wall ( length=6, thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA );
      }  // end if CASE_HS_MOUNT_TYPE does not include a standoff under the XU4 PCB
      translate ([ CASE_XU4_INSET+50, CASE_XU4_INSET+35, CASE_THICK-MSA ]) 
        rounded_wall ( length=8+CASE_XU4_INSET, thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA );
      translate ([ CASE_XU4_INSET+76, CASE_XU4_INSET+40, CASE_THICK-MSA ]) 
        rounded_wall ( length=(CASE_XU4_INSET+XU4_PCB_WIDTH)-(CASE_XU4_INSET+76), 
                       thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA );
      // place some where ducts would be, just in case ducts are disabled or ducts are N/A for the CASE_FMOUNT_TYPE
      // path of routed duct on emmc side
      translate ([ CASE_XU4_INSET+20, CASE_XU4_INSET+19.5, CASE_THICK-MSA ]) 
        rounded_wall ( length=6, thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA );
      translate ([ CASE_XU4_INSET+50, CASE_XU4_INSET+19.5, CASE_THICK-MSA ]) 
        rounded_wall ( length=8+CASE_XU4_INSET, thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA );
      translate ([ CASE_XU4_INSET+76, CASE_XU4_INSET+xu4_ldi, CASE_THICK-MSA ]) 
        rounded_wall ( length=(CASE_XU4_INSET+XU4_PCB_WIDTH)-(CASE_XU4_INSET+76), 
                         thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA );
      // path of straight duct on USB3.0 side
      translate ([ CASE_XU4_INSET+30, CASE_XU4_INSET+50, CASE_THICK-MSA ])
        rounded_wall ( length=6, thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA ); 
      translate ([ CASE_XU4_INSET+49, CASE_XU4_INSET+50, CASE_THICK-MSA ])
        rounded_wall ( length=6, thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA ); 
      translate ([ CASE_XU4_INSET+68, CASE_XU4_INSET+50, CASE_THICK-MSA ])
        rounded_wall ( length=6, thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA ); 
      

      // add uprights to form fan wiring channel, if specified
      if ( CASE_WCHAN_TYPE == 1 ) {  // include struts for wiring channel on USB3.0 side
        // first one needs to clear some SMD components at about 14mm inset from board edge 
        for (x=[ 15.5:16:63.5 ])      // so start at x=15.5mm and place every 16mm for total of four
          translate ([ CASE_XU4_INSET+x, CASE_XU4_INSET+55, CASE_THICK-MSA ]) 
            channel_wall ( length=4, thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA, base=1 );
      }  // end if CASE_WCHAN_TYPE is wire channel on USB3.0 side
      if ( CASE_WCHAN_TYPE == 2 ) {  // include struts for wiring channel on RJ45 side
        // center one under DC power connector, with additional ones equally spaced under RJ45 and HDMI connectors
        for ( x=[ 11.5:25:61.5 ] )   // start at x=11.5 and space every 25mm for total of three
          translate ([ CASE_XU4_INSET+x, CASE_XU4_INSET+4, CASE_THICK-MSA ]) 
            channel_wall ( length=4, thick=CASE_BWT, height=CASE_BH-CASE_THICK+MSA, base=1 );
      }  // end if CASE_WCHAN_TYPE is wiring channel on RJ45 side
      
    }  // end union on forming the solids for the case bottom
    
    // remove cutout for emmc access, if specified, along with any support ribs crossing through it
    if ( CASE_EMMC_CUTOUT != 0 ) {    // include EMMC cutout
      translate ([ CASE_XU4_INSET+44, CASE_XU4_INSET+3, -MSA ]) 
        cube ([ 16, 16.5, CASE_THICK+CASE_RIB_HEIGHT+MDA ]);
    }  // end if CASE_EMMC_CUTOUT is present
    
    // remove cutout for fan recess if specified and necessary
    if ( CASE_FMOUNT_TYPE == 2 ) {     // outside fan mount
      if ( CASE_FAN_BRECESS > 0  ) {  // fan cutout is necessary 
        translate ([ 2*CASE_XU4_INSET+XU4_PCB_WIDTH+CASE_FAN_ADJ, CASE_XU4_INSET+xu4_ldi, -MSA ]) 
          // stanchions were already lopped off, so no need to include that height here
          cube ([ EXT_FAN_T+2*CASE_FAN_FIT, EXT_FAN_SIZE+2*CASE_FAN_FIT, CASE_THICK+MDA ]);
      }  // end if fan cutout is necessary
    }  // end if CASE_FMOUNT_TYPE is case fan with outside fan mount
    
    // bore the XU4 mounting/case assembly screw holes
    for ( x=[ XU4_PCB_MHO, XU4_PCB_WIDTH-XU4_PCB_MHO ] ) {
      for ( y=[ XU4_PCB_MHO, XU4_PCB_DEPTH-XU4_PCB_MHO ] ) {
        if ( CASE_XU4_BHOLE_TYPE == 5 ) {     // limited depth hole for threading M3
          translate ([ CASE_XU4_INSET+x, CASE_XU4_INSET+y, 
                      CASE_THICK+CASE_BSO_RISER_HEIGHT+case_bso_height-CASE_XU4_BHOLE_TD ])
            polyhole( d=M3_HID_THREADED, h=CASE_XU4_BHOLE_TD+MSA );
        }  // end if CASE_XU4_BHOLE_TYPE is limited depth hole for threading M3
        else {                                // some form of through-hole is needed
          translate ([ CASE_XU4_INSET+x, CASE_XU4_INSET+y, -MSA ])
            form_screw_thru_hole ( type=CASE_XU4_BHOLE_TYPE, 
                                   oh=CASE_THICK+CASE_BSO_RISER_HEIGHT+case_bso_height+MDA, 
                                   hfn=CASE_XU4_HFN );
        }  // end else some form of through-hole is needed
      }  // end for loop on y
    }  // end for loop on x
    
    // bore through-case heatsink mounting holes, if specified (no through-holes for PCB spring clip heatsink mount)
    if ( CASE_HS_MOUNT_TYPE != 0 ) {     // heatsink mount is anything but PCB spring clips
      if ( CASE_HS_HOLE_TYPE == 5 ) {    // heatsink mount hole is limited depth for threading
        translate ([ CASE_XU4_INSET+XU4_HS_MH1_OX, CASE_XU4_INSET+XU4_HS_MH1_OY, CASE_BH-CASE_HS_HOLE_TD-MSA ])
          polyhole( d=M3_HID_THREADED, h=CASE_HS_HOLE_TD+MDA );
        translate ([ CASE_XU4_INSET+XU4_HS_MH2_OX, CASE_XU4_INSET+XU4_HS_MH2_OY, CASE_BH-CASE_HS_HOLE_TD-MSA])  
          polyhole( d=M3_HID_THREADED, h=CASE_HS_HOLE_TD+MDA );
      }  // end if CASE_HS_HOLE_TYPE is limited depth for threading
      else {   // some form of through-hole is desired
        translate ([ CASE_XU4_INSET+XU4_HS_MH1_OX, CASE_XU4_INSET+XU4_HS_MH1_OY, -MSA ])
          form_screw_thru_hole ( type=CASE_HS_HOLE_TYPE, oh=CASE_BH+MDA, hfn=CASE_HS_HFN );  
        translate ([ CASE_XU4_INSET+XU4_HS_MH2_OX, CASE_XU4_INSET+XU4_HS_MH2_OY, -MSA])  
          form_screw_thru_hole ( type=CASE_HS_HOLE_TYPE, oh=CASE_BH+MDA, hfn=CASE_HS_HFN );
      }  // end else some form of through-hole is desired 
    }  // end if CASE_HS_MOUNT_TYPE is anything but spring clips       
  }  // end difference
  
  // add mounting tabs
  if ( CASE_MTAB_TYPE == 1 ) {   // mounting tabs on perimeter of case bottom
    translate ([ case_cr+CASE_MTAB_WIDTH/2, -case_mtab_mho, 0 ])
      mounting_tab (); 
    translate ([ case_width-case_cr-CASE_MTAB_WIDTH/2, -case_mtab_mho, 0 ])
      mounting_tab (); 
    translate ([ case_width-case_cr-CASE_MTAB_WIDTH/2, case_depth+case_mtab_mho, 0 ])
      rotate ([ 0, 0, 180 ])
        mounting_tab (); 
    translate ([ case_cr+CASE_MTAB_WIDTH/2, case_depth+case_mtab_mho, 0 ])
      rotate ([ 0, 0, 180 ])
        mounting_tab (); 
  }  // end if CASE_MTAB_TYPE is tabs on case bottom
  
}  // end module make_bottom

//----------------------------------------------------------
module form_screw_thru_hole ( type, oh, hfn ) {
// Prepare a screw hole and applicable head provision
//  type: 0=clearance, 1=clearance with countersink, 2=clearance with recessed head, 3=clearance with nut trap, 4=threaded
//  oh: overall height of the screw hole
//  hfn: $fn factor to be applied to hole, if clearance type
//  Other hole sizing will be based on global parameters
// Needs to be applied as part of a difference for the hole to appear
// Hole is centered at xy=0, starts at z=0 and grows positive z.  
//   Recess for countersink, screw head, or nut trap all start at z=0
//   Apply rotate x or y by 180 to flip the hole upside down
//   Then apply translate to where ever the hole is needed

  if ( type == 0 ) {  // clearance hole only
    cylinder ( d=M3_HID_CLEAR, h=oh, $fn=hfn );                // bore the clearance through-hole
  }  // end if type is clearance hole only
  if ( type == 1 ) {  // clearance hole with countersink for head
    cylinder ( d=M3_HID_CLEAR, h=oh, $fn=hfn );                // bore the clearance through-hole
// ERRATA: hardcoded countersink characteristics only apply to M3 size
    cylinder ( d=M3_HEAD_OD_CLEAR, h=0.6+MSA, $fn=hfn );       // shoulder ring for top of head
    translate ([ 0, 0, 0.6-MSA ]) 
      cylinder ( d1=M3_HEAD_OD_CLEAR, d2=M3_HID_CLEAR, h=1.5+MSA, $fn=hfn ); // the countersink cone
  }  // end if type is clearance hole with countersink for head
  if ( type == 2 ) {  // clearance hole with full head recess
    cylinder ( d=M3_HID_CLEAR, h=oh, $fn=hfn );                // bore the clearance through-hole
    cylinder ( d=M3_HEAD_OD_CLEAR, h=M3_HEAD_RD+MSA, $fn=hfn );// form the head recess
  }  // end if type is clearance hole with full head recess
  if ( type == 3 ) {  // clearance hole with recessed nut trap
    cylinder ( d=M3_HID_CLEAR, h=oh, $fn=hfn );                // bore the clearance through-hole
    cylinder ( d=M3_NUT_AFD_CLEAR/cos(30), h=M3_NUT_RD+MSA, $fn=6 );// form the nut trap
  }  // end if type is clearance hole with recessed nut trap
  if ( type == 4 ) {  // for threading
    polyhole( d=M3_HID_THREADED, h=oh );                     // bore the hole for threading, using polyhole
  }  // end if type is for threading
  
}  // end module form_screw_thru_hole

//----------------------------------------------------------
//make_test_plate();              // uncomment for local debug if all in top level geometry is disabled
module make_test_plate () {
  // make_test_plate can be used to check hole sizes before printing case
  // Slicer settings, filament characteristics, and printer calibration affect results 
  //
  // SUGGESTED TESTS -
  // #1 Hole size for the optional case mounting tabs
  //   Check planned hardware for clearance fit through the printed hole. 
  //   Adjust parameter CASE_MTAB_HID as required  
  // #2 Hole size for M3 threaded holes
  //   Test for adequate self-tap thread and grip of M3 hardware into the hole
  //   Adjust parameter M3_HID_THREADED as required
  // #3 Hole size for M3 clearance holes and M3 nut trap
  //   Printed in case orientation (upside down) for realism and testing any support added in slicing
  //   Test through-hole for adequate clearance with M3 screw
  //     Adjust parameter M3_HID_CLEAR as required
  //   Test nut trap diameter and recess depth with M3 nut
  //     Adjust parameter M3_NUT_AFD_CLEAR for proper nut diameter 
  //     Adjust parameter M3_NUT_RD for proper nut recess depth
  // #4 Hole for recessing M3 screw head (other than flat)
  //   Printed in case orientation (upside down) for realism and testing any support added in slicing
  //   Test for adequate depth and diameter of head recess with M3 screw
  //   Remainder of hole is printed with M3 clearance hole defined by M3_HID_CLEAR
  //   Adjust parameter M3_HEAD_RD to tweak recess depth
  //   Adjust parameter M3_HEAD_OD_CLEAR to tweak recess hole diameter
  //   Note that dimensions vary by head type; original values reflect worst case of all head types
  // #5 Hole for recessing M3 flat-head screws
  //   Printed in case orientation (upside down) for realism and testing any support added in slicing
  //   Test for reasonable flush-fit of M3 flat head screw 
  //   Remainder of hole is printed with M3 clearance hole defined by M3_HID_CLEAR
  //   Best to check with nut tighted on rear of test plate before making changes
  //   Unfortunately, countersink characteristics are currently hardcoded in case_bottom and test_plate modules
  // #6 Hole for case fan mount
  //   Test for adequate fit with the mounting scheme to be used
  //   Note that the hole prints vertically on the test plate; actual fan mount orientation prints this horizontally
  //     Isolation pins ( CASE_FMOUNT_HOLE_TYPE=0 ):
  //       Isolation pins should pull through the hole and fit snuggly against the test plate
  //       Note that the test plate is printed to the same thickness as the fan mount
  //       Adjust parameter CASE_FMOUNT_THICK if the height of the last step in the pin differs from plate thickness.
  //       Adjust parameter CASE_FMOUNT_ISO_HID_CLEAR to allow pin to pass through while still fitting the final pin step
  //     Properly sized self-tapping case fan screws ( CASE_FMOUNT_HOLE_TYPE=1 ):
  //       Fan screws should pass through the test plate as a clearance hole to engage the fan behind it
  //       Adjust parameter CASE_FMOUNT_THICK if the screws are too short to engage the fan properly
  //       Adjust parameter CASE_FMOUNT_CSCREW_HID_CLEAR to provide adequate clearance hole for your fan screws
  //     M3 clearance ( CASE_FMOUNT_HOLE_TYPE = 2 ):
  //       M3 screw should pass through test plate
  //       If necessary, adjust same M3_HID_CLEAR parameter as test hole #3 
  //     M3 screws threading into fan mount ( CASE_FMOUNT_HOLE_TYPE = 3 ):
  //       Test for adequate self-tap thread and grip of M3 hardware into the hole
  //       If necessary, adjust same M3_HID_THREADED parameter as test hole #2
  // (no number) Case fan cutout
  //   Test for adequate fit of the fan to be used with the case bottom
  //     Parameter EXT_FAN_SIZE captures the actual long dimension of your fan. 
  //       Although parameterized, this design will only work with a 40mm fan
  //     Parameter EXT_FAN_T captures the actual thin dimension of the fan. ie, 7, 10, or 20mm
  //     Adjust parameter CASE_FAN_FIT to provide a tighter or looser fit of the fan edge in the cutout
  //       The CASE_FAN_FIT tolerance will be added to all four sides of the cutout
  //     

  // baseline the size of the test plate    
  tplate_thick = CASE_FMOUNT_THICK;
  tplate_sx = 25+EXT_FAN_T+2*CASE_FAN_FIT;
  tplate_sy = 50;
    
  // Send any pertinent info to the openSCAD console pane
  echo ( str("Test plate size is x: ",tplate_sx,"mm, y: ",tplate_sy,"mm, z: ",tplate_thick,"mm"));
    
  difference () {
      
    // start with a flat plate set to case fan mount thickness
    rounded_plate ( size=[ 25+EXT_FAN_T+2*CASE_FAN_FIT, 50, tplate_thick ], radius=2 );
      
    // #6 - case fan mounting hole
    if ( CASE_FMOUNT_HOLE_TYPE == 0 ) {  // clearance hole for iso pin
      translate ([ 3+10, 43+2, -MSA ])
        cylinder ( d=CASE_FMOUNT_ISO_HID_CLEAR, tplate_thick+MDA, $fn=18 );
    }  // end if CASE_FMOUNT_HOLE_TYPE is clearance hole for iso pin
    if ( CASE_FMOUNT_HOLE_TYPE == 1 ) {  // clearance hole for case screw
      translate ([ 3+10, 43+2, -MSA ])
        cylinder ( d=CASE_FMOUNT_CSCREW_HID_CLEAR, tplate_thick+MDA, $fn=18 );
    }  // end if CASE_FMOUNT_HOLE_TYPE is clearance hole for case screw
    if ( CASE_FMOUNT_HOLE_TYPE == 2 ) {  // clearance hole for M3 screw
      translate ([ 3+10, 43+2, -MSA ])
        cylinder ( d=M3_HID_CLEAR, tplate_thick+MDA, $fn=18 );
    }  // end if CASE_FMOUNT_HOLE_TYPE is clearance hole for M3
    if ( CASE_FMOUNT_HOLE_TYPE == 3 ) {  // hole for threading M3, so use polyhole
      translate ([ 3+10, 43+2, -MSA ])
        polyhole( d=M3_HID_THREADED, h=tplate_thick+MDA );
    }  // end if CASE_FMOUNT_HOLE_TYPE is M3 thread hole
      
    // #5 - M3 countersunk flat head recess
    translate ([ 3+8, 35+2, -MSA ])
//      rotate ([ 180, 0, 0 ])
        form_screw_thru_hole ( type=1, oh=tplate_thick+MDA, hfn=CASE_XU4_HFN );
    
    // #4 - M3 full head recess
    translate ([ 3+8, 27+2, -MSA ])
        form_screw_thru_hole ( type=2, oh=tplate_thick+MDA, hfn=CASE_XU4_HFN );
     
    // #3 - M3 clear hole with bonus nut trap
    translate ([ 3+8, 19+2, -MSA ])
//      rotate ([ 180, 0, 0 ])
        form_screw_thru_hole ( type=3, oh=tplate_thick+MDA, hfn=CASE_XU4_HFN );
    
    // #2 - M3 threaded hole
    translate ([ 3+8, 11+2, -MSA ])
      form_screw_thru_hole ( type=4, oh=tplate_thick+MDA );
    
    // #1 - mounting tab hole
    translate ([ 3+8, 3+2, -MSA ])
      cylinder ( d=CASE_MTAB_HID, h=tplate_thick+MDA, $fn=18 );
    
    // fan cutout
    translate ([ 20, (50-(EXT_FAN_SIZE+2*CASE_FAN_FIT))/2, -MSA ])
      cube ([ EXT_FAN_T+2*CASE_FAN_FIT, EXT_FAN_SIZE+2*CASE_FAN_FIT, tplate_thick+MDA ]);
    
  }  // end difference
  
  // add test item numbers to plate
  color ( "lightblue" ) {
    translate ([ 3, 43, tplate_thick - MSA]) 
      linear_extrude( height = 1+MSA )  // 1mm character depth
        text("6",size=5); 
    translate ([ 3, 35, tplate_thick - MSA]) 
      linear_extrude( height = 1+MSA )  // 1mm character depth
        text("5",size=5); 
    translate ([ 3, 27, tplate_thick - MSA]) 
      linear_extrude( height = 1+MSA )  // 1mm character depth
        text("4",size=5); 
    translate ([ 3, 19, tplate_thick - MSA]) 
      linear_extrude( height = 1+MSA )  // 1mm character depth
        text("3",size=5);
    translate ([ 3, 11, tplate_thick - MSA]) 
      linear_extrude( height = 1+MSA )  // 1mm character depth 
        text("2",size=5);
    translate ([ 3, 3, tplate_thick - MSA]) 
      linear_extrude( height = 1+MSA )  // 1mm character depth
        text("1",size=5);
  }  // end color 

}  // end module test_plate

//----------------------------------------------------------
// show_assembly ( xu4_opacity=0.6 );    // uncomment for local debug if all in top level geometry is disabled
module show_assembly ( xu4_opacity ) {
  // show_assembly provides a view of the XU4 sandwiched between the case bottom and case top
  //  xu4_opacity: degree of transparency to be applied to the xu4, ranging from 0 (can't see) to 1 (full opacity)
 
  if ( xu4_opacity > 0 ) {    // xu4 is to be included
    echo ( str("Showing XU4 with ",xu4_opacity*100,"% opacity between case bottom and case top..."));
    // For the case bottom to show through the XU4, the bottom evidently has to be shown first
    make_bottom ();
    translate ( xu4_pcb_origin )
      odroid_xu4( emmc_type=XU4_EMMC_TYPE, sd_type=XU4_USD_TYPE, hs_type=XU4_HS_TYPE, perch_t=XU4_PERCH_T, 
                  opacity=xu4_opacity, a_type=XU4_HS_ADAPT_TYPE );   // show the xu4
    translate ([ 0, case_depth, CASE_BH + XU4_PCB_THICK + case_th ])
      rotate ([ 180, 0, 0 ])
        case_top ();
  }  // end if xu4_opacity is more than zero
  else {                       // xu4 is fully transparent, so don't show it
    echo ("Showing case bottom and case top with XU4 hidden...");
    make_bottom ();
    translate ([ 0, case_depth, CASE_BH + XU4_PCB_THICK + case_th ])
      rotate ([ 180, 0, 0 ])
        case_top ();
  }  // end else xu4 is fully transparent

  // show user provided standoffs
  // default to showing user provided standoffs without any threading
  color("tan") 
    translate ([ CASE_XU4_INSET + XU4_PCB_MHO, CASE_XU4_INSET + XU4_PCB_MHO, CASE_BH+XU4_PCB_THICK ])
      cylinder ( h=CASE_TSO_USER_H, d=CASE_TSO_OD, $fn=6 );  
  color("tan") 
    translate ([ CASE_XU4_INSET + XU4_PCB_WIDTH - XU4_PCB_MHO, CASE_XU4_INSET + XU4_PCB_MHO, CASE_BH+XU4_PCB_THICK ])
      cylinder ( h=CASE_TSO_USER_H, d=CASE_TSO_OD, $fn=6 );  
  color("tan") 
    translate ([ CASE_XU4_INSET + XU4_PCB_WIDTH - XU4_PCB_MHO, CASE_XU4_INSET + XU4_PCB_DEPTH - XU4_PCB_MHO, 
               CASE_BH+XU4_PCB_THICK ])
      cylinder ( h=CASE_TSO_USER_H, d=CASE_TSO_OD, $fn=6 );  
  color("tan") 
    translate ([ CASE_XU4_INSET + XU4_PCB_MHO, CASE_XU4_INSET + XU4_PCB_DEPTH - XU4_PCB_MHO, CASE_BH+XU4_PCB_THICK ])
      cylinder ( h=CASE_TSO_USER_H, d=CASE_TSO_OD, $fn=6 );  

  // show user provided fan
  if ( CASE_FMOUNT_TYPE == 1 ) {                  // fan mounted on case top plate
    color("tan") 
      translate ([ CASE_XU4_INSET + XU4_HS_CENTER_X, 
                  CASE_XU4_INSET + XU4_HS_CENTER_Y, 
                  case_height + CASE_FAN_FIT ]) 
        view_fan( fan_size=EXT_FAN_SIZE, fan_w=EXT_FAN_T, fan_ho=EXT_FAN_MHO, 
                    fan_hid=EXT_FAN_HID, fan_wall=EXT_FAN_WALL, opacity=xu4_opacity );
  }  // end if CASE_FMOUNT_TYPE is fan mounted on case top plate
  if ( CASE_FMOUNT_TYPE == 2 ) {                  // outside type fan mount
    color("tan") 
      translate ([ case_width - CASE_FMOUNT_THICK - CASE_FAN_FIT, 
                  (CASE_XU4_INSET+xu4_ldi)+EXT_FAN_SIZE/2 + CASE_FAN_FIT, 
                  CASE_THICK - CASE_FAN_BRECESS + CASE_FAN_FIT + EXT_FAN_SIZE/2 ]) 
        rotate ([ 0, -90, 0 ])
          view_fan( fan_size=EXT_FAN_SIZE, fan_w=EXT_FAN_T, fan_ho=EXT_FAN_MHO, 
                    fan_hid=EXT_FAN_HID, fan_wall=EXT_FAN_WALL, opacity=xu4_opacity );
  }  // end if CASE_FMOUNT_TYPE is outside type fan mount
  if ( CASE_FMOUNT_TYPE == 3 ) {                  // inside type fan mount
    color("tan") 
      translate ([ case_width, 
                  (CASE_XU4_INSET+xu4_ldi)+EXT_FAN_SIZE/2 + CASE_FAN_FIT, 
                  CASE_THICK - CASE_FAN_BRECESS + CASE_FAN_FIT + EXT_FAN_SIZE/2 ]) 
        rotate ([ 0, 90, 0 ])
          view_fan( fan_size=EXT_FAN_SIZE, fan_w=EXT_FAN_T, fan_ho=EXT_FAN_MHO, 
                    fan_hid=EXT_FAN_HID, fan_wall=EXT_FAN_WALL, opacity=xu4_opacity );
  }  // end if CASE_FMOUNT_TYPE is inside type fan mount
   
  // show user-installed UART strap, if there is a UART mount included in the case design
  if ( CASE_UHOLDER_TYPE == 1) {                  // UART is on RJ45 side
    color("brown") 
      translate ([ CASE_XU4_INSET+17.5, 20, case_height-CASE_THICK-9+1 ])
        rotate ([ 180, 0, 0 ]) 
          uart_strap();
  }  // end if CASE_UHOLDER_TYPE is UART on RJ45 side
  if ( CASE_UHOLDER_TYPE == 2 ) {                 // UART is on USB3 side
    color("brown") 
      translate ([ CASE_XU4_INSET+17.5, case_depth-20, case_height-CASE_THICK-9+1 ])
        rotate ([ 180, 0, 0 ]) 
          uart_strap();
  }  // end if CASE_UHOLDER_TYPE is UART on USB3 side 
  
  // add feet to case bottom, if specified
  if ( CASE_BFOOT_TYPE != 0 ) {                   // view case with feet 
    color ( "tan" ) {
      for ( x=[ XU4_PCB_MHO, XU4_PCB_WIDTH-XU4_PCB_MHO ]) {
        for ( y=[ XU4_PCB_MHO, XU4_PCB_DEPTH-XU4_PCB_MHO ]) {
          if ( CASE_BFOOT_TYPE == 1 )
            translate ([ x+CASE_XU4_INSET, y+CASE_XU4_INSET, 0 ]) 
              rotate ([ 0, 180, 0 ])
                view_720_foot();
          if ( CASE_BFOOT_TYPE == 2 )
            translate ([ x+CASE_XU4_INSET, y+CASE_XU4_INSET, 0 ]) 
              rotate ([ 0, 180, 0 ])
                foot ( nut_trap=0 ) ;
          if ( CASE_BFOOT_TYPE == 3 )
            translate ([ x+CASE_XU4_INSET, y+CASE_XU4_INSET, 0 ]) 
              rotate ([ 0, 180, 0 ])
                foot ( nut_trap=1 ) ;
        }  // end for loop on y
      }  // end for loop on x
    }  // end color
  }  // end if CASE_FOOT_TYPE is Keystone 720 
   
  // show fan mounting pins, if specified
  if ( CASE_FAN_HW_SHOW == 1 ) {                // show assembly with isolator pins used to mount fan
    if ( CASE_FMOUNT_TYPE == 2 ) {              // outside mounted fan in cutout
      color("tan") {
        for ( y=[ CASE_XU4_INSET + xu4_ldi + CASE_FAN_FIT + EXT_FAN_MHO, 
                  CASE_XU4_INSET + xu4_ldi + CASE_FAN_FIT + EXT_FAN_SIZE - EXT_FAN_MHO ]) {
          for ( z=[ CASE_THICK - CASE_FAN_BRECESS + CASE_FAN_FIT + EXT_FAN_MHO,
                    CASE_THICK - CASE_FAN_BRECESS + CASE_FAN_FIT + EXT_FAN_SIZE - EXT_FAN_MHO ]) {
            translate ([ case_width+3.3, y, z ])  // the +3.3 adjusts placement to match hominoid's placement 
              rotate ([ 0, -90, 0 ])
                view_fan_iso_pin();
          }  // end for loop on z
        }  // end for loop on y
      }  // end color
    }  // end if CASE_FMOUNT_TYPE is outside mounted fan in cutout
    if ( CASE_FMOUNT_TYPE == 3 ) {             // inside mounted fan on case end
      color("tan") {
        for ( y=[ CASE_XU4_INSET + xu4_ldi + CASE_FAN_FIT + EXT_FAN_MHO,
                  CASE_XU4_INSET + xu4_ldi + CASE_FAN_FIT + EXT_FAN_SIZE - EXT_FAN_MHO ]) {
          for ( z=[ CASE_THICK - CASE_FAN_BRECESS + CASE_FAN_FIT + EXT_FAN_MHO,
                    CASE_THICK - CASE_FAN_BRECESS + CASE_FAN_FIT + EXT_FAN_SIZE - EXT_FAN_MHO ]) {             
            translate ([ case_width + EXT_FAN_T, y, z ])
              rotate ([ 0, -90, 0 ])
                view_fan_iso_pin();
          }  // end for loop on z
        }  // end for loop on y
      }  // end color
    }  // end if CASE_FMOUNT_TYPE is inside mounted fan on case end
  }  // end if CASE_FAN_HW_SHOW is show isolator pins
  
  // show Keystone 720 bumper or printed foot on fan, if specified
  if ( CASE_FAN_HW_SHOW > 1 ) {                // show assembly with some sort of bumper or foot
    if ( CASE_FMOUNT_TYPE == 2 ) {             // outside mounted fan in cutout
      color("tan") {
        for ( y=[ CASE_XU4_INSET + xu4_ldi + CASE_FAN_FIT + EXT_FAN_MHO,
                  CASE_XU4_INSET + xu4_ldi + CASE_FAN_FIT + EXT_FAN_SIZE - EXT_FAN_MHO ]) {
          for ( z=[ CASE_THICK - CASE_FAN_BRECESS + CASE_FAN_FIT + EXT_FAN_MHO,
                  CASE_THICK - CASE_FAN_BRECESS + CASE_FAN_FIT + EXT_FAN_SIZE - EXT_FAN_MHO ]) {
            translate ([ case_width, y, z ]) {
              rotate ([ 0, 90, 0 ]) {
                if ( CASE_FAN_HW_SHOW == 2 )   // keystone 720
                  view_720_foot();
                if ( CASE_FAN_HW_SHOW == 3 )   // printed foot
// ERRATA: defaults foot here to no nut trap, but printed accessory will actually follow CASE_BFOOT_TYPE
                  foot( nut_trap=0 );
              }  // end rotate
            }  // end translate
          }  // end for loop on z
        }  // end for loop on y
      }  // end color
    }  // end if CASE_FMOUNT_TYPE is outside mounted fan in cutout
    if ( CASE_FMOUNT_TYPE == 3 ) {             // inside mounted fan on case end
      color("tan") {
        for ( y = [ CASE_XU4_INSET + xu4_ldi + CASE_FAN_FIT + EXT_FAN_MHO,
                    CASE_XU4_INSET + xu4_ldi + CASE_FAN_FIT + EXT_FAN_SIZE - EXT_FAN_MHO ]) {
          for ( z = [ CASE_THICK - CASE_FAN_BRECESS + CASE_FAN_FIT + EXT_FAN_MHO,
                      CASE_THICK - CASE_FAN_BRECESS + CASE_FAN_FIT + EXT_FAN_SIZE - EXT_FAN_MHO ]) {
            translate ([ case_width + EXT_FAN_T, y, z ]) {
              rotate ([ 0, 90, 0 ]) {
                if ( CASE_FAN_HW_SHOW == 2 )   // keystone 720
                  view_720_foot();
                if ( CASE_FAN_HW_SHOW == 3 )   // printed foot
// ERRATA: defaults foot here to no nut trap, but printed accessory will actually follow CASE_BFOOT_TYPE
                  foot( nut_trap=0 );
              }  // end rotate
            }  // end translate
          }  // end for loop on z
        }  // end for loop on y
      }  // end color
    }  // end if CASE_FMOUNT_TYPE is inside mounted fan on case end
  }  // end if CASE_FAN_HW_SHOW is show some sort of bumper or foot

}  // end module show_assembly

//----------------------------------------------------------
// show_bottom ( xu4_opacity=0.6 );      // uncomment for local debug if all in top level geometry is disabled
module show_bottom ( xu4_opacity ) {
  // show_bottom displays the xu4 resting on the case bottom
  //  xu4_opacity is the degree of transparency to be used on the xu4
  //    Range of 0 to 1.0, with 0=no xu4 and 1=full opacity

  if ( xu4_opacity > 0 ) {    // xu4 is to be included
    echo ( str("Showing XU4 with ",xu4_opacity*100,"% opacity resting on case bottom..."));
    // For the case bottom to show through the XU4, the bottom evidently has to be shown first
    make_bottom ();
    translate ( xu4_pcb_origin )
      odroid_xu4( emmc_type=XU4_EMMC_TYPE, sd_type=XU4_USD_TYPE, hs_type=XU4_HS_TYPE, perch_t=XU4_PERCH_T, 
                  opacity=xu4_opacity, a_type=XU4_HS_ADAPT_TYPE );   // show the xu4
  }  // end if xu4_opacity is more than zero
  else {                       // xu4 is fully transparent, so don't show it
    echo ("Showing case bottom with XU4 hidden...");
    make_bottom ();
  }  // end else xu4 is fully transparent
}  // end module show_bottom
  
//----------------------------------------------------------
// make_all ();                  // uncomment for local debug if all in top level geometry is disabled
module make_all () {
  // make_all prepares the case top, case bottom, and necessary accessories for printing

  make_case ();
  translate ([ -20, 0, 0 ])      // shift the group to align it with the two case halves
    rotate ([ 0, 0, 90 ])        // rotate the group in line with y-axis
      translate ([ -50, 0, 0 ])  // center the group over y=0
        make_accessories ();     // make the group of enabled accessories
    
}  // end module make_all

//----------------------------------------------------------
// make_accessories ();          // uncomment for local debug if all in top level geometry is disabled
module make_accessories () {
  // make_accessories will prepare anything other than case top and case bottom needed for full assembly
  // if a desired accessory doesn't appear, it means the parameter for that item hasn't enabled it
    
  // bottom side standoffs    
  if ( CASE_BSO_TYPE == 2 ) {   // bottom side standoffs are printed accessory
    for ( y=[ 0:10:30 ]) {      // start at y=0 and space every 10mm for total of four
      translate ([ -12, y, 0 ]) 
        difference (){ 
          cylinder ( d=CASE_BSO_OD, h=case_bso_height, $fn=CASE_BSO_FN );  // user specified hex vs round
          if ( CASE_XU4_BHOLE_TYPE < 4 )     // clearance through-hole is required
            translate ([ 0, 0, -MSA ]) 
              cylinder ( d=M3_HID_CLEAR, h=case_bso_height+MDA, $fn=CASE_XU4_HFN );
          if ( CASE_XU4_BHOLE_TYPE == 4 )     // through-hole for threading
            translate ([ 0, 0, -MSA ]) 
              polyhole ( d=M3_HID_THREADED, h=case_bso_height+MDA );
          // really no reason to provide any support for limited depth standoff accessory
        }  // end difference
    }  // end for loop on y
}  // end if CASE_BSO_TYPE is printed accessory

  // top side standoffs
  if ( CASE_TSO_TYPE == 2 ) {   // top side standoffs are printed accessory
    for ( y=[ 0:10:30 ]) {      // start at y=0 and space every 10mm for total of four                 
      translate ([ 0, y, 0 ]) {
        difference (){ 
          cylinder ( d=CASE_TSO_OD, h=case_tso_height, $fn=CASE_TSO_FN );  // user specified hex vs round
          if ( CASE_XU4_THOLE_TYPE < 4 )     // clearance through-hole is required
            translate ([ 0, 0, -MSA ]) 
              cylinder ( d=M3_HID_CLEAR, h=case_tso_height+MDA, $fn=CASE_XU4_HFN );
          if ( CASE_XU4_THOLE_TYPE == 4 )     // through-hole for threading
            translate ([ 0, 0, -MSA ]) 
              polyhole ( d=M3_HID_THREADED, h=case_tso_height+MDA );
          // really no reason to provide any support for limited thread depth standoff accessory
        }  // end difference
      }  // end translate
    }  // end for loop on y
  }  // end if CASE_TSO_TYPE is printed accessory
  
  // user-installed top side standoffs
  if ( CASE_TSO_USER_TYPE == 1 ) {  // print user installed standoffs as accessory
    for ( y=[ 0:10:30 ]) {          // start at y=0 and space every 10mm for total of four                  
      translate ([ 12, y, 0 ]) {
        difference (){ 
          cylinder ( d=TSO_USER_OD, h=CASE_TSO_USER_H, $fn=TSO_USER_FN );  // user specified hex vs round
          if ( TSO_USER_HOLE_TYPE == 0 )      // clearance through-hole
            translate ([ 0, 0, -MSA ]) 
              cylinder ( d=M3_HID_CLEAR, h=CASE_TSO_USER_H+MDA, $fn=12 );
          if ( TSO_USER_HOLE_TYPE == 1 )     // through-hole for threading
            translate ([ 0, 0, -MSA ]) 
              polyhole( d=M3_HID_THREADED, h=CASE_TSO_USER_H+MDA );
          if ( TSO_USER_HOLE_TYPE == 2 ) {   // limited depth hole for threading
            // upper hole
            translate ([ 0, 0, CASE_TSO_USER_H-TSO_USER_HOLE_TD ]) 
               polyhole( d=M3_HID_THREADED, h=TSO_USER_HOLE_TD+MSA );
            // lower hole
            translate ([ 0, 0, -MSA ]) 
               polyhole ( d=M3_HID_THREADED, h=TSO_USER_HOLE_TD+MSA );
          }  // end if TSO_USER_HOLE_TYPE is limited depth hole for threading
        }  // end difference
      }  // end translate
    }  // end for loop on y
  }  // end if CASE_TSO_USER_TYPE is print user installed standoffs as accessory
    
  // feet, if specified as printed accessory
  if ( CASE_BFOOT_TYPE == 2 ) {  // feet with head recesses
    for ( y=[ 0:12:36 ] ) {
      translate ([ 24, y, 0 ]) 
        foot ( nut_trap = 0 );
    }  // end for loop on y
  }  // end if CASE_BFOOT_TYPE is feet with head recesses
  if ( CASE_BFOOT_TYPE == 3 ) {  // feet with nut traps
    for ( y=[ 0:12:36 ] ) {
      translate ([ 24, y, 0 ]) 
        foot ( nut_trap = 1 );
    }  // end for loop on y
  }  // end if CASE_BFOOT_TYPE is feet with nut traps
    
  // heatsink adapter, if required
  if ( XU4_HS_TYPE == 5 ) {    // gdstime gold heatsink with custom adapters
    translate ([ 60, 20, 0 ]) 
      heatsink_adapter_gold( type=XU4_HS_ADAPT_TYPE, opacity=1 );
  }  // end if XU4_HS_TYPE is gdstime gold with custom adapters
  
  // heatsink spacer, if specified
  if ( HS_SPACER_TYPE == 1 ) {  // include a heatsink spacer accessory
    translate ([ 50, 20, 0 ])
      heatsink_spacer ( size=HS_SPACER_SIZE, height=HS_SPACER_HEIGHT, wt=HS_SPACER_WT );
    translate ([ 70, 20, 0 ])
      heatsink_spacer ( size=HS_SPACER_SIZE, height=HS_SPACER_HEIGHT, wt=HS_SPACER_WT );
  }  // end if HS_SPACER_TYPE is include a spacer
    
  // UART strap, if required
  if ( CASE_UHOLDER_TYPE != 0 ) {  // any kind of uart is used
    translate ([ 0, 40, 0 ])
      uart_strap();
  }  // end if CASE_UHOLDER_TYPE is any kind of uart used

  // UART holder, if specified as printed accessory
  if ( CASE_UHOLDER_TYPE == 3 ) {  // include UART holder as printed accessory, with mounting tabs
    translate ([ 0, 47, 0 ])
//      rotate ([ 0, 0, -90 ])
        uart_holder( mtab=1 );
  }  // end if CASE_UHOLDER_TYPE is print UART holder as accessory with mounting tabs
  
  // RTC battery holder, if specified as printed accessory
  if ( CASE_BHOLDER_TYPE == 3 ) {  // include battery holder as printed accessory, with mounting tabs
    translate ([ 60, 57, 0 ])
      batt_holder( mtab=1 );
  }  // end if CASE_BHOLDER_TYPE is print battery holder as accessory with mounting tabs

}  // end module make_accessories

//----------------------------------------------------------
// make_case ();                 // uncomment for local debug if all in top level geometry is disabled
module make_case () {
  // make_case will orient and position the case bottom and case top for printing flat side down
    
  // case bottom
  make_bottom();
  // case top
  translate ([0, -case_depth - 1.5*CASE_MTAB_WIDTH, 0 ]) 
    make_top();
    
}  // end module make_case 

//----------------------------------------------------------
// make_top ();                  // uncomment for local debug if all in top level geometry is disabled
module make_top () {
  // make_top will orient and position the case case top for printing flat side down
  
  // no translations or rotations are required  
  case_top();
    
}  // end module make_top 

//----------------------------------------------------------
// make_bottom ();               // uncomment for local debug if all in top level geometry is disabled
module make_bottom () {
  // make_bottom will orient and position the case case top for printing flat side down
  
  // no translations or rotations are required  
  case_bottom();
    
}  // end module make_bottom 

//----------------------------------------------------------
// grid_vent_plate ();           // uncomment for local debug if all in top level geometry is disabled
module grid_vent_plate () {
  // grid_vent_plate forms a vent plate with a square grid pattern
    
  // fixed parameters
  VENT_SW = 2;                   // strut width - the material between vent holes
  PLATE_SX = CASE_HS_CUTOUT_SIZE+MDA;  // size plate to fill the HS cutout
  PLATE_SY = CASE_HS_CUTOUT_SIZE+MDA;  // size plate to fill the HS cutout
  PLATE_SZ = 1.6;                // 1.6 has worked well on previous projects
  echo ( str("Square vent hole size: ",hole_size,"mm"));
    
  // calculated parameters
  hole_size = (CASE_HS_CUTOUT_SIZE - 4*VENT_SW)/4; // assume four holes, three struts between and 1/2 strut each side
 
  // top level translate centers the plate at xy=0, with bottom of plate at z=0
  // comment out the top level translate for local debug as necessary
  translate ([ -PLATE_SX/2, -PLATE_SY/2, 0 ]) {        // top level translate 
      
    // the hexagon vent plate is formed by removing hexagon holes from a solid plate
    difference () {
      // form with the solid plate
      cube ([ PLATE_SX, PLATE_SY, PLATE_SZ ]);

      for ( y=[0:3] ) {      // four rows 
          for ( x=[0:3] ) {  // four columns in each row
            translate ([ (MSA+VENT_SW/2) + x*(hole_size+VENT_SW), (MSA+VENT_SW/2) + y*(hole_size+VENT_SW), -MSA ])
              rounded_plate ( size=[ hole_size, hole_size, PLATE_SZ+MDA ], radius=1.6 );
          }  // end for loop x
        }  // end for loop on y
    }  // end difference
  }  // end top level translate
    
}  // end module vent_plate

//----------------------------------------------------------
module rounded_wall ( length, thick, height ) {
  // rounded_wall forms an upright wall with round ends
  //   length: overall length of the wall, including rounded ends
  //   thick: thickness or width of the wall
  //   height: height of the wall
    
  union () {  // form the solids
    translate ([ thick/2, thick/2, 0 ])
      cylinder ( d=thick, h=height, $fn=24 );
    translate ([ thick/2, 0, 0 ])
      cube ([ length - thick, thick, height ]);
    translate ([ length - thick/2, thick/2, 0 ])
      cylinder ( d=thick, h=height, $fn=24 );
  }  // end union to form the solids
  
}  // end module rounded_wall

//----------------------------------------------------------
// fan_mount_plate ();           // uncomment for local debug if all in top level geometry is disabled
module fan_mount_plate () {
  // fan_mount_plate forms a generic fan mounting plate
  // the result is in the first quadrant; rotate and then translate the result to position it as desired

  // form the fan mount
//  rotate ([ 90, 0, 90 ])                // uncomment for local debug only to view as-used rotation
  difference() {
    // start with a solid plate 
    cube ([ (EXT_FAN_SIZE+2*CASE_FAN_FIT), 
           (EXT_FAN_SIZE+2*CASE_FAN_FIT-CASE_FAN_BRECESS-CASE_FAN_TRECESS), 
            CASE_FMOUNT_THICK ], false);
    // bore the airflow hole
    translate ([ (EXT_FAN_SIZE+2*CASE_FAN_FIT)/2, (EXT_FAN_SIZE/2+CASE_FAN_FIT-CASE_FAN_BRECESS), -MSA ]) 
      cylinder ( d=EXT_FAN_SIZE-2*EXT_FAN_WALL, h=CASE_FMOUNT_THICK+MDA, $fn=180 );
    // bore the fan mounting holes 
    for ( x=[CASE_FAN_FIT+EXT_FAN_MHO,                                      // x=boot select sw side
             CASE_FAN_FIT+EXT_FAN_SIZE-EXT_FAN_MHO] ){                      // x=i2s connector side
      for ( y=[CASE_FAN_FIT-CASE_FAN_BRECESS+EXT_FAN_MHO,                   // y=case bottom
               CASE_FAN_FIT-CASE_FAN_BRECESS+EXT_FAN_SIZE-EXT_FAN_MHO] ) {  // y=case top
        if ( CASE_FMOUNT_HOLE_TYPE == 0 ) {   // clearance hole for iso pin
          translate ([ x, y, - MSA ])
            cylinder ( d=CASE_FMOUNT_ISO_HID_CLEAR, h=CASE_FMOUNT_THICK+MDA, $fn=18 );
        }  // end if CASE_FMOUNT_HOLE_TYPE is clearance hole for iso pin
        if ( CASE_FMOUNT_HOLE_TYPE == 1 ) {   // clearance hole for case screw
          translate ([ x, y, - MSA ])
            cylinder ( d=CASE_FMOUNT_CSCREW_HID_CLEAR, h=CASE_FMOUNT_THICK+MDA, $fn=18 );
        }  // end if CASE_FMOUNT_HOLE_TYPE is clearance hole for case screw
        if ( CASE_FMOUNT_HOLE_TYPE == 2 ) {   // clearance hole for M3 screw
          translate ([ x, y, - MSA ])
            cylinder ( d=M3_HID_CLEAR, h=CASE_FMOUNT_THICK+MDA, $fn=18 );
        }  // end if CASE_FMOUNT_HOLE_TYPE is clearance hole for M3 screw
        if ( CASE_FMOUNT_HOLE_TYPE == 3 ) {   // hole for threading M3, use polyhole method
          translate ([ x, y, - MSA ])
            polyhole( d=M3_HID_THREADED, h=CASE_FMOUNT_THICK+MDA );
        }  // end if CASE_FMOUNT_HOLE_TYPE is hole for threading
      }  // end for loop on y
    }  // end for loop on x
  }  // end difference that forms a fan mounting plate
}  // end module fan_mount_plate

//----------------------------------------------------------
//mounting_tab ();               // uncomment for local debug if all in top level geometry is disabled
module mounting_tab () {
  // mounting_tab creates a pre-bored mounting tab for use at the case level
    
  // top level translate centers mounting hole at xy=0 to allows z-axis rotation at mounting hole in usage
  translate ([ -CASE_MTAB_WIDTH/2, case_mtab_mho, 0 ]) {
    difference () {
      union () {
        translate ([ 0, -case_mtab_mho, 0 ]) 
          cube ([ CASE_MTAB_WIDTH, case_mtab_mho+MSA, CASE_THICK ]);  // extend y by MSA for mesh
        translate ([ CASE_MTAB_WIDTH/2, -case_mtab_mho, 0 ])
          cylinder ( d=CASE_MTAB_WIDTH, h=CASE_THICK, $fn=36 );
      } // end union on tab solids
    
      // bore the mounting hole
      translate ([ CASE_MTAB_WIDTH/2, -case_mtab_mho, -MSA ])
        cylinder ( d=CASE_MTAB_HID, h=CASE_THICK+MDA, $fn=18 );
    
    }  // end difference
  }  // end top level translate
}  // end module mounting tab

//----------------------------------------------------------
module channel_wall ( length, thick, height, base ) {
  // forms an upright wall with round ends and a conical base
  //   length: overall length of the wall, including rounded ends
  //   thick: thickness or width of the wall
  //   height: overall height of the wall, including base
  //   base: height and additional thickness of the conical base
    
  union () {       // form the solids
    hull () {      // form conical base
      translate ([ thick/2, thick/2, 0 ])
        cylinder ( d1=thick+2*base, d2=thick, h=base, $fn=24 );
      translate ([ length-thick/2, thick/2, 0 ])
        cylinder ( d1=thick+2*base, d2=thick, h=base, $fn=24 );
    }  // end hull to form the conical base
    // add a rounded wall segment above the base
    translate ([ 0, 0, base-MSA ])
      rounded_wall ( length, thick, height-base+MSA );
  }  // end union to form the solids
}  // end module channel_wall

//----------------------------------------------------------
//batt_holder ( mtab=0 );        // uncomment for local debug if all in top level geometry is disabled
module batt_holder ( mtab ) {
  // batt_holder forms a holder for the optional rtc battery offered by Odroid
  //  mtab: 0=no mounting tabs, 1=include mounting tabs
    
  // result centers the battery disk at xy=0, with the bottom of the holder at z=0.
  // If integrated into another solid like a case, translate the flat bottom down into the surface to reduce height.
  // If mounting tabs are used, the holes are assumed to be for threading, not clearance
    
  // RTC battery holder parameters
  BHOLDER_H = 6;           // battery holder overall height
  BHOLDER_OD = 25.5;       // battery holder outer diameter
  BHOLDER_ID = 20.4;       // battery holder recess inside diameter
  BHOLDER_MTAB_WIDTH = 6;  // battery holder mounting tab width set to M3 head, just to cover mounting options
  BHOLDER_MTAB_T = 3;      // battery holder mounting tab thickness
  // calculated parameters
  bholder_bt = CASE_THICK-MSA; // battery holder bottom thickness set to match case thickness for clean mesh at case level
  bholder_mtab_mho = BHOLDER_MTAB_WIDTH/2 + 1;   // battery mounting hole offset oversized for screw head clearance
  bholder_mtab_hsp = BHOLDER_OD + 2*bholder_mtab_mho;   // battery holder mounting tab hole spacing
  if ( mtab == 1 )                  // printed battery holder accessory will include mounting tabs
    echo ( str("Battery holder mounting tab hole spacing: ",bholder_mtab_hsp,"mm"));
    
  difference () {
    // form the solids
    union () {
      // start with cylinder for the battery holder
      cylinder ( d=BHOLDER_OD, h=BHOLDER_H, $fn=180 ); 
      // add solids for mounting tabs, if specified
      if ( mtab == 1 ) {   // add mounting tabs
        // add plate
        translate ([ -bholder_mtab_hsp/2, -BHOLDER_MTAB_WIDTH/2, 0 ])
          cube ([ bholder_mtab_hsp, BHOLDER_MTAB_WIDTH, BHOLDER_MTAB_T ]);
        // add rounded ends
        translate ([ -bholder_mtab_hsp/2, 0, 0 ])
          cylinder ( d=BHOLDER_MTAB_WIDTH, h=BHOLDER_MTAB_T, $fn=36 );
        translate ([ bholder_mtab_hsp/2, 0, 0 ])
          cylinder ( d=BHOLDER_MTAB_WIDTH, h=BHOLDER_MTAB_T, $fn=36 );
      }  // end if mtab is add mounting tabs
    }  // end union to form the solids
    // carve out recess for the battery
    translate ([ 0, 0, bholder_bt ])
      cylinder ( d=BHOLDER_ID, h=BHOLDER_H-bholder_bt+MSA, $fn=180 );
    // notch out finger grip access
    translate ([ 0, 0, bholder_bt + (BHOLDER_H-bholder_bt)/2 + MSA  ])
      cube ([ 14, BHOLDER_OD+MDA, BHOLDER_H-bholder_bt+MDA ], true);   // 14 is the notch width
    // bore the holes in the mounting tabs, if specified
    if ( mtab == 1 ) {  // mounting tabs are included
      // polyhole is used to bore holes instead of cylinder since holes are assumed to be for threading
      translate ([ -bholder_mtab_hsp/2, 0, -MSA ])
        polyhole ( h=BHOLDER_MTAB_T + MDA, d=BHOLDER_MTAB_HID );
      translate ([ bholder_mtab_hsp/2, 0, -MSA ])
        polyhole ( h=BHOLDER_MTAB_T + MDA, d=BHOLDER_MTAB_HID );
    }  // end if mtab is mounting tabs are included
  }  // end difference
    
}  // end module batt_holder

//----------------------------------------------------------
// uart_holder ( mtab=1 );       // uncomment for local debug if all in top level geometry is disabled
module uart_holder( mtab ) {
  // uart_holder forms a holder for the optional micro usb uart module offered by Odroid
  //  mtab: 0=no mounting tabs, 1=include mounting tabs
  // The result is centered over x=0, with the USB connector face at y=0 and bottom of the holder at z=0.
  // If used on a solid surface like a case, recess the holder into the case to save overall height
    
  // UART holder parameters
  BODY_W = 18;
  BODY_D = 24;
  BODY_H = 9;
  BASE_T = 3;        // base thickness or height
  CAVITY_W = 11;
  CAVITY_D = 14;     
  CAVITY_H = 1;
  SLOT_W = 11;
  SLOT_D = 1;
  UHOLDER_MTAB_WIDTH = 6;  // UART holder mounting tab width set to M3 head, just to cover mounting options
  UHOLDER_MTAB_T = 3;      // UART holder mounting tab thickness
  // calculated parameters
  uholder_mtab_mho = UHOLDER_MTAB_WIDTH/2 + 1;      // battery mounting hole offset from body for screw head clearance
  uholder_mtab_hsp = BODY_W + 2*uholder_mtab_mho;   // battery holder mounting tab hole spacing

  difference () {
    union () {                 // union to form the solids
      // start with a solid block
      translate ([ -BODY_W/2, 0, 0 ])
        cube ([ BODY_W, BODY_D, BODY_H ]);
      // add mounting post round ends
      translate ([ -UHOLDER_POST_HSP/2, BODY_D-1-UHOLDER_POST_OD/2, 0 ])
        cylinder ( d=UHOLDER_POST_OD, h=BODY_H, $fn=26 );
      translate ([ UHOLDER_POST_HSP/2, BODY_D-1-UHOLDER_POST_OD/2, 0 ])
        cylinder ( d=UHOLDER_POST_OD, h=BODY_H, $fn=26 );
      // add plate between round ends to eliminate acute angles at body
      translate ([ -UHOLDER_POST_HSP/2, BODY_D-1-UHOLDER_POST_OD, 0 ])
        cube ([ UHOLDER_POST_HSP, UHOLDER_POST_OD, BODY_H ]);
      // add solids for mounting tabs, if specified
      if ( mtab == 1 ) {   // add mounting tabs
        // add plate
        translate ([ -uholder_mtab_hsp/2, 8.5-UHOLDER_MTAB_WIDTH/2, 0 ])
          cube ([ uholder_mtab_hsp, UHOLDER_MTAB_WIDTH, UHOLDER_MTAB_T ]);
        // add rounded ends
        translate ([ -uholder_mtab_hsp/2, 8.5, 0 ])
          cylinder ( d=UHOLDER_MTAB_WIDTH, h=UHOLDER_MTAB_T, $fn=36 );
        translate ([ uholder_mtab_hsp/2, 8.5, 0 ])
          cylinder ( d=UHOLDER_MTAB_WIDTH, h=UHOLDER_MTAB_T, $fn=36 );
      }  // end if mtab is add mounting tabs
    }  // end union to form the solids
        
    // hollow out the block
    translate ([ -(BODY_W-4)/2, -MSA, BASE_T ])            // leave 2mm at sides
      cube ([ BODY_W-4, BODY_D+MDA, BODY_H-BASE_T+MSA ]);
    // form slot for pins
    translate ([ -SLOT_W/2, 16, -MSA ]) 
      cube ([ SLOT_W, SLOT_D, BASE_T+MDA ]);
    // carve out component bed
    translate ([ -CAVITY_W/2, 1.5, BASE_T - CAVITY_H ])    // leave 1.5mm at front
      cube ([ CAVITY_W, CAVITY_D, CAVITY_H + MSA ]);
    // trim sidewalls
    translate ([ -BODY_W/2-MSA, -MSA, 6 ]) 
      cube ([ BODY_W+MDA, BODY_D-7+MSA, BODY_H-6+MSA ]);    // leave 7mm deep step at rear
    // bore strap mounting post holes
    translate ([ -UHOLDER_POST_HSP/2, BODY_D-1-UHOLDER_POST_OD/2, -MSA ])
      polyhole ( d=UHOLDER_POST_HID, h=BODY_H+MDA );
    translate ([ UHOLDER_POST_HSP/2, BODY_D-1-UHOLDER_POST_OD/2, -MSA ])
      polyhole ( d=UHOLDER_POST_HID, h=BODY_H+MDA );
    // bore holes in mounting tabs, if specified
    if ( mtab == 1 ) {   // include mounting tabs
      // bore the mounting tabs
      translate ([ -uholder_mtab_hsp/2, 8.5, -MSA ])
        polyhole ( d=UHOLDER_MTAB_HID, h=UHOLDER_MTAB_T+MDA );
      translate ([ uholder_mtab_hsp/2, 8.5, -MSA ])
        polyhole ( d=UHOLDER_MTAB_HID, h=UHOLDER_MTAB_T+MDA );
    }  // end if mtab is include mounting tabs
  }  // end difference
  
}  // end module uart_holder

//----------------------------------------------------------
//uart_strap ();                 // uncomment for local debug if all in top level geometry is disabled
module uart_strap () {           
  // uart_strap forms an accessory that be used to keep the uart in the uart_holder
  // the result is centered at xy=0, with base of the strap at z=0. 
    
  // UART holder strap parameters
  USTRAP_H = 2;                         // height or thickness of the block forming the UART strap
  USTRAP_RISER_HEIGHT = 1;              // height of additional ring under strap screw heads (for appearance)
  USTRAP_WT = 2;                        // wall thickness of strut between screw heads (for appearance)
  // calculated parameters
  ustrap_d = UHOLDER_POST_OD;           // UART strap depth same as UART holder post OD
  ustrap_w = UHOLDER_POST_HSP+ustrap_d; // UART strap width (long dimension)
  ustrap_hsp = UHOLDER_POST_HSP;        // hole spacing matches UART holder posts
    
  difference () {
      
    union () {   // form the solids
      // start with a basic block 
      translate ([ -ustrap_w/2, -ustrap_d/2, 0 ])
        rounded_plate ([ ustrap_w, ustrap_d, USTRAP_H ], 1 );  // corner radius set to 1mm
      // add some riser under screw heads
      translate ([ -ustrap_hsp/2, 0, USTRAP_H-MSA ]) 
        cylinder ( d=ustrap_d, h=USTRAP_RISER_HEIGHT+MSA, $fn=36 );
      translate ([ ustrap_hsp/2, 0, USTRAP_H-MSA ]) 
        cylinder ( d=ustrap_d, h=USTRAP_RISER_HEIGHT+MSA, $fn=36 );
      // add strut between risers
      translate ([ -ustrap_hsp/2, -USTRAP_WT/2, USTRAP_H-MSA ])
        cube ([ ustrap_hsp, USTRAP_WT, USTRAP_RISER_HEIGHT + MSA ]);
      }  // end union to form the solids
      
    // bore the screw holes (cylinder is used rather than polyhole since clearance hole is assumed)
    translate ([ -ustrap_hsp/2, 0, -MSA ]) 
      cylinder ( d=USTRAP_HID, h=USTRAP_H+USTRAP_RISER_HEIGHT+MDA, $fn=24 );    
    translate ([ ustrap_hsp/2, 0, -MSA ]) 
      cylinder ( d=USTRAP_HID, h=USTRAP_H+USTRAP_RISER_HEIGHT+MDA, $fn=24 );   
      
  }  // end difference 
}  // end module uart_strap

//----------------------------------------------------------
// foot ( nut_trap=1 );          // uncomment for local debug if all in top level geometry is disabled
module foot ( nut_trap ) {
  // foot forms a riser post that can be used under the case bottom.
  //  nut_trap: 0=bore for head recess, 1=M3 nut trap
  //
  // A recessed nut trap can be used with screw that passes through
  // the case assembly from the top.
  // 
  // The result is centered xy at the origin, and raises up from z=0.
  // Translate and rotate will need to be applied to place the foot as desired. 

  difference (){ 
    // start with a solid cylinder
    cylinder ( d1=foot_od1, d2=FOOT_OD2, h=FOOT_H, $fn=36 );
    // bore the through-hole for mounting
    translate ([ 0, 0, -MSA ])
      cylinder ( d=M3_HID_CLEAR, h=FOOT_H+MDA, $fn=18 );
    if ( nut_trap == 0 ) {        // no nut trap - screw head only
      translate ([ 0, 0, 2 ]) 
        cylinder ( d=M3_HEAD_OD_CLEAR, h=FOOT_H-2+MSA, $fn=24 );
    }  // end if nut_trap is no nut trap
    if ( nut_trap == 1 ) {        // include a hex nut trap
      translate ([ 0, 0, 2 ]) 
        cylinder ( d=M3_NUT_AFD_CLEAR/cos(30), h=FOOT_H-2+MSA, $fn=6 );
    }  // end if nut_trap is include a nut trap
  }  // end difference

}  // end module foot

//----------------------------------------------------------
// view_720_foot ();             // uncomment for local debug if all in top level geometry is disabled
module view_720_foot () {
  // foot forms a view of the model 720 bumper from Keystone Electronics.
  // The 720 is the narrowest-OD of the bumpers in the Keystone product line
   
  // The result is centered xy at the origin, and raises up from z=0.
  // Rotate and translate will need to be applied to place the foot as desired. 

  FOOT_H = 6.35;    // height
  FOOT_OD = 12.7;   // outside diameter
  FOOT_HID = 3.18;  // through-hole inside diameter
  FOOT_RD = 4.75;   // recess depth
  FOOT_RID = 6.35;  // recess inside diameter

  difference (){ 
    // start with a solid cylinder
    cylinder ( d=FOOT_OD, h=FOOT_H, $fn=36 );
    // bore the through-hole
    translate ([ 0, 0, -MSA ])
      cylinder ( d=FOOT_HID, h=FOOT_H+MDA, $fn=18 );
    // form the recess for the screw head
    translate ([ 0, 0, FOOT_H-FOOT_RD ]) 
      cylinder ( d=FOOT_RID, h=FOOT_RD+MSA, $fn=24 );
  }  // end difference
  
}  // end module foot

//----------------------------------------------------------
module view_fan_iso_pin () {
  // fan_iso_pin forms the rubber mounting pin provided with some fans
  // this is intended for model view only, not printing
    
  // Translation and/or rotation will typically be required to place the pin as desired.
  // The result centers xy at the origin, and raises up from z = 2 to ease rotating
  // the view against a mounting surface in the x or y axis.
    
// ERRATA: script design is from hominoid; iso pins for new Noctua 20mm thick fans seem to differ from these
    
  translate ([ 0, 0, -2 ]) {  // top level translate to place the ring flush with a mounting surface
    // flat stopper
    cylinder ( h=2, r=5, $fn=36 );
    // wide shaft
    translate ([ 0, 0, 2-MSA ]) 
      cylinder ( h=9+MSA, r=2.0625, $fn=36 );
    // ring
    translate ([ 0, 0, 4 ]) 
      cylinder ( h=1.25, r=2.625, $fn=36 );
    // tapered column
    translate ([ 0, 0, 9-MSA ]) 
      cylinder ( h=7+MSA, r1=2.625, r2=1.5, $fn=36 );
    // thin shaft
    translate ([ 0, 0, 16-MSA ]) 
      cylinder ( h=8+MSA, r=1.5, $fn=36 );
    // round end
    translate ([ 0, 0, 24 ]) 
      sphere( d=3.75, $fn=36 );
  }  // end top level translation
}  // end module fan_iso_pin

//----------------------------------------------------------
// heatsink_spacer( size=8, height=1.0, wt=1.5 );// uncomment for local debug
module heatsink_spacer( size, height, wt ) {
  // heatsink_spacer forms spacer accessories that can be used under heatsink
  //  size is width and depth of the spacer
  //  height is the z-axis thickness of the spacer
  //  wt is the wall thickness
    
  // result is centered at xy=0, with bottom of spacer at z=0
  // Note: printers won't do well with fine tuning of height value; sand or file printed spacer as needed
    
  difference () {
    translate ([ 0, 0, height/2 ]) 
      cube ([ size, size, height ], center=true );
    translate ([ 0, 0, height/2 ]) 
      cube ([ size-2*wt, size-2*wt, height+MDA ], center=true );
  }  // end difference
}  // add module heatsink_spacer

//----------------------------------------------------------

// end of file


