//-------------------------------------------------
// Shapes2d
//-------------------------------------------------

//-------------------------------------------------
// polarToXY
//	r = radius of circle
//	angle = to point in degrees
function polarToXY (r, angle) 
	= [r*cos (angle), r*sin (angle)];

//-------------------------------------------------
// Xextents
//	points = list of points
function Xextents (points)
	= let (values = [ for (i = [0:1:len(points)-1])
		points[i].x]) [min (values), max (values)];

//-------------------------------------------------
// Yextents
//	points = list of points
function Yextents (points)
	= let (values = [ for (i = [0:1:len(points)-1])
		points[i].y])  [min (values), max (values)];

//-------------------------------------------------
// drawpt2d
//	pt = [x, y]
//	dia = diameter of point
module drawpt2d (pt, dia=0.5)
{
    translate (pt)
    circle (d=dia);
}

//-------------------------------------------------
// drawpts2d
//	pts = list of points [x, y]
//	dia = diameter of point
module drawpts2d (pts, dia=0.5)
{
	assert (is_list (pts), "pts");
	
	colors = ["red", "blue", "green" ,"white",
				"cyan", "magenta", "yellow", "black"];
	numcols = len (colors);
	//echo ("numcols", numcols);
		
	numpts = len (pts);
	for (j=[0:1:numpts-1])
	{
		color (colors[j%numcols])
		drawpt2d (pts[j]);
	}
}

//----------------------------------------
// drawline2d
//	pt0 = start point
//	pt1 = end point
//	dia = diameter of line
module drawline2d (pt0, pt1, dia=0.125)
{
	//echo ("drawline2d", pt0, pt1, dia);

	dx = pt1.x-pt0.x;
	dy = pt1.y-pt0.y;
	length = sqrt (dx*dx+dy*dy);
	if (length)
	{
		anglez = atan2 (dy, dx);
		//echo ("anglez", anglez);
		
		translate ([pt0.x-dia/2, pt0.y])
		rotate (anglez)
		square([length, dia], false);
	}
}

//-------------------------------------------------
// drawlines2d
//	pts = list of points [x, y]
//	dia = diameter of line
module drawlines2d (pts, dia=0.125)
{
	assert (is_list (pts), "pts");
	//echo ("pts", pts);
	
	numseg = len (pts)-1;
	for (j=[0:1:numseg-1])
	{
		//echo ("j", j);
		drawline2d (pts [j], pts [j+1], dia);
	}
}

//-------------------------------------------------
// myPolygon
//	points = list of points [x, y]
// 	thickness
//		nonzero height for linear_extrude
//		zero linear_extrude is not done 
//	center 
//		true = center polygon about the origin
//		false = place polygon in the first quadrant
//	showlines used for debugging
module myPolygon (points, thickness=0, center=true,
					showlines=false)
{
	assert (is_list (points), "points");
	assert (is_num (thickness), "thickness");
	assert (is_bool (center), "center");
	assert (is_bool (showlines), "showlines");
	
	// get min and max of x values
	Xlimits = Xextents (points);
	xmin = Xlimits [0];
	xmax = Xlimits [1];
	
	// get min and max of y values
	Ylimits = Yextents (points);
	ymin = Ylimits [0];
	ymax = Ylimits [1];
	
	// get x and y shifts
	shift = center 
			? [-(xmax+xmin)/2, -(ymax+ymin)/2, -thickness/2] 
			: [-xmin, -ymin, 0];

	// translate to center in xy plane
	translate ([shift.x, shift.y, shift.z])
	{
		if (showlines)
			%drawlines2d (points);
		if (thickness)
		{
			linear_extrude (height=thickness)
			polygon (points);
		}
		else
		{
			polygon (points);
		}
	}
}

//-------------------------------------------------
// triangles
//-------------------------------------------------

//-------------------------------------------------
// triangle_right
//	base = length of base
//	height = height triangle
//	thickness = see myPolygon
//	center = see myPolygon
module triangle_right (base, height, 
			thickness=0, center=true)
{
	points = [[0, 0], [0, height], [base, 0]];
	myPolygon (points, thickness, center);
}

//-------------------------------------------------
// triangle_isosceles
//	base = length of base
//	height = height of triangle
//	thickness = see myPolygon
//	center = see myPolygon
module triangle_isosceles (base, height, 
			thickness=0, center=true)
{
	points = [[0, 0], [base/2, height], [base, 0]];
	myPolygon (points, thickness, center);
}

//-------------------------------------------------
// triangle_equilateral
//	base = length of base
//	thickness = see myPolygon
//	center = see myPolygon
module triangle_equilateral (base, 
			thickness=0, center=true)
{
	triangle_isosceles (base, base,
			thickness, center);
}

//-------------------------------------------------
// quadrilaterals
//-------------------------------------------------

//-------------------------------------------------
// parallelogram 
//	width = parallelogram width
//	height = parallelogram height
//	angle = angle of left side
//	thickness = see myPolygon
//	center = see myPolygon
module parallelogram  (width, height, angle, 
			thickness=0, center=true)
{
	adjx = height*sin(angle);
	pt0 = [0, 0];
	pt1 = [width, 0];
	pt2 = [width+adjx, height];
	pt3 = [adjx, height];
	points = [pt0, pt1, pt2, pt3];
	myPolygon (points, thickness, center);
}

//-------------------------------------------------
// Isosceles_Trapezoid
//	base = length of base
//	top = length of top
//	height = height Trapezoid
//	thickness = see myPolygon
//	center = see myPolygon
module isosceles_trapezoid (base, top, height,
			thickness=0, center=true)
{
	pt0 = [-base/2, 0];
	pt1 = [base/2, 0];
	pt2 = [top/2, height];
	pt3 = [-top/2, height];
	points = [pt0, pt1, pt2, pt3];
	myPolygon (points, thickness, center);
}

//-------------------------------------------------
// rhombus
//	width = rhombus width
//	height = rhombus height
//	thickness = see myPolygon
//	center = see myPolygon
module rhombus (width, height, 
			thickness=0, center=true)
{
	pt0 = [0, height/2];
	pt1 = [-width/2, 0];
	pt2 = [0, -height/2];
	pt3 = [width/2, 0];
	points = [pt0, pt1, pt2, pt3];
	myPolygon (points, thickness, center);
}

//-------------------------------------------------
// kite
//	width = kite width
//	height1 = top height
//	height2 = bottom height
//	thickness = see myPolygon
//	center = see myPolygon
module kite (width, height1, height2, 
			thickness=0, center=true)
{
	pt0 = [0, height1];
	pt1 = [-width/2, 0];
	pt2 = [0, -height2];
	pt3 = [width/2, 0];
	points = [pt0, pt1, pt2, pt3];
	myPolygon (points, thickness, center);
}

//-------------------------------------------------
// pentagons
//-------------------------------------------------

//-------------------------------------------------
// pentagon cyclic side
//	side = length of side
//	thickness = see myPolygon
//	center = see myPolygon
module pentagon_cyclic_side (side, 
			thickness=0, center=true)
{
	diagonal = side * (1 + sqrt(5)) / 2;
	pt0 = [-side/2, 0];
	pt1 = [side/2, 0];
	pt2 = [diagonal/2, side];
	pt3 = [0, diagonal];
	pt4 = [-diagonal/2, side];
	points = [pt0, pt1, pt2, pt3, pt4];
	myPolygon (points, thickness, center);
}

//-------------------------------------------------
// pentagon cyclic radius
//	radius = radius of circumscribed circle
//	thickness = see myPolygon
//	center = see myPolygon
module pentagon_cyclic_radius (radius,
			thickness=0, center=true)
{
	side = radius * 1.1756;
	pentagon_cyclic_side (side, thickness, center);
}

//-------------------------------------------------
// pentagon convex
//	base = length of base
//	side = length of side
//	height = length of height
//	thickness = see myPolygon
//	center = see myPolygon
module pentagon_convex (base, side, height,
			thickness=0, center=true)
{
	pt0 = [-base/2, 0];
	pt1 = [base/2, 0];
	pt2 = [base/2, side];
	pt3 = [0, height];
	pt4 = [-base/2, side];
	points = [pt0, pt1, pt2, pt3, pt4];
	myPolygon (points, thickness, center);
}

//-------------------------------------------------
// star side
//	side = length of star point
//	solid = center of star
//	thickness = see myPolygon
//	center = see myPolygon
module star_side (side, solid,
			thickness=0, center=true)
{
	numPoints = 5;
	angleInitial = 90;
	angleStep = 360/numPoints;
	angleEnd = 360+angleInitial;
	diagonal = side * (1 + sqrt(5)) / 2;
	
	// radius of outter circle
	radiusOD = diagonal/2;
	b = side * 0.309;
	c = 2*b;
	
	// radius of inner circle
	radiusID = b;

	ptA = polarToXY (radiusOD,
				angleInitial+1*angleStep);
	ptB = polarToXY (radiusID, 
				-90-2*angleStep);
	ptC = polarToXY (radiusOD, 
				angleInitial-3*angleStep);
	ptD = polarToXY (radiusID, -90);
	ptE = polarToXY (radiusOD,
				angleInitial+3*angleStep);
	ptF = polarToXY (radiusID,
				-90+1*angleStep);
	ptG = polarToXY (radiusOD,
				angleInitial+4*angleStep);
	ptH = polarToXY (radiusID,
				-90+2*angleStep);
	ptI = polarToXY (radiusOD,
				angleInitial);
	ptJ = polarToXY (radiusID,
				-90-1*angleStep);

	if (solid)
	{
		points = [ptA, ptJ, ptI, ptH, ptG, ptF,
					ptE, ptD, ptC, ptB];
		myPolygon (points, thickness, center);
	}
	else
	{
		points = [ptE, ptA, ptG, ptC, ptI];
		myPolygon (points, thickness, center);
	}
}

//-------------------------------------------------
// heart
//	size = size of heart
//	thickness = see myPolygon
//	center = see myPolygon
module heart2d (size, thickness=0, center=true)
{
	//-------------------------------------------------
	// heartarc generate a list of points to form an arc
	//	origin = center of circle [x, y]
	//	r = radius of circle
	//	start = angle degrees of start
	//	end = angle degrees of end
	//	numsegs = number of points
	function heartarc (origin, r, start, end, numsegs)
		= [ for (ang =[start:(end-start)/numsegs:end])
				polarToXY (r, ang)+origin];
	segs = 30;
	points2 = heartarc ([size/2, 0], size/2, 
		-45, 170, segs);
	points1 = heartarc ([-size/2, 0], size/2, 
		10, 225, segs);
	hypot = sqrt (size*size+size*size);
	points = concat (points2, points1, [[0, -hypot]]);
	myPolygon (points, thickness, center);
}
