// Creative Commons - Attribute - Share Alike

// OrganicMonkeyMotion
// organicmonkeymotion@gmail.com
// http://organicmonkeymotion.wordpress.com

//
// TWO-PART SMT COVER TAPE TAKE-UP PULLEY
// Parametric, with press-fit thin disc (really add some super glue there) and Mx holes ready for tapping
// Grub screw size sets the thickness of thicker disc.
// Shaft tolerance shared between shaft (either of 5mm or 8mm linear rod) and thin press fit disc.


/* [Main Pulley Flange with Spindle] */
tape_width = 8;               // [8: "8 mm", 12: "12 mm", 16: "16 mm"]

shaft_diameter  = 5;          // [5: "5 mm", 8: "8 mm"]

// Adjust to allow free spin, so grub screw is the main source of friction.
shaft_tolerance = 0.3;   // [0.0:0.1:0.5]

hub_diameter = shaft_diameter + 10;

flange_diameter = hub_diameter + tape_width + 16;

step_diameter = ((hub_diameter - shaft_diameter) / 2) + shaft_diameter;

// Well actually be taping that!
grub_diameter = 1.25;         // [1.25: "M1.6", 1.6: "M2", 2.05: "M2.5", 2.5: "M3"]

thick_disc_thickness = grub_diameter + 4;

// For bevelled edges.
bevel = 0.6;    // [0.3:0.1:1.0]

// Of the notch that allows access to the grub screw.
notch_angle = 40;   // [35:5:60]

// Set % of flange as notch depth.
notch_depth = 75;    // [0:10:100]

hub_r    = hub_diameter / 2;
flange_r = flange_diameter / 2;

// notch_depth: 0–100 (%)
notch_r1 = flange_r - ((flange_r - hub_r) * (notch_depth / 100));
notch_r2 = flange_r;  // outer radius for notch sector

// Well cut a slit for locking cover tape onto pulley.
slit_angle        = 135;   // [90:15:270]

/* [Secondary Flange] */
// Can of course be thick, adjust if needing to adjust spacing between pulleys.
thin_disc_thickness = 1.5;    // [1.5 : 0.1 : 10.0]

// Adjust for press fitting of secondary flange to spool step.
step_tolerance = 0.3;   // [0.0:0.1:0.5]





// -----------------------------------------------------------
// Cover tape feed slot (triangular tapered channel)
// -----------------------------------------------------------

// slot spans from inner radius to outer radius
slit_length = flange_r - hub_r;

// -------------------------------------------------------
//  Main calls
// -------------------------------------------------------

main_hub();

translate([flange_diameter + 10, 0, 0])
    thin_disc();

// debug_slot_tool();
// -------------------------------------------------------
//  Modules
// -------------------------------------------------------

module main_hub() {
    difference() {
        union() {

            thick_disc();

            translate([0, 0, thick_disc_thickness])
                notched_shaft();
        }
    }
}


module thin_disc() {
    difference() {
        cylinder(d = flange_diameter,
                 h = thin_disc_thickness,
                 $fn = 120);

        translate([0, 0, -0.1])
            cylinder(d = step_diameter + step_tolerance,
                     h = thin_disc_thickness + 0.2,
                     $fn = 80);
    }
}


module thick_disc() {

    difference() {

        //
        // CGAL-SAFE bevelED DISC
        //
        //     4___________5
        //    /            |
        //   3             |
        //   |             |
        //   2             |
        //    \            |
        //     1___________0
        
        
        profile = [

            [0, 0],           // 0

            [flange_r - bevel, 0],  // 1

            [flange_r, bevel],      // 2

            [flange_r , thick_disc_thickness - bevel], // 3

            [flange_r - bevel, thick_disc_thickness],  // 4

            [0, thick_disc_thickness]            // 5 
        ];

        
        // thick disc
        rotate_extrude($fn=180)
            polygon(profile);


        // Shaft bore
        translate([0,0,-0.1])
            cylinder(d = shaft_diameter + shaft_tolerance,
                     h = thick_disc_thickness + 0.2, $fn=120);

        // Segment for grub screw access
        arc3d(
            notch_r1,
            notch_r2 + 1,
            0,
            notch_angle,
            thick_disc_thickness
        );

        // Grub screw hole for tapping
        translate([0,0,thick_disc_thickness/2])
            rotate([0,90,notch_angle/2])
                cylinder(h = flange_diameter,
                         r = grub_diameter/2,
                         center=false, $fn=64);

        // Slit cutter
        rotate([0,0,slit_angle])
            slit_cutter();
    }
}



module shaft() {
    difference() {
        cylinder(d = hub_diameter,
                 h = tape_width + thin_disc_thickness,
                 $fn = 120);

        cylinder(d = shaft_diameter + shaft_tolerance,
                 h = tape_width + thin_disc_thickness,
                 $fn = 120);
    }
}


module notch() {
    difference() {
        cylinder(d = hub_diameter,
                 h = thin_disc_thickness,
                 $fn = 120);

        cylinder(d = step_diameter + shaft_tolerance,
                 h = thin_disc_thickness,
                 $fn = 120);
    }
}


module notched_shaft() {
    difference() {
        shaft();
        translate([0, 0, tape_width])
            notch();
    }
}


// -------------------------------------------------------
//  Arc utilities
//  from https://raw.org/snippet/circular-sector-and-arcs-with-openscad/
// -------------------------------------------------------

module arc(r1, r2, a1, a2, $fn = 128) {

    r0 = min(r1, r2);
    r  = max(r1, r2);

    a = (a1 % 360 + 360) % 360;
    b = (a2 % 360 + 360) % 360;

    d = (b - a) % 360;
    sweep = (d < 0 ? d + 360 : d);

    if (sweep == 0) {
        difference() {
            circle(r = r,  $fn = $fn);
            if (r0 > 0)
                circle(r = r0, $fn = $fn);
        }
    } else {
        k = max(3, ceil($fn * sweep / 360));

        outer = [for (i = [0:k])
            [ r  * cos(a + sweep * i / k),
              r  * sin(a + sweep * i / k) ]
        ];

        inner = [for (i = [0:k])
            [ r0 * cos(b - sweep * i / k),
              r0 * sin(b - sweep * i / k) ]
        ];

        polygon(concat(outer, inner));
    }
}


module arc3d(r1, r2, a1, a2, h, res = 128) {
    linear_extrude(height = h)
        arc(r1, r2, a1, a2, res);
}

module debug_slot_tool() {
    color("red")
    // Cover tape feed slot tool
//    translate([hub_r,0,0]) rotate([90,0,slot_angle])            slot_tool();
               slot_cutter();

}

module slit_cutter() {
    
    slit_inner_width  = 3.0;   // set distance between B and C

    // A = apex
    A    = [0, 0, -1];  // The -1 makes the peak of the cutter dip below the 
                        // surface and leave a slit for the cover tape 
                        // to slop through. 

    // B', C' = widened lip from the original triangular cutter so as to
    // add a bevel.
    // B was [0, -(slit_inner_width/2), thick_disc_thickness];
    // C was [0, (slit_inner_width/2), thick_disc_thickness];
    
    
    //  B'< B ------- C > C'  < and > = bevel
    //      |         |       | = bevel
    //      N         M
    //       \       /
    //        \     /
    //         \   /
    //          \ /
    //           A
    //
    // provile is now A,N,B',C',M and back to A
    
    NewB = [0, -(slit_inner_width/2 + bevel), thick_disc_thickness];
    NewC = [0,  (slit_inner_width/2 + bevel), thick_disc_thickness];

    // N, M = bevel points
    N = [0, -slit_inner_width/2, thick_disc_thickness - bevel];
    M = [0,  slit_inner_width/2, thick_disc_thickness - bevel];

    // Extend cutter a bit to guarantee clean subtraction
    cutter_extend = bevel * 2;

    // Build polyhedron
    polyhedron(
        points = [

            // ---- INNER FACE (x = hub_r) ----
            [hub_r, A[1],    A[2]],      // 0: A
            [hub_r, N[1],    N[2]],      // 1: N
            [hub_r, NewB[1], NewB[2]],   // 2: NewB
            [hub_r, NewC[1], NewC[2]],   // 3: NewC
            [hub_r, M[1],    M[2]],      // 4: M

            // ---- OUTER FACE (x = flange_r + ext) ----
            [flange_r + cutter_extend, A[1],    A[2]],      // 5: A'
            [flange_r + cutter_extend, N[1],    N[2]],      // 6: N'
            [flange_r + cutter_extend, NewB[1], NewB[2]],   // 7: NewB'
            [flange_r + cutter_extend, NewC[1], NewC[2]],   // 8: NewC'
            [flange_r + cutter_extend, M[1],    M[2]]       // 9: M'
        ],

        faces = [

            // Inner 5-sided face
            [0,1,2,3,4],

            // Outer 5-sided face (reverse winding)
            [5,9,8,7,6],

            // Side walls A–A'
            [0,5,6,1],

            // N–N'
            [1,6,7,2],

            // NewB – NewB'
            [2,7,8,3],

            // NewC – NewC'
            [3,8,9,4],

            // M – M'
            [4,9,5,0]
        ]
    );
}





