/*=============================================================================

  3.5" HDD

  (c) David Annett
  26 October 2025

  Screws are 6-32 UNC-2B
 
=============================================================================*/

// Imported modules

//include <metrichardware.scad>;

// Control rendering quality

$fa     = 1;
$fs     = 0.1;
Overlap = 0.1;
showsample = true;
DefaultClearance  = 0.25;  // Fit clearance

// Dimenions from SFF-8301 Rev 1.9 specification

A1  =  26.10; // Z axis, height. Thinner drives can be 42 or 17.80
A2  = 147.00; // Y axis, length max
A3  = 101.60; // X axis, width +- A11
A4  =  95.25; // X axis, bottom left screws, from right screws
A5  =   3.18; // X axis, bottom right screws, from case right
A6  =  44.45; // Y axis, bottom center screws, from rear screws
A7  =  41.28; // Y axis, bottom rear screws, from case rear
A8  =  28.50; // Y axis, side rear screws, from case rear
A9  = 101.60; // Y axis, side front screws, from rear screws
A10 =   6.35; // Z axis, side screws, from case bottom
A11 =   0.25; // Tolerance +- 
A12 =   0.50; // Tolerance +- 
A13 =  76.20; // Y axis,  bottom front screws, from rear screws
ScrewHoleDepth    = 3.56;  // Screw hold depth for maximum length screw

// 6-32 UNC-2B screws

UNCHoleDiameter = 3.505; // From https://www.machiningdoctor.com/threadinfo/?tid=12

UNCFlatDiameter = 7;    // Diameter of head
UNCFlatHeight   = 1.4;  // Height of head
UNCFlatLength   = 6.4;  // Thread only, not including head

UNCCountersinkDiameter = 5;    // Diameter of head
UNCCountersinkHeight   = 1.25;  // Height of head
UNCCountersinkLength   = 3.75;  // Thread only, not including head


// Derived dimensions

// Case body

HDD35Width  = A3;
HDD35Length = A2;
HDD35Height = A1;

// Side holes

HDD35SideZ      = A10;
HDD35SideYFront = A2-(A9+A8);
HDD35SideYRear  = A2-A8;

// Bottom holes

HDD35BottomXLeft   = A3-(A4+A5);
HDD35BottomXRight  = A3-A5;
HDD35BottomYFront  = A2-(A7+A13);
HDD35BottomYCenter = A2-(A7+A6);
HDD35BottomYRear   = A2-A7;

/*
  Sample renders
*/

if (showsample) {
  color("gray")
  HDD35Model();
  
  difference() {
    color("green")
    translate([-7, HDD35SideYFront, 0])
    cube([7, 10, 13]);

    HDD35SideScrewHoles(7, "Flat", 4, 2);
  }
  
  color("blue", 0.1)
  HDD35SideScrews(7, "Flat", 4);
  
  color("red")
  translate([-10, 0, 0])
  HDD35Screw(UNCFlatLength, "Flat"); 

  color("red")
  translate([-20, 0, 0])
  HDD35Screw(UNCCountersinkLength, "Countersink"); 
}


/*
  HDD model
*/
module HDD35Model (
  )
{
  difference() {
    cube([HDD35Width, HDD35Length, HDD35Height]);

    // Left holes

    translate([ScrewHoleDepth/2 - Overlap, HDD35SideYFront, HDD35SideZ])
    rotate([0, 90, 0])
      cylinder(h = ScrewHoleDepth + Overlap, d = UNCHoleDiameter, center = true);
  
    translate([ScrewHoleDepth/2 - Overlap, HDD35SideYRear, HDD35SideZ])
    rotate([0, 90, 0])
      cylinder(h = ScrewHoleDepth + Overlap, d = UNCHoleDiameter, center = true);
  
    // Right holes
  
    translate([HDD35Width + Overlap - ScrewHoleDepth/2, HDD35SideYFront, HDD35SideZ])
    rotate([0, 90, 0])
      cylinder(h = ScrewHoleDepth + Overlap, d = UNCHoleDiameter, center = true);

    translate([HDD35Width + Overlap - ScrewHoleDepth/2, HDD35SideYRear, HDD35SideZ])
    rotate([0, 90, 0])
      cylinder(h = ScrewHoleDepth + Overlap, d = UNCHoleDiameter, center = true);
  
    // Front holes
  
    translate([HDD35BottomXLeft, HDD35BottomYFront, ScrewHoleDepth/2 - Overlap])
      cylinder(h = ScrewHoleDepth + Overlap, d = UNCHoleDiameter, center = true);

    translate([HDD35BottomXRight, HDD35BottomYFront, ScrewHoleDepth/2 - Overlap])
      cylinder(h = ScrewHoleDepth + Overlap, d = UNCHoleDiameter, center = true);
  
    // Center holes
  
    translate([HDD35BottomXLeft, HDD35BottomYCenter, ScrewHoleDepth/2 - Overlap])
      cylinder(h = ScrewHoleDepth + Overlap, d = UNCHoleDiameter, center = true);

    translate([HDD35BottomXRight, HDD35BottomYCenter, ScrewHoleDepth/2 - Overlap])
      cylinder(h = ScrewHoleDepth + Overlap, d = UNCHoleDiameter, center = true);
  
    // Rear holes
  
    translate([HDD35BottomXLeft, HDD35BottomYRear, ScrewHoleDepth/2 - Overlap])
      cylinder(h = ScrewHoleDepth + Overlap, d = UNCHoleDiameter, center = true);

    translate([HDD35BottomXRight, HDD35BottomYRear, ScrewHoleDepth/2 - Overlap])
      cylinder(h = ScrewHoleDepth + Overlap, d = UNCHoleDiameter, center = true);
  }
}


/*
  6-32 UNC-2B screw

  Length is of thread, from bottom of head

  Zero reference point:
  - XY Center line of thread
  - Z Top of thread where it joins the head

  Head can be:
    None - Used to make a simple pass through hole
    Flat - Normal low profile flat head
    Dome - Curved head
    PC   - PC case style bolt/crosshead
    Countersink - Cone profile
  For countersink zero point is top of head.
  For others if is top of thread if Recess is 0
  or top of head if Recess is non zero.

  Clearance is added to diameter, top of head and
  end of thread for use as difference cutout.

  The Recess value extends the head above top of the
  screw if clearance is non-zero as assumed to be used
  as a cutout for a hole.

*/
module HDD35Screw(
  Length,
  Head,
  Clearance = 0,
  Recess = 0
)
{
  if (Head == "Flat") {
    // Head
    translate([0, 0, (UNCFlatHeight+Clearance+Recess)/2])
    cylinder(h = UNCFlatHeight + Clearance + Recess,
             d = UNCFlatDiameter + Clearance * 2,
             center = true);
    // Thread
    translate([0, 0, -Length/2])
    cylinder(h = Length + Clearance * 2,
             d = UNCHoleDiameter + Clearance * 2,
             center = true);
  }
  
  if (Head == "Countersink") {
    // Head
    translate([0, 0,
               (Clearance + Recess)/2 - UNCCountersinkHeight/2])
    cylinder(h = Clearance + Recess + UNCCountersinkHeight,
             d2 = UNCCountersinkDiameter + Clearance * 2,
             d1 = UNCHoleDiameter + Clearance * 2,
             center = true);
    // Thread
    translate([0, 0, (Length+UNCCountersinkHeight)/-2])
    cylinder(h = Length + Clearance * 2,
             d = UNCHoleDiameter + Clearance * 2,
             center = true);
  }
}


/*
  Side screws

  Standoff is distance from case to screw head
*/

module HDD35SideScrews(
  Length,
  Head,
  Standoff
)
{
  // Front left
  translate([-Standoff, HDD35SideYFront, HDD35SideZ])
  rotate([0, -90, 0])
  HDD35Screw(Length, Head, 0);

  // Rear left
  translate([-Standoff, HDD35SideYRear, HDD35SideZ])
  rotate([0, -90, 0])
  HDD35Screw(Length, Head, 0);

  // Front right
  translate([HDD35Width + Standoff, HDD35SideYFront, HDD35SideZ])
  rotate([0, 90, 0])
  HDD35Screw(Length, Head, 0);

  // Rear right
  translate([HDD35Width + Standoff, HDD35SideYRear, HDD35SideZ])
  rotate([0, 90, 0])
  HDD35Screw(Length, Head, 0);

}


/*
  Side screw holes

  Standoff is distance from case to screw head.
  Recess is how far below a surface the head will be.
*/

module HDD35SideScrewHoles(
  Length,
  Head,
  Standoff,
  Recess
)
{
  // Front left
  translate([-Standoff, HDD35SideYFront, HDD35SideZ])
  rotate([0, -90, 0])
  HDD35Screw(Length, Head, DefaultClearance, Recess);

  // Rear left
  translate([-Standoff, HDD35SideYRear, HDD35SideZ])
  rotate([0, -90, 0])
  HDD35Screw(Length, Head, DefaultClearance, Recess);

  // Front right
  translate([HDD35Width + Standoff, HDD35SideYFront, HDD35SideZ])
  rotate([0, 90, 0])
  HDD35Screw(Length, Head, DefaultClearance, Recess);

  // Rear right
  translate([HDD35Width + Standoff, HDD35SideYRear, HDD35SideZ])
  rotate([0, 90, 0])
  HDD35Screw(Length, Head, DefaultClearance, Recess);
}
