// Corsair 4000D GPU Anti-Sag Bracket
// Parametric OpenSCAD model

/* ===== MEASURED DIMENSIONS ===== */
base_diameter = 20;          // Main circular base
base_thickness = 5.5;        // Thickness of circular base
tab_width = 6.5;               // Width of mounting tabs

tab_thickness = 2.75;           // Height/thickness of tabs (vertical)
screw_hole = 7;              // Hole through base cylinder

arm_diameter = 9;            // Support arm cylinder diameter
arm_hole = 4;                // Hole through arm cylinder

// Extended shelf parameters
shelf_width = 26;            // Width of the rounded rectangle arm
arm_total_length = 35;       // Total arm length (same as original)

// Rendering quality
$fn = 60;

/* ===== MODULES ===== */

module mounting_base() {
    difference() {
        union() {
            // Main circular base
            cylinder(h=base_thickness, d=base_diameter);
            
            // Single rectangle crossing the bottom of base
            // Constrained to stay within the circular base
            intersection() {
                // The rectangle
                translate([-tab_width/2, -base_diameter/2, -tab_thickness])
                cube([tab_width, base_diameter, tab_thickness]);
                
                // Cylinder to constrain it
                translate([0, 0, -tab_thickness])
                cylinder(h=tab_thickness, d=base_diameter);
            }
        }
        
        // Hole through base (7mm) - only through the base, not the rectangle
        translate([0, 0, -1])
        cylinder(h=base_thickness + 2, d=screw_hole);
        
        // Larger hole through rectangle - extends slightly into base to prevent artifact
        translate([0, 0, -tab_thickness])
        cylinder(h=tab_thickness + 0.2, d=screw_hole + 3);
    }
}

module reinforcement_gusset() {
    // Tapered cylinder connecting base to arm for strength
    gusset_height = 5;  // Height of reinforcement (was 3mm, now 5mm)
    difference() {
        translate([0, 0, base_thickness])
        cylinder(h=gusset_height, d1=base_diameter, d2=arm_diameter);
        
        // Hole through gusset
        translate([0, 0, base_thickness - 1])
        cylinder(h=gusset_height + 2, d=arm_hole);
    }
}

module rounded_rectangle_arm() {
    // Creates a rounded rectangle arm - rounded on one edge, flat on the far edge
    // 25mm wide, 9mm thick, 35mm tall
    difference() {
        // Rounded rectangle using hull between cylinder and flat edge
        hull() {
            // Rounded edge (near the base)
            translate([0, 0, 0])
            cylinder(h=arm_total_length, d=arm_diameter);
            
            // Flat far edge - use a thin cube
            translate([shelf_width - 0.1, -arm_diameter/2, 0])
            cube([0.1, arm_diameter, arm_total_length]);
        }
        
        // Hole through the arm at the cylinder position
        translate([0, 0, -1])
        cylinder(h=arm_total_length + 2, d=arm_hole);
    }
}

module gpu_bracket() {
    // Large flat cylinder - the mounting base
    mounting_base();
    
    // Reinforcement gusset between base and arm
    reinforcement_gusset();
    
    // Rounded rectangle arm - extended down into gusset area for strength
    // Starts at base_thickness instead of base_thickness + 5
    translate([0, 0, base_thickness])
    rounded_rectangle_arm();
}

/* ===== BUILD ===== */
gpu_bracket();


