/********************************************************************
 File: parametric_sheave_pulley.scad
 Author: 2025 Fabio D. Bernard.
 License: This work is released with CC0 into the public domain.
   https://creativecommons.org/publicdomain/zero/1.0/
 Description:
   Parametric flanged sheave pulley with central bore.
   Designed for Thingiverse Customizer — all key dimensions
   are adjustable via parameters.
 Usage:
   - Open in OpenSCAD or Thingiverse Customizer.
   - Adjust parameters to fit belt, rope, or cable size.
   - Export STL for 3D printing.
********************************************************************/

// global resolution for all ($fn)
resolution = 20;   // [20:180] 

// Groove width (axial depth)
groove_width = 1.5;

// Sheave outer diameter
sheave_diameter = 10;

// Flange Thickness
flange_thickness = 1.5;

// // Flange edge rounding radius
flange_fillet = 0.7;

// Flange outer diameter
flange_diameter = 11;

// Central through-hole diameter
bore_diameter = 2;

/* [Hidden] */

// --- Module for sheave groove (belt channel) ---
module groove(outer_diameter=20, groove_diameter=8) {
    rotate_extrude($fn=resolution)
        translate([outer_diameter/2, 0, 0])
            circle(r=groove_diameter/2, $fn=resolution);
}

// --- Rounded flange ---
module flange_round(thickness=0.5, flange_diameter=11, fillet_radius=0.3) {
    minkowski() {
        translate([0,0,fillet_radius])
            cylinder(h=thickness-fillet_radius*2, r=flange_diameter/2-fillet_radius, $fn=resolution);
        sphere(r=fillet_radius, $fn=resolution);
    }
}

// --- Complete Pulley Assembly (sheave + flanges + bore) ---
difference() {
    union() {
        // Sheave body with groove
        difference() {
            cylinder(h=groove_width, r=sheave_diameter/2, $fn=resolution);
            translate([0,0,groove_width/2])
                groove(outer_diameter=sheave_diameter, groove_diameter=groove_width);
        }
        // Upper rounded flange
        translate([0,0,groove_width])
            flange_round(thickness=flange_thickness, flange_diameter=flange_diameter, fillet_radius=flange_fillet);
        // Lower rounded flange
        translate([0,0,-flange_thickness])
            flange_round(thickness=flange_thickness, flange_diameter=flange_diameter, fillet_radius=flange_fillet);
    }
    // Central bore (through-hole)
    translate([0,0,-flange_thickness])
        cylinder(h=groove_width+2*flange_thickness, r=bore_diameter/2, $fn=resolution);
}
