// Air Vent Register Generator
// Author: Thomas Ng
// Date: 2025-12-01
// Version: 1.0
// License: Creative Commons - Attribution - Share Alike
// 
// Generates customizable air vent register from SVG patterns with:
//  - Tapered border frame around the vent pattern
//  - Optional hollow mounting frame underneath
// - Optional support bars for structural re-inforcement
// - Optional angled fins for airflow direction control
//
//
// IMPORTANT SETUP INSTRUCTIONS:
//    This script includes 5 pre-configured SVG patterns. Select one from the dropdown below.
//    
//    To use your own SVG:
//    1. Run crop_svg.py on your SVG file to remove whitespace and get dimensions
//    2. Select "custom" from the svg_file dropdown
//    3. Enter your cropped SVG path in custom_svg_path
//    4. Enter the width and height from the cropped filename (e.g., pattern-cropped-40.1x245.6.svg)

/* [SVG Dimensions - REQUIRED] */
// Select SVG file (or choose "custom" to use your own)
svg_file = "art_deco-cropped-40.1x245.6.svg";  // [art_deco-cropped-40.1x245.6.svg, branches-cropped-117.0x241.5.svg, celtic-cropped-128.9x241.5.svg, chain-cropped-67.5x245.6.svg, floral-cropped-61.9x245.6.svg, custom]
// Custom SVG file path (only used if svg_file = "custom")
custom_svg_path = "your-pattern-cropped-40.1x245.6.svg";
// Custom SVG width (only used if svg_file = "custom")
custom_svg_width = 40.1;
// Custom SVG height (only used if svg_file = "custom")
custom_svg_height = 245.6;

// Calculated dimensions (do not edit)
svg_actual_width = svg_file == "custom" ? custom_svg_width : svg_file == "art_deco-cropped-40.1x245.6.svg" ? 40.1 : svg_file == "branches-cropped-117.0x241.5.svg" ? 117.0 : svg_file == "celtic-cropped-128.9x241.5.svg" ? 128.9 : svg_file == "chain-cropped-67.5x245.6.svg" ? 67.5 : 61.9;
svg_actual_height = svg_file == "custom" ? custom_svg_height : svg_file == "art_deco-cropped-40.1x245.6.svg" ? 245.6 : svg_file == "branches-cropped-117.0x241.5.svg" ? 241.5 : svg_file == "celtic-cropped-128.9x241.5.svg" ? 241.5 : svg_file == "chain-cropped-67.5x245.6.svg" ? 245.6 : 245.6;
svg_path = svg_file == "custom" ? custom_svg_path : svg_file;

// NOTE: Panel dimensions are for the SVG vent pattern only, NOT including the border
/* [Panel Dimensions] */
// Width of the SVG vent pattern (excluding border)
vent_width = 50;
// Height (0 = auto-scale proportionally, excluding border)
vent_height = 0;
// Height of the main panel extrusion
extrude_height = 3; // [2:0.5:8]

/* [Border Settings] */
// How far border extends from panel edge
border_width = 5;  // [0:1:50]
// Flat border width before taper starts (must be less than border_width)
flat_section = 1;  // [0.2:0.5:3]

/* [Frame Settings] */
// Enable/disable hollow frame
enable_frame = true;
// How far hollow frame sticks down
frame_height = 5;  // [1:1:50]
// Wall thickness of hollow frame
frame_thickness = 2;  // [1:0.5:10]
// Size adjustment (negative = smaller, positive = larger)
frame_offset = 0;  // [-10:0.5:10]

/* [Support Bars] */
// Enable/disable support bars
support_bars = true;
// Thickness of support bars
support_thickness = 2;  // [1:0.5:10]
// How far bars extend up from base
support_height_up = 0;  // [0:0.5:10]
// How far bars extend down from base
support_height_down = 5;  // [0:1:50]

/* [Fins] */
// Enable/disable fins (requires frame)
enable_fins = true;
// Number of fins
fin_count = 2;  // [2:1:8]
// Axis along which fins run
fin_direction = "y";  // [x:X-axis, y:Y-axis]
// Angle of fins
fin_pitch = 30;  // [0:5:40]
// Thickness of fin bars
fin_thickness = 2;  // [1:0.2:4]
// Height of fins (0 = auto-calculate from frame_height)
fin_height = 15;  // [0:1:20]

/* [Hidden] */
border_height = extrude_height;  // mm - height of border (matches panel)



//   Validation checks
assert(svg_actual_width > 0, "svg_actual_width must be greater than 0 - UPDATE THIS FROM YOUR CROPPED SVG FILENAME!");
assert(svg_actual_height > 0, "svg_actual_height must be greater than 0 - UPDATE THIS FROM YOUR CROPPED SVG FILENAME!");
assert(flat_section < border_width, "flat_section must be less than border_width");
assert(vent_width > 0, "vent_width must be greater than 0");
assert(extrude_height > 0, "extrude_height must be greater than 0");
assert(border_width >= 0, "border_width must be 0 or greater");
assert(frame_thickness > 0 || !enable_frame, "frame_thickness must be greater than 0 when frame is enabled");
assert(fin_count >= 2 && fin_count <= 8, "fin_count must be between 2 and 8");
assert(fin_pitch >= 0 && fin_pitch <= 90, "fin_pitch must be between 0 and 90 degrees");

// Get panel dimenssions - calculated from SVG dimensions  and desired size
panel_w = vent_width;
panel_h = (vent_height == 0) ? (vent_width * (svg_actual_height / svg_actual_width)) : vent_height;  // use vent_height if specified, otherwise maintain aspect ratio

// Import panel
module panel() {
  translate([-panel_w/2, -panel_h/2, 0])
    resize([vent_width, vent_height, 0], auto=true)
      import(svg_path);
}

// Panel
linear_extrude(height = extrude_height)
  panel();

// Border frame with inward rising taper
difference() {
  hull() {
    // Outer edge at 0.2mm height
    linear_extrude(height = 0.2)
      offset(r=border_width) square([panel_w, panel_h], center=true);
    // Flat section at full height
    translate([0, 0, extrude_height])
      linear_extrude(height = 0.01)
        offset(delta=flat_section) square([panel_w, panel_h], center=true);
    // Inner edge at full extrude_height
    translate([0, 0, extrude_height])
      linear_extrude(height = 0.01)
        offset(delta=-1) square([panel_w, panel_h], center=true);
  }
  // Cut out center for panel
  translate([0, 0, -0.1])
    linear_extrude(height = extrude_height + 0.2)
      offset(delta=-1) square([panel_w, panel_h], center=true);
}

// Hollow rectangle frame underneath (Frame? Protrusion? I don't know what it's called)
if (enable_frame) {
  translate([0, 0, -frame_height])
    difference() {
      linear_extrude(height = frame_height)
        offset(delta=frame_offset+frame_thickness) square([panel_w, panel_h], center=true);
      translate([0, 0, -0.1])
        linear_extrude(height = frame_height + 0.2)
          offset(delta=frame_offset) square([panel_w, panel_h], center=true);
    }
}

// Support bars (crossbeam)
if (support_bars) {
  translate([0, 0, -support_height_down])
    linear_extrude(height = support_height_down + support_height_up) {
      // Horizontal bar (across Y center)
      square([panel_w, support_thickness], center=true);
      // Vertical bar (across X center)
      square([support_thickness, panel_h], center=true);
    }
}

// Fins
if (enable_fins && enable_frame) {
  // Fin length matches frame inner dimension plus half the wall thickness
  base_length = (fin_direction == "x") ? panel_w : panel_h;
  actual_fin_length = base_length + 2*frame_offset + frame_thickness;
  fin_spacing_dim = (fin_direction == "x") ? panel_h : panel_w;
  actual_fin_height = (fin_height == 0) ? (frame_height / tan(fin_pitch)) : fin_height;
  
  for (i = [1:fin_count]) {
    position = (i / (fin_count + 1)) * fin_spacing_dim - fin_spacing_dim/2;
    fin_center_z = 1 - actual_fin_height/2;
    
    if (fin_direction == "x") {
      translate([0, position, fin_center_z])
        rotate([fin_pitch, 0, 0])
          cube([actual_fin_length, fin_thickness, actual_fin_height], center=true);
    } else {
      translate([position, 0, fin_center_z])
        rotate([0, -fin_pitch, 0])
          cube([fin_thickness, actual_fin_length, actual_fin_height], center=true);
    }
  }
}
