/*  Storage Box

   Box size is configurable - see bh, th, l and w parameters below. 
   Can specify one or two latches. Option for text label inside and 
   outside of top cover.
   
   >>> Use the OpenSCAD Customizer window for making adjustments <<<

   Each box uses M3x40mm screws for hinges and latches.
   
   V3 - Initial relaease
   V4 - Text label option to outside of top cover as well as inside
        Tighter latch
   v5 - Smoother hinge operation
        Stronger top hinge

   Scott Neumann
   December 12, 2025
   
   Copyright 2025 Scott Neumann
   scott.neumann@ieee.org
   
*/

/* Adjust these as needed */

// bottom height>=12mm
bh = 24;
// top height>=16mm
th = 16;
// length>=80mm
l = 80;
// width>=12mm
w = 40; 
// number of latches (1 or 2)  
n = 1;    

// Label for personalization
label = "";  
//label = "Lucy";  // personalized label for top
//label = "Alice";  // personalized label for top
//label = "Scott";  // personalized label for top
//label = "Scott.Neumann@IEEE.org";   // personalized label for top

// text size for label
ts=18; 
// left offset for label  
toff=10; 
// font style
fnt = "Arial:style=Bold";      

/* Don't touch these */

tt=0.8;    // text thickness for label
t = 3;     // thickness of case
t2 = t+2;  // width of lip
p = 0.4;   // hinge padding (for smoother hinge action)
p2 = 0.2;  // hinge spacing

r1 = 1.9;  // Loose M3 screw hole
r2 = 1.7;  // Tight M3 screw hole
r3 = 3.0;  // Hinge cylinder radius
r4 = 2.0;  // Hinge gap radius


module shell1(h){
      hull(){
         translate([0,0,t]) cylinder(h,t,t,$fn=40); // corners
         translate([0,l,t]) cylinder(h,t,t,$fn=40);
         translate([w,0,t]) cylinder(h,t,t,$fn=40);
         translate([w,l,t]) cylinder(h,t,t,$fn=40);
         
         translate([0,0,t]) sphere(t,$fn=40);  // bottom rounding
         translate([0,l,t]) sphere(t,$fn=40);
         translate([w,0,t]) sphere(t,$fn=40);
         translate([w,l,t]) sphere(t,$fn=40);
     }
     translate([0,0,h-2]) cylinder(2,t,t2,$fn=40);  // corners for lip
     translate([0,l,h-2]) cylinder(2,t,t2,$fn=40);
     translate([w,0,h-2]) cylinder(2,t,t2,$fn=40);
     translate([w,l,h-2]) cylinder(2,t,t2,$fn=40);
}

module shell(h){
   difference(){
      shell1(h);
      translate([0,0,t]) cube([w,l,h]); 
   }
   difference(){
      hull(){
         translate([0,0,h]) cylinder(t,t2,t2,$fn=40);
         translate([0,l,h]) cylinder(t,t2,t2,$fn=40);
         translate([w,0,h]) cylinder(t,t2,t2,$fn=40);
         translate([w,l,h]) cylinder(t,t2,t2,$fn=40);
      }
      translate([0,0,h]) cube([w,l,h]); 
   }
   translate([-t,0,h-2]) 
      rotate([270,180,0]) linear_extrude(l) polygon([[0,0],[0,2],[2,2]]);
   translate([w+t2,0,h]) 
      rotate([270,90,0]) linear_extrude(l) polygon([[0,0],[0,2],[2,2]]);
   translate([0,-t2,h]) 
      rotate([270,90,270]) linear_extrude(w) polygon([[0,0],[0,2],[2,2]]);
   translate([w,l+t2,h]) 
      rotate([270,90,90]) linear_extrude(w) polygon([[0,0],[0,2],[2,2]]);
}

module hinge(len){  // Basic hinge shape
   translate([0,0,-20])
   difference(){
      translate([0,-20,0]) linear_extrude(len) 
         polygon([[-3,10],[-3,20],[0,20],[0,23],[-3,23],[-3,35],[6,25],[6,18] ]);   
      translate([3,0,0])cylinder(len,2,2,$fn=60);
   }
}

module topHinge(len){  // Top hinge
   rotate([90,180,0])
      difference(){
         hinge(len);
         translate([-3,-8,12-p2]) cube([11,30,9]);
         translate([-3,-8,-20]) cube([11,30,8+p2]);
         translate([0,-20,-20]) cube([8,20,40]);
         translate([-4,-20,-20]) cube([8,20,40]);
      }
   rotate([90,180,0])
      difference(){
         translate([3,0,-12+p2]) cylinder(24-2*p2,r3,r3);
         translate([3,0,-20]) cylinder(len,r1,r1);
      }
   rotate([90,180,0])
         translate([-3,0,-12+p2]) cube([3,3,24-2*p2]);
}

module bottomHinge(len){  // Bottom hinge
   rotate([90,0,180])
      difference(){
         hinge(len);
         translate([0,0,-20]) cube([8,30,40]);
         translate([-3,t2,-20]) cube([9,30,40]);
         translate([-3,t,-20]) cube([9,30,40]);
         translate([0,-16,-12]) cube([8,30,24]);
         translate([0,0,-12]) cylinder(24,r4,r4);
      }
   rotate([90,0,180])
      difference(){
         translate([3,0,-20]) cylinder(len,r3,r3);
         translate([3,0,-20]) cylinder(len,r2,r2);   
         translate([0,-16,-12]) cube([8,30,24]);
      }
}

module rim(hr){
      t3 = t2/2;
      hull(){
         translate([0,0,0]) cylinder(hr,t3,t3,$fn=40);
         translate([0,l,0]) cylinder(hr,t3,t3,$fn=40);
         translate([w,0,0]) cylinder(hr,t3,t3,$fn=40);
         translate([w,l,0]) cylinder(hr,t3,t3,$fn=40);
     }
}

module top(){
   difference(){
      shell(th);
      translate([0,0,th+1.5]) rim(1.6);
      if(len(label)>0){
         translate([w/2+ts/2,l-toff,0.5]) 
            rotate([180,0,270]) 
               linear_extrude(2.8) text( label,size=ts,font=fnt);
      }
   }
   translate([-t2-p,l/4,th+t]) topHinge(40);
   translate([-t2-p,3*l/4,th+t]) topHinge(40);
   if (n==1){
      translate([w+t2,l/2,th+t-8]) rotate([0,0,180]) bottomHinge(40);
   } else {
      translate([w+t2,l/4,th+t-8]) rotate([0,0,180]) bottomHinge(40);
      translate([w+t2,3*l/4,th+t-8]) rotate([0,0,180]) bottomHinge(40);
   }
   if(len(label)>0){
      translate([w/2-ts/2,l-toff,t-tt/2]) 
         rotate([0,0,270]) 
            linear_extrude(0.8) text( label,size=ts,font=fnt);
   }

}

module bottom(){
   difference(){
      shell(bh);
      translate([-t2,l/4-12,bh+t]) rotate([270,0,0]) cylinder(24,r4,r4);
      translate([-t2,3*l/4-12,bh+t]) rotate([270,0,0]) cylinder(24,r4,r4);
   }
   difference(){
      translate([0,0,bh+t]) rim(1.4);
      translate([0,0,bh]) cube([w,l,5]); 
   }
   translate([-t2-p,l/4,bh+t]) bottomHinge(40);
   translate([-t2-p,3*l/4,bh+t]) bottomHinge(40);
   if (n==1){
      translate([w+t2,l/2,bh+t-8]) rotate([0,0,180]) bottomHinge(40);
   } else {
      translate([w+t2,l/4,bh+t-8]) rotate([0,0,180]) bottomHinge(40);
      translate([w+t2,3*l/4,bh+t-8]) rotate([0,0,180]) bottomHinge(40);
   }
}

module latch1(){
   cylinder(23,3.5,3.5);
   translate([16,0,0]) cylinder(23,3.5,3.5);
   cube([18,3,23]);
   translate([16,0,0]) rotate([0,0,15]) cube([10,3,23]);

}

module latch(){
   difference(){
      latch1();
      cylinder(23,r1,r1);
      translate([16,0,0]) cylinder(23,r1,r1);
      translate([8,-4,0]) cube([8.8,4,23]);  
   }
}

// RENDER THE DESIRED COMPONENTS - comment out lines as needed

translate([0,-15,0])latch();  // print a latch

if(n==2) translate([0,-25,0])latch();  // print a second latch if n==2

bottom();  // print the bottom of the box

translate([w+25,0,0]) top();  // print the top of the box

