/*  Configurable Storage Box

   Box size is configurable - see bh, th, l and w parameters below. 
   The total inside volume of a box is (th+bh)*l*w/1000 liters.
   
   Can specify one or two latches. Option for text label inside and 
   outside of top cover, with options for pre-text, main and post-text labels. 
   Can also have three lines of text inside the bottom.
   If a label is an empty string, it won't be rendered.
   An image can also be place on the inside and outside of the box top.
   Side strap attachment points can be added to the bottom.
   The bottom and latch can have an M3x12mm locking screw for a more secure closure.
   
   >>> Use the OpenSCAD Customizer window for making adjustments! <<<

   Each box uses M3x40mm screws for hinges and latches. In place of each 40mm s
   crew, you can alternatively use two 12-20mm screws.
   
   V3 - Initial relaease
   V4 - Text label option to outside of top cover as well as inside
        Tighter latch
   V5 - Smoother hinge operation
        Stronger top hinge
   v6 - Support for boxes with length between 40mm and 80mm (one hinge)
        Added pre-text and post-text label options
        Added vertical offset option for text labels
        Deeper labels on top of box
        Added option for three lines of text inside box bottom
   V7 - Added option for placing PNG image on box top
        Hid values not to be edited from Customizer
        Added basic checks for minimum dimensions
   V8 - Rendering options now controllable via Customizer - no need to edit file
        Added option for coin slot on top of box
        Added option for strap attachment points on bottom of case
        Made the latch slightly tighter
        Added option for non-snapping latches for more secure closing
        Added option for securing the latch with an M3 locking screw

   Scott Neumann
   December 14, 2025
   
   Copyright 2025 Scott Neumann
   scott.neumann@ieee.org
   
*/

/* Adjust these as needed - Use Customizer window*/

// bottom height>=16mm
bh = 34;
// top height>=16mm
th = 16;
// length>=40mm
l = 120;
// width>=12mm
w = 60; 
// number of latches (1 or 2)  
n = 2; // [1,2]   

// render all parts
renderAll = false;
// render latches
renderLatches = false;
// render box top
renderTop = true;
// render bottom
renderBottom = true;

// coin slot
coinSlot = false;
// strap attachment (w>=40)
strapAttach = false;
// no snap latch
noSnap = false;
// latch/bottom hole for M3 locking screw
lockingScrew = false;

// Main text label
label = "";  
// text size for label
ts=22; 
// left offset for label  
toff=40; 
// vertical offset label  
voff=0; 
// font style
fnt = "Arial:style=Bold";      

// Pre-text label2
label2 = "";  
// text size for label2
ts2=8; 
// left offset for label2  
toff2=4; 
// vertical offset label2  
voff2=0; 
// font style 2
fnt2 = "Arial:style=Normal";   

// Post-text label3
label3 = "";  
// text size for label3
ts3=8; 
// left offset for label3  
toff3=16; 
// vertical offset label3
voff3=4; 
// font style 3
fnt3 = "Arial:style=Normal";   

// Bottom text label4
label4 = "";  
// text size for label4
ts4=10; 
// left offset for label4  
toff4=10; 
// vertical offset label4  
voff4=0; 
// font style 4
fnt4 = "Arial:style=Bold";      

// Bottom label5
label5 = "";  
// text size for label5
ts5=8; 
// left offset for label5  
toff5=16; 
// vertical offset label5  
voff5=0; 
// font style 5
fnt5 = "Arial:style=Normal";   

// Bottom label6
label6 = "";  
// text size for label6
ts6=8; 
// left offset for label6  
toff6=16; 
// vertical offset label6
voff6=4; 
// font style 6
fnt6 = "Arial:style=Normal";   

// logo PNG filename
logo=".png";
// offset from left
logox = 15; 
// offset from bottom
logoy = 0;
// logo scale factor
ls = 0.31; //[::float]


/* Don't touch these */

/* [Hidden] */
tt = 0.8;  // text thickness for label
t = 3;     // thickness of case
t2 = t+2;  // width of lip
p = 0.4;   // hinge padding (for smoother hinge action)
p2 = 0.2;  // hinge spacing

r1 = 1.9;  // Loose M3 screw hole
r2 = 1.7;  // Tight M3 screw hole
r3 = 3.0;  // Hinge cylinder radius
r4 = 2.0;  // Hinge gap radius

d = 0.8;   // Depth for label extrusion
lt = 2.0;  // logo maximum thickness mm


module shell1(h){ // Basic box shape
      hull(){
         translate([0,0,t]) cylinder(h,t,t,$fn=40); // corners
         translate([0,l,t]) cylinder(h,t,t,$fn=40);
         translate([w,0,t]) cylinder(h,t,t,$fn=40);
         translate([w,l,t]) cylinder(h,t,t,$fn=40);
         
         translate([0,0,t]) sphere(t,$fn=40);  // bottom rounding
         translate([0,l,t]) sphere(t,$fn=40);
         translate([w,0,t]) sphere(t,$fn=40);
         translate([w,l,t]) sphere(t,$fn=40);
     }
     translate([0,0,h-2]) cylinder(2,t,t2,$fn=40);  // corners for lip
     translate([0,l,h-2]) cylinder(2,t,t2,$fn=40);
     translate([w,0,h-2]) cylinder(2,t,t2,$fn=40);
     translate([w,l,h-2]) cylinder(2,t,t2,$fn=40);
}

module shell(h){  // Hollowed box shape
   difference(){
      shell1(h);
      translate([0,0,t]) cube([w,l,h]); 
   }
   difference(){
      hull(){
         translate([0,0,h]) cylinder(t,t2,t2,$fn=40);
         translate([0,l,h]) cylinder(t,t2,t2,$fn=40);
         translate([w,0,h]) cylinder(t,t2,t2,$fn=40);
         translate([w,l,h]) cylinder(t,t2,t2,$fn=40);
      }
      translate([0,0,h]) cube([w,l,h]); 
   }
   translate([-t,0,h-2]) 
      rotate([270,180,0]) linear_extrude(l) polygon([[0,0],[0,2],[2,2]]);
   translate([w+t2,0,h]) 
      rotate([270,90,0]) linear_extrude(l) polygon([[0,0],[0,2],[2,2]]);
   translate([0,-t2,h]) 
      rotate([270,90,270]) linear_extrude(w) polygon([[0,0],[0,2],[2,2]]);
   translate([w,l+t2,h]) 
      rotate([270,90,90]) linear_extrude(w) polygon([[0,0],[0,2],[2,2]]);
}

module hinge(len){  // Basic hinge shape
   translate([0,0,-20])
   difference(){
      translate([0,-20,0]) linear_extrude(len) 
         polygon([[-3,10],[-3,20],[0,20],[0,23],[-3,23],[-3,35],[6,25],[6,18] ]);   
      translate([3,0,0])cylinder(len,2,2,$fn=60);
   }
}

module topHinge(len){  // Top hinge
   rotate([90,180,0])
      difference(){
         hinge(len);
         translate([-3,-8,12-p2]) cube([11,30,9]);
         translate([-3,-8,-20]) cube([11,30,8+p2]);
         translate([0,-20,-20]) cube([8,20,40]);
         translate([-4,-20,-20]) cube([8,20,40]);
      }
   rotate([90,180,0])
      difference(){
         translate([3,0,-12+p2]) cylinder(24-2*p2,r3,r3);
         translate([3,0,-20]) cylinder(len,r1,r1);
      }
   rotate([90,180,0])
         translate([-3,0,-12+p2]) cube([3,3,24-2*p2]);
}

module bottomHinge(len){  // Bottom hinge
   rotate([90,0,180])
      difference(){
         hinge(len);
         translate([0,0,-20]) cube([8,30,40]);
         translate([-3,t2,-20]) cube([9,30,40]);
         translate([-3,t,-20]) cube([9,30,40]);
         translate([0,-16,-12]) cube([8,30,24]);
         translate([0,0,-12]) cylinder(24,r4,r4);
      }
   rotate([90,0,180])
      difference(){
         translate([3,0,-20]) cylinder(len,r3,r3);
         translate([3,0,-20]) cylinder(len,r2,r2);   
         translate([0,-16,-12]) cube([8,30,24]);
      }
}

module rim(hr){ // Rim for top or bottom boxes
      t3 = t2/2;
      hull(){
         translate([0,0,0]) cylinder(hr,t3,t3,$fn=40);
         translate([0,l,0]) cylinder(hr,t3,t3,$fn=40);
         translate([w,0,0]) cylinder(hr,t3,t3,$fn=40);
         translate([w,l,0]) cylinder(hr,t3,t3,$fn=40);
     }
}

module top(){  // Top of box
   assert( th>=16 );
   assert( l>=40 );
   assert( w>=12 );
   difference(){
      shell(th);
      translate([0,0,th+1.5]) rim(1.6);
      if(len(label)>0){  // main text
         translate([w/2+ts/2-voff,l-toff,d]) 
            rotate([180,0,270]) 
               linear_extrude(2.8) text( label,size=ts,font=fnt);
      }
      if(len(label2)>0){ // pre text
         translate([w/4-voff2,l-toff2,d]) 
            rotate([180,0,270]) 
               linear_extrude(2.8) text( label2,size=ts2,font=fnt2);
      }
      if(len(label3)>0){  // post text
         translate([3*w/4+ts3/2+4-voff3,l-toff3,d]) 
            rotate([180,0,270]) 
               linear_extrude(2.8) text( label3,size=ts3,font=fnt3);
      }
      if (len(logo)>4 ){  // draw logo      
         translate([w-logoy,l-logox,-1.7]) 
            rotate([180,0,270])            
               scale([ls,ls,lt/100])surface(file=logo, invert=true); 
      }
      if(coinSlot)
         translate([w/2,l/2,0]) cube([4,35,10],center=true);
   }
   if (l<80){
      translate([-t2-p,l/2,th+t]) topHinge(40);
   } else {
      translate([-t2-p,l/4,th+t]) topHinge(40);
      translate([-t2-p,3*l/4,th+t]) topHinge(40);
   }
   if (n==1 || l<80){
      translate([w+t2,l/2,th+t-8]) rotate([0,0,180]) bottomHinge(40);
   } else {
      translate([w+t2,l/4,th+t-8]) rotate([0,0,180]) bottomHinge(40);
      translate([w+t2,3*l/4,th+t-8]) rotate([0,0,180]) bottomHinge(40);
   }
   if(len(label)>0){  // main text
      translate([w/2-ts/2+voff,l-toff,t-tt/2]) 
         rotate([0,0,270]) 
            linear_extrude(0.8) text( label,size=ts,font=fnt);
   }
   if(len(label2)>0){  // pre text
      translate([3*w/4+voff2,l-toff2,t-tt/2]) 
         rotate([0,0,270]) 
            linear_extrude(0.8) text( label2,size=ts2,font=fnt2);
   }
   if(len(label3)>0){  // post text
      translate([w/4-ts3/2-6+voff3,l-toff3,t-tt/2]) 
         rotate([0,0,270]) 
            linear_extrude(0.8) text( label3,size=ts3,font=fnt3);
   }
   if (len(logo)>4 ){  // draw logo
      translate([logoy,l-logox,t+1.8]) 
         rotate([0,0,270])            
            scale([ls,ls,lt/100])surface(file=logo, invert=true); 
   }
}

module bottom(){  // Bottom of box
   assert( bh>=16);
   assert( l>=40 );
   assert( w>=12 );
   difference(){
      shell(bh);
      translate([-t2,l/4-12,bh+t]) rotate([270,0,0]) cylinder(24,r4,r4);
      translate([-t2,3*l/4-12,bh+t]) rotate([270,0,0]) cylinder(24,r4,r4);
      if (lockingScrew){  // Hole for M3 locking screw
         if(n==1 || l<80){
            translate([w,l/2,bh-1])
               rotate([0,90,0]) cylinder(10,r2,r2);  
         } else {
            translate([w,l/4,bh-1])
               rotate([0,90,0]) cylinder(10,r2,r2);  
            translate([w,3*l/4,bh-1])
               rotate([0,90,0]) cylinder(10,r2,r2);    
         }  
      }
   }
   difference(){
      translate([0,0,bh+t]) rim(1.4);
      translate([0,0,bh]) cube([w,l,5]); 
   }
   if (l<80){
      translate([-t2-p,l/2,bh+t]) bottomHinge(40);   
   } else {
      translate([-t2-p,l/4,bh+t]) bottomHinge(40);
      translate([-t2-p,3*l/4,bh+t]) bottomHinge(40);
   }
   if (n==1 || l<80){
      translate([w+t2,l/2,bh+t-8]) rotate([0,0,180]) bottomHinge(40);
   } else {
      translate([w+t2,l/4,bh+t-8]) rotate([0,0,180]) bottomHinge(40);
      translate([w+t2,3*l/4,bh+t-8]) rotate([0,0,180]) bottomHinge(40);
   }
   if(len(label4)>0){  // first bottom line
      translate([w/4-ts4/2-voff4+ts4-4,toff4,t-tt/2])       
         rotate([0,0,90]) 
            linear_extrude(0.8) text( label4,size=ts4,font=fnt4);
   }
   if(len(label5)>0){  // middle bottom line
      translate([w/2-ts4/2-voff5+ts5,toff5,t-tt/2]) 
         rotate([0,0,90]) 
            linear_extrude(0.8) text( label5,size=ts5,font=fnt5);
   }
   if(len(label6)>0){  // last bottom line
      translate([3*w/4-voff6+ts6,toff6,t-tt/2]) 
         rotate([0,0,90]) 
            linear_extrude(0.8) text( label6,size=ts6,font=fnt6);
   }
   if(strapAttach){
      translate([w/2,0,0])
        rotate([0,0,180]) strap();
      translate([w/2,l,0])
        strap();
   }
}

module latch1(){
   cylinder(23,3.5,3.5);
   translate([16,0,0]) cylinder(23,3.5,3.5);
   cube([18,3,23]);
   translate([16,0,0]) rotate([0,0,15]) cube([10,3,23]);
}

module latch(){  // Box latch
   difference(){
      latch1();
      cylinder(23,r1,r1);
      translate([16,0,0]) cylinder(23,r1,r1);
      if (!noSnap)
         translate([8,-4,0]) cube([8.6,4,23]);  
      if (lockingScrew)
         translate([12,5,12])
            rotate([90,0,0])
               cylinder(10,r1,r1,$fn=40);         
   }
}

module strap(){  // Strap attachment bracket
   assert(w>=40);
   translate([-20,0,0])
   difference(){
      hull(){
         cube([40,9,5]);
         translate([5,8,0]) cylinder(5,5,5);
         translate([35,8,0]) cylinder(5,5,5);
      }
      translate([7,0,0]) cube([26,6,5]); 
   }
}

// RENDER THE DESIRED COMPONENTS - comment out lines as needed or use Customizer!

if (renderLatches || renderAll ){
   translate([0,-15,0])latch();  // render a latch
   if(n==2 && l>=80) translate([0,-25,0])latch();  // render a second latch if n==2
}

if (renderBottom || renderAll )
   bottom();  // render the bottom of the box

if (renderTop || renderAll)
   translate([w+25,0,0]) top();  // render the top of the box
   
