/* Customizable Simple Coat Knob
   Author: torwan - December 1st, 2025, v1.0

   Description / Hints:
   - More details at: https://www.thingiverse.com/thing:7221778
   - All dimensions in [mm].
   - To generate STL file:
     - Render the model using F6
     - Save the STL file using F7
   - $fn controls circular smoothness. Recommend: $fn >= 50.
*/


// ---------- Mounting-related parameters --------
/* [Mounting-related parameters] */

// Mounting method: choose between an offset cylindrical hole, or a shaped screw nest with countersink.
mnt_method = "just_shifted_hole"; //["just_shifted_hole", "screw_hole"]

// Diameter of the main mounting hole
mounting_hole_dia = 3.7;

// Vertical offset of the mounting hole. A positive value prevents the hole from passing entirely through the knob.
mounting_hole_off = 3.0;

// Enable recess on the mounting surface (clears wall plug protrusion)
mnt_wall_recess = false;

// Diameter of the mounting-surface recess
recess_dia = 8.5;

// Depth of the mounting-surface recess
recess_depth = 1.4;

// Diameter of the screw head for the screw-hole option
screw_head_dia = 8.0; 

// How deep the screw head sits inside the material
screw_head_drop = 1.0;

// Height of the screw head
screw_head_h = 2.0;

// Outer diameter of the screw shank (recommended: real diameter +0.1–0.2 mm)
screw_leg_dia = 4.2;

// Whether to close the screw hole with a thin sealing layer
screw_hole_closing = false;

// Thickness of the closing layer if enabled
screw_hole_closing_h = 0.28;

// ------------- Knob neck parameters ------------
/* [Knob neck parameters] */

// Diameter of the neck
knob_neck_dia = 12.0;

// Height of the straight cylindrical section of the neck
knob_neck_straight_h = 10.0;

// -------- Transition neck-head parameters ------
/* [Transition neck-head parameters] */

// Vertical height of the transition section
z_l = 5.0;

// Horizontal width of the transition section
xy_l = 2.0;

// ------------- Knob head parameters ------------
/* [Knob head parameters] */

// Diameter of the head at the base
knob_head_base_dia = 17.0;

// Total diameter of the knob head
knob_head_dia = 20.0;

// Height of the head
knob_head_h = 8.0;

// ------------ Supporting parameters ------------
/* [Supporting parameters] */
// Number of fragments used for circles (smoothness)
$fn = 100;

// ------ Helping parameters (do not modify)------

// Small correction constant for avoiding rendering artifacts
cc = 0.01;
// Scale factor for transition smoothing
smth_scale_factor = xy_l / z_l; 
donut_scale_factor = (knob_head_dia - knob_head_base_dia)/knob_head_h;

// --------------- GENERATION AREA ---------------
/* [GENERATION AREA] */

generate_knob();

// ------------------ MODULES --------------------
// (you do not need to touch anything below unless you
// know what you do :) )

module generate_knob()
// Main assembly of the knob: head, transition, neck, and mounting features
{
    difference()
    {
        union()
        {
            // Straight neck
            translate([0, 0, knob_head_h-2*cc])
                cylinder(d = knob_neck_dia, h = z_l + knob_neck_straight_h);

            // Transition part (head-neck)
            translate([0, 0, knob_head_h-cc])
                transition_part(min_dia = knob_neck_dia);

            // Head (disk)
            cylinder(d = knob_head_base_dia, h = knob_head_h);

            // Head's donut
            head_donut();
        }
        
        // Mounting hole
        if (mnt_method == "just_shifted_hole")
            translate([0, 0, mounting_hole_off])
                cylinder(d = mounting_hole_dia,
                         h = knob_head_h + z_l + knob_neck_straight_h);
        
        // Mounting hole
        if (mnt_method == "screw_hole")
            translate([0, 0, -cc])
                screw_nest(head_dia = screw_head_dia,
                           head_drop = screw_head_drop,
                           screw_head_h = screw_head_h,
                           total_length = knob_head_h
                                         +knob_neck_straight_h
                                         +z_l+2*cc,
                           screw_leg_dia = screw_leg_dia,
                           closed = screw_hole_closing,
                           cl_layer_h = screw_hole_closing_h);       

        // Recess at the mounting surface
        if (mnt_wall_recess == true)
        translate([0, 0, knob_head_h
                        +knob_neck_straight_h
                        +z_l+2*cc
                        -recess_depth])
            cylinder(d = recess_dia, h = recess_depth+cc);
    }
}
module transition_part(min_dia = 10)
// Creates a smooth filleted transition between the neck and the head
{
    rotate_extrude(angle = 360)
        translate([min_dia/2, 0, 0])
            difference()
            {
                square([xy_l, z_l]);
                    translate([xy_l, z_l, 0])
                        scale([smth_scale_factor, 1])
                            circle(d = 2*z_l);
            }
}

module head_donut()
//Generates the rounded outer contour (“donut”) on the knob head
{
    translate([0, 0, knob_head_h/2])
        rotate_extrude(angle = 360)
            translate([knob_head_base_dia/2-cc, 0, 0])
                scale([donut_scale_factor, 1])
                    circle(d = knob_head_h);
}

module screw_nest(head_dia = 6,
                  head_drop = 1,
                  screw_head_h = 3,
                  total_length = 10,
                  screw_leg_dia = 2,
                  closed = false,
                  cl_layer_h = 0.2)
// Creates a screw-ready mounting cavity with adjustable head recess, shank diameter, and optional closing layer
{
    difference()
    {
        rotate_extrude(angle = 360)
            polygon([[0, 0],
                     [head_dia/2, -cc],
                     [head_dia/2, head_drop],
                     [screw_leg_dia/2, head_drop + screw_head_h],
                     [0, head_drop + screw_head_h],
                     [screw_leg_dia/2, head_drop + screw_head_h],
                     [screw_leg_dia/2, total_length+cc],
                     [0, total_length+cc]
                    ]);
        if (closed == true)
            translate([0, 0, head_drop + screw_head_h])
                cylinder(d = screw_leg_dia + cc, h = cl_layer_h);
    }
}