/*************************************************************************************
 *
 * Parametric MasterCraft Isolated Mini Screw Driver Holder
 *
 *************************************************************************************
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHOR OR COPYRIGHT
 * HOLDER BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * IT IS NOT PERMITTED TO MODIFY THIS COMMENT BLOCK.
 *
 * (c)2025, Claude "Tryphon" Theroux, Montreal, Quebec, Canada
 * http://www.ctheroux.com/
 *
 *************************************************************************************/
 
// Items widths in mm
ItemWidths = [ 18, 18, 18, 18, 18, 18 ];
 
// Slot widths in mm
SlotWidths = [ 8, 8, 8, 8, 8, 8 ];
 
// Item clearance in mm
//ItemClearance = 5;

// Tooth thickness in mm
ToothThickness = 5;

// Tooth inner depth in mm (Excluding back holder and tips)
ToothDepth = 16;

// Tooth tip height in mm
ToothTipHeight = 12;

// French cleat holder clearance with socket holder in mm
lct_FrenchCleatHolderSocketHolderClearance = 10;

lct_Debug = true;
lct_MainParameters = true;
merged_code = false;

module __Customizer_Limit__ () {}

function lctVEC_VectorSum(p_Vector, p_UpperLimit = undef, p_Index = 0, p_CurrentSum = 0) = (len(p_Vector) == 0 ? undef : (p_UpperLimit == undef || p_UpperLimit + 1 > len(p_Vector) ? lctVEC_VectorSumInternal(p_Vector, len(p_Vector) - 1, p_Index, p_CurrentSum) : lctVEC_VectorSumInternal(p_Vector, p_UpperLimit, p_Index, p_CurrentSum)));

function lctVEC_VectorSumInternal(p_Vector, p_UpperLimitp_Index, p_Index = 0, p_CurrentSum = 0) = p_Index <= p_UpperLimitp_Index ? lctVEC_VectorSumInternal(p_Vector, p_UpperLimitp_Index, p_Index + 1, p_CurrentSum + p_Vector[p_Index]) : p_CurrentSum;

function lctVEC_VectorMaximum(p_Vector, p_UpperLimit = undef, p_Index = 0, currentMaximum = 0) = (len(p_Vector) == 0 ? undef : (p_UpperLimit == undef || p_UpperLimit + 1 > len(p_Vector) ? lctVEC_VectorMaximumInternal(p_Vector, len(p_Vector) - 1, p_Index, currentMaximum) : lctVEC_VectorMaximumInternal(p_Vector, p_UpperLimit, p_Index, currentMaximum)));

function lctVEC_VectorMaximumInternal(p_Vector, p_UpperLimitp_Index, p_Index = 0, currentMaximum = 0) = p_Index <= p_UpperLimitp_Index ? lctVEC_VectorMaximumInternal(p_Vector, p_UpperLimitp_Index, p_Index + 1, (currentMaximum > p_Vector[p_Index] ? currentMaximum : p_Vector[p_Index] )) : currentMaximum;

/*************************************************************************************
 *
 * Library inner parameters
 *
 *************************************************************************************/

// Skadis hole length in mm
lctSFC_SkadisHoleLength = 15;

// Skadis hole radius in mm
lctSFC_SkadisHoleRadius = 2.5;

// Skadis thickness in mm
lctSFC_SkadisThickness = 5;

// Skadis distance between two vertical holes in mm
lctSFC_SkadisVerticalHoleDistance = 40;

// Back Holder thickness in mm
lctSFC_BackHolderThickness = 4;

// Holder link height in mm
lctSFC_HolderLinkHeight = 6;

// French cheat bar width in mm
lctSFC_FrontHolderWidth = 10;

// French cleat depth in mm
lctSFC_FrontHolderThickness = 4;

// General clearance in mm
lctSFC_Clearance = 0.5;

/*************************************************************************************
 *
 * Nothing shall be modified below this line
 *
 *************************************************************************************/
 
$fn = 60;

// Back holder width in mm
lctSFC_BackHolderWidth = lctSFC_SkadisHoleRadius * 2 - lctSFC_Clearance;

// Back holder core length in mm
lctSFC_BackHolderCoreLength = lctSFC_SkadisHoleLength - 2 * lctSFC_SkadisHoleRadius;

// French cheat bar length in mm
lctSFC_FrontHolderLength = lctSFC_BackHolderCoreLength + lctSFC_SkadisHoleRadius - lctSFC_Clearance / 2;

module lctSFC_Debug(p_EnableDebug, p_HolderNumber, p_FrontHolderLength, p_FrontHolderWidth, p_FrontHolderThickness) {
    if( p_EnableDebug ) {
        echo("lctSFC_SkadisHoleLength:", lctSFC_SkadisHoleLength);
        echo("lctSFC_SkadisHoleRadius:", lctSFC_SkadisHoleRadius);
        echo("lctSFC_SkadisThickness:", lctSFC_SkadisThickness);
        echo("lctSFC_SkadisVerticalHoleDistance:", lctSFC_SkadisVerticalHoleDistance);
        echo("p_HolderNumber:", p_HolderNumber);
        echo("p_FrontHolderLength:", p_FrontHolderLength);
        echo("p_FrontHolderWidth:", p_FrontHolderWidth);
        echo("p_FrontHolderThickness:", p_FrontHolderThickness);
        echo("lctSFC_BackHolderThickness:", lctSFC_BackHolderThickness);
        echo("lctSFC_BackHolderWidth:", lctSFC_BackHolderWidth);
        echo("lctSFC_BackHolderCoreLength:", lctSFC_BackHolderCoreLength);
        echo("lctSFC_HolderLinkHeight:", lctSFC_HolderLinkHeight);
        echo("lctSFC_Clearance:", lctSFC_Clearance);
    }
}

module lctSFC_BackHolder() {
    hull() {
        translate([ -lctSFC_BackHolderCoreLength / 2, 0, 0 ]) cylinder(d = lctSFC_BackHolderWidth, h = lctSFC_BackHolderThickness);
        translate([ lctSFC_BackHolderCoreLength / 2, 0, 0 ]) cylinder(d = lctSFC_BackHolderWidth, h = lctSFC_BackHolderThickness);
    } 
}

module lctSFC_HolderLink() {
    difference() {
        cylinder( d = lctSFC_BackHolderWidth, h = lctSFC_SkadisThickness + lctSFC_Clearance);
        translate([ 0, -lctSFC_BackHolderWidth / 2, 0 ]) cube([ lctSFC_BackHolderWidth / 2, lctSFC_BackHolderWidth, lctSFC_SkadisThickness + lctSFC_Clearance ]);
    }
    translate([ 0, -lctSFC_BackHolderWidth / 2, 0 ]) cube( [ lctSFC_HolderLinkHeight - lctSFC_BackHolderWidth / 2, lctSFC_BackHolderWidth, lctSFC_SkadisThickness + lctSFC_Clearance ] );
}

module lctSFC_FrontHolder( p_FrontHolderLength, p_FrontHolderWidth, p_FrontHolderThickness)
{
    translate([ 0, p_FrontHolderWidth / 2, 0 ]) rotate([ 90, 0, 0 ])
            linear_extrude(p_FrontHolderWidth) polygon([ 
                                                    [ 0, 0 ], 
                                                    [ p_FrontHolderLength, 0 ], 
                                                    [ p_FrontHolderLength + p_FrontHolderThickness, p_FrontHolderThickness ], 
                                                    [ 0, p_FrontHolderThickness ]
                                                       ]);    
}

module lctSFC_FrenchCleatHolder( p_HolderNumber = 1, p_FrontHolderLength = lctSFC_FrontHolderLength, 
                                p_FrontHolderWidth = lctSFC_FrontHolderWidth, 
                                p_FrontHolderThickness = lctSFC_FrontHolderThickness, 
                                p_EnableDebug = false)
{
    lctSFC_Debug(p_EnableDebug, p_HolderNumber, p_FrontHolderLength, p_FrontHolderWidth, p_FrontHolderThickness);
    translate([ 0, 0, lctSFC_BackHolderWidth / 2 ]) rotate([ 90, 0, 180 ]) union() {

        for( n = [ 0 : p_HolderNumber - 1 ] ) {
            translate([ -lctSFC_SkadisVerticalHoleDistance * n, 0, 0 ]) lctSFC_BackHolder();
            translate([ lctSFC_SkadisHoleLength / 2 - lctSFC_Clearance / 2 - lctSFC_HolderLinkHeight - lctSFC_SkadisVerticalHoleDistance * n, 0, lctSFC_BackHolderThickness ]) lctSFC_HolderLink();
        }

        translate([ -p_FrontHolderLength -  + lctSFC_SkadisVerticalHoleDistance * (p_HolderNumber - 1) + lctSFC_BackHolderCoreLength / 2, p_FrontHolderWidth / 2 - lctSFC_BackHolderWidth / 2, lctSFC_BackHolderThickness + lctSFC_SkadisThickness + lctSFC_Clearance ]) lctSFC_FrontHolder(p_FrontHolderLength + lctSFC_SkadisVerticalHoleDistance * (p_HolderNumber - 1), p_FrontHolderWidth, p_FrontHolderThickness);    
    }
}

function lctSFC_FrenchCleatHolderKnockoutLength(p_HolderNumber = 1, 
                                            p_FrontHolderLength = lctSFC_FrontHolderLength, 
                                            p_FrontHolderWidth = lctSFC_FrontHolderWidth, 
                                            p_FrontHolderThickness = lctSFC_FrontHolderThickness) = p_FrontHolderLength + lctSFC_SkadisVerticalHoleDistance * (p_HolderNumber - 1) + lctSFC_Clearance / 2;

function lctSFC_FrenchCleatHolderKnockoutThickness(p_FrontHolderThickness = lctSFC_FrontHolderThickness) = p_FrontHolderThickness + lctSFC_Clearance / 2;

function lctSFC_FrenchCleatHolderKnockoutWidth(p_FrontHolderWidth = lctSFC_FrontHolderWidth) = p_FrontHolderWidth + lctSFC_Clearance / 2;

function lctFrenchCleatHolderOffset(p_InnerSkadisHoles = 0, 
                                    p_FrontHolderWidth = lctSFC_FrontHolderWidth) = (lctSFC_SkadisVerticalHoleDistance * p_InnerSkadisHoles) - p_FrontHolderWidth + lctSFC_BackHolderWidth;

module lctSFC_FrenchCleatHolderKnockout(p_HolderNumber = 1, 
                                p_FrontHolderLength = lctSFC_FrontHolderLength, 
                                p_FrontHolderWidth = lctSFC_FrontHolderWidth, 
                                p_FrontHolderThickness = lctSFC_FrontHolderThickness, 
                                p_EnableDebug = false)
{
    lctSFC_Debug(p_EnableDebug, p_HolderNumber, p_FrontHolderLength, p_FrontHolderWidth, p_FrontHolderThickness);
    
    lctSFC_FrontHolder(p_FrontHolderLength + lctSFC_SkadisVerticalHoleDistance * (p_HolderNumber - 1) + lctSFC_Clearance / 2, p_FrontHolderWidth + lctSFC_Clearance / 2, p_FrontHolderThickness + lctSFC_Clearance / 2);

}

lctCSH_Debug = false;
lctCSH_Test = false;

module __Customizer_Limit__ () {}

/*************************************************************************************
 *
 * Library inner parameters
 *
 *************************************************************************************/

// Item clearance in mm
lctCSH_ItemClearance = 5;

// French cleat holder height clearance in mm
lctCSH_FrenchCleatHeightClearance = 5;

// French cleat holder thickness clearance in mm
lctCSH_FrenchCleatHolderThicknessClearance = 5;

//lctCSH_BaseInnerDepth = p_ToothDepth;

lctCSH_BaseHeight = lctSFC_FrenchCleatHolderKnockoutLength() + lctCSH_FrenchCleatHeightClearance;

/*************************************************************************************
 *
 * Nothing shall be modified below this line
 *
 *************************************************************************************/
 
function getBaseWidth(p_ItemWidths, p_SlotWidths) = lctVEC_VectorSum(p_ItemWidths) + (len(p_ItemWidths) + 1) * lctCSH_ItemClearance;

function getSkadisHoles(p_ItemWidths, p_SlotWidths) = floor((getBaseWidth(p_ItemWidths, p_SlotWidths) - lctSFC_FrenchCleatHolderKnockoutWidth()) / lctSFC_SkadisVerticalHoleDistance);

module lct_DebugCombStyleHolder(p_EnableDebug, lctCSH_BaseWidth, lctCSH_BaseInnerDepth) {
    if( p_EnableDebug ) {
        echo("lctCSH_BaseWidth:", lctCSH_BaseWidth);
        echo("lctCSH_BaseInnerDepth:", lctCSH_BaseInnerDepth);
        echo("lctCSH_BaseHeight:", lctCSH_BaseHeight);
    }
}

module lctCSH_Base(p_BaseWidth, p_ToothDepth, p_ToothThickness, p_ToothTipHeight) {
    cube( [ BackHolderThickness, p_BaseWidth, lctCSH_BaseHeight ] );
    translate( [ BackHolderThickness, 0, 0 ] ) cube( [ p_ToothDepth, p_BaseWidth, p_ToothThickness ] );
    translate( [ BackHolderThickness + p_ToothDepth, 0, 0 ] ) cube( [ p_ToothThickness, p_BaseWidth, p_ToothThickness + p_ToothTipHeight ] );    
}

module lctCSH_Slots(p_ItemWidths, p_SlotWidths, p_ToothDepth, p_ToothThickness, p_ToothTipHeight, p_Offset = 0, p_Index = 0) {
    
    translate( [ BackHolderThickness, p_Offset, 0 ] ) cube( [  p_ToothDepth + p_ToothThickness, p_SlotWidths[p_Index], p_ToothThickness + p_ToothTipHeight ] );
    
    if( p_Index < len(p_SlotWidths) - 1 ) {
        nextOffset = (p_ItemWidths[p_Index] + p_ItemWidths[p_Index + 1]) / 2 + lctCSH_ItemClearance;
        lctCSH_Slots(p_ItemWidths, p_SlotWidths, p_ToothDepth, p_ToothThickness, p_ToothTipHeight, p_Offset + nextOffset, p_Index + 1);
    }
}

module lctCSH_Generate(p_ItemWidths, p_SlotWidths, p_ToothDepth, p_ToothThickness, p_ToothTipHeight) {
    
    lctCSH_BaseWidth = getBaseWidth(p_ItemWidths, p_SlotWidths);

    initialOffset = lctCSH_ItemClearance + p_ItemWidths[0] / 2 - p_SlotWidths[0] / 2;
    
    difference() {
        lctCSH_Base(lctCSH_BaseWidth, p_ToothDepth, p_ToothThickness, p_ToothTipHeight);
        lctCSH_Slots(p_ItemWidths, p_SlotWidths, p_ToothDepth, p_ToothThickness, p_ToothTipHeight, initialOffset );
    }

    lct_DebugCombStyleHolder(lctCSH_Debug, lctCSH_BaseWidth, p_ToothDepth);
}


// Base thickness in mm
BackHolderThickness = lctSFC_FrenchCleatHolderKnockoutThickness() + lctCSH_FrenchCleatHeightClearance;

innerSkadisHoles = floor((getBaseWidth(ItemWidths, SlotWidths) - lctSFC_FrenchCleatHolderKnockoutWidth()) / lctSFC_SkadisVerticalHoleDistance);

difference() {
    lctCSH_Generate(ItemWidths, SlotWidths, ToothDepth, ToothThickness, ToothTipHeight);
    
    initialOffset = (getBaseWidth(ItemWidths, SlotWidths) - (innerSkadisHoles - 1) * lctSFC_SkadisVerticalHoleDistance) / 2 - (lctSFC_FrontHolderWidth / 2 - lctSFC_BackHolderWidth / 2 );
    
    translate( [ 0, initialOffset, 0 ] ) for( i = [ 0 : 1 : innerSkadisHoles - 1  ] ) {
        translate( [ 0, lctSFC_SkadisVerticalHoleDistance * i, 0] ) rotate( [ 0, -90, 180 ] ) {
        lctSFC_FrenchCleatHolderKnockout();
        }
    }
}

for( i = [ 0 : 1 : innerSkadisHoles - 1 ] ) {
    translate( [ -lctSFC_BackHolderCoreLength - 10, i * ( lctSFC_FrontHolderThickness + lctSFC_SkadisThickness + lctSFC_BackHolderThickness + 10), 0 ] ) lctSFC_FrenchCleatHolder();
}
