#define STRIP_PIN 2      // пин ленты
#define NUMLEDS 156      // кол-во светодиодов

#define COLOR_DEBTH 3
#include <microLED.h>   // подключаем библу
microLED<NUMLEDS, STRIP_PIN, MLED_NO_CLOCK, LED_WS2818, ORDER_GRB, CLI_AVER> strip;

// Вспомогательные функции для работы с геометрией ёлки
// Возвращает примерную высоту диода (0 = основание, 255 = вершина)
byte getHeight(int led) {
  // Приблизительное распределение: внизу больше диодов, вверху меньше
  // Используем нелинейное распределение для конической формы
  return (byte)((led * 255) / NUMLEDS);
}

// Проверяет, находится ли диод в нижней части (первые 40%)
bool isBottom(int led) {
  return led < (NUMLEDS * 40 / 100);
}

// Проверяет, находится ли диод в верхней части (последние 30%)
bool isTop(int led) {
  return led >= (NUMLEDS * 70 / 100);
}

void setup() {
  strip.clear();
  strip.setBrightness(150);
}

int effectIndex = 1;

void loop() {
  // Автоматическое переключение эффектов каждые 30 секунд
  static unsigned long effectStartTime = 0;
  
  // Инициализируем время начала эффекта при первом запуске
  if (effectStartTime == 0) {
    effectStartTime = millis();
  }
  
  // Проверяем, прошло ли 30 секунд (30000 мс)
  if (millis() - effectStartTime >= 30000) {
    // Переключаем на следующий эффект
    effectIndex++;
    if (effectIndex > 8) {
      effectIndex = 1;  // Возвращаемся к первому эффекту
    }
    effectStartTime = millis();  // Сбрасываем таймер
  }

  switch (effectIndex) {
    case 1:
      rainbow(); // Радуга
      break;
    case 2:
      colorBlink(); // Классика
      break;  
    case 3:
      snow(); // Снег
      break;
    case 4:
      sparkle(); // Сияние
      break;
    case 5:
      wave(); // Волны
      break;
    case 6:
      aurora(); // Заря
      break;
    case 7:
      chase(); // Бегунок
      break;
    case 8:
      comet(); // Гирлянды на елке
      break;        
    default:
      effectIndex = 1;
      break;
  }

}

// Эффект 1: Радуга
void rainbow() {
  static byte counter = 0;
  for (int i = 0; i < NUMLEDS; i++) {
    strip.set(i, mWheel8(counter + i * 255 / NUMLEDS));   // counter смещает цвет
  }
  counter += 3;   // counter имеет тип byte и при достижении 255 сбросится в 0
  strip.show();   // вывод
  delay(50);      
}

// Эффект 2: Классика
void colorBlink() {
  for (int i = 0; i < NUMLEDS; i++) {
    strip.set(random(0, NUMLEDS), mWheel(random(1,1530), 150));
  }
  strip.show();   // вывод
  delay(300);    
}

// Эффект 3: Снег
void snow(){
  static byte snowBrightness[NUMLEDS] = {0};  // массив яркости для каждого диода
  static byte snowFadeSpeed[NUMLEDS] = {0};   // скорость затухания для каждой снежинки
  static unsigned long lastUpdate = 0;
  static unsigned long lastSpawn = 0;
  
  unsigned long currentTime = millis();
  
  // Периодически добавляем новые снежинки (чаще вверху, реже внизу)
  if (currentTime - lastSpawn > 50) {
    lastSpawn = currentTime;
    
    // Добавляем больше снежинок для заполнения ёлки
    // Снежинки чаще появляются в верхней части (как будто падают с неба)
    for (int i = 0; i < 12; i++) {
      int led;
      // 60% шанс появиться в верхней половине, 40% в нижней
      if (random(0, 100) < 60) {
        led = random(NUMLEDS / 2, NUMLEDS);  // верхняя половина
      } else {
        led = random(0, NUMLEDS);  // вся ёлка
      }
      
      if (snowBrightness[led] == 0) {  // только если диод не активен
        snowBrightness[led] = random(180, 255);  // случайная начальная яркость
        snowFadeSpeed[led] = random(2, 5);  // случайная скорость затухания (2-4)
        
        // Вариации цвета: от белого до голубоватого
        byte colorVar = random(0, 3);
        if (colorVar == 0) {
          strip.set(led, mRGB(255, 255, 255));  // чисто белый
        } else if (colorVar == 1) {
          strip.set(led, mRGB(240, 250, 255));  // слегка голубоватый
        } else {
          strip.set(led, mRGB(220, 240, 255));  // более голубоватый
        }
      }
    }
  }
  
  // Плавное обновление яркости всех активных снежинок
  if (currentTime - lastUpdate > 30) {
    lastUpdate = currentTime;
    
    for (int i = 0; i < NUMLEDS; i++) {
      if (snowBrightness[i] > 0) {
        // Плавно уменьшаем яркость с индивидуальной скоростью
        byte fadeAmount = snowFadeSpeed[i];
        if (snowBrightness[i] > fadeAmount) {
          snowBrightness[i] -= fadeAmount;
        } else {
          snowBrightness[i] = 0;
        }
        
        if (snowBrightness[i] > 0) {
          // Применяем затемнение с текущей яркостью
          strip.fade(i, snowBrightness[i]);
        } else {
          // Полностью гасим диод
          strip.set(i, mRGB(0, 0, 0));
          snowFadeSpeed[i] = 0;
        }
      }
    }
    
    strip.show();
  }
  
  delay(20);  // небольшая задержка для плавности
}

// Эффект 4: Сияние
void sparkle() {
  static unsigned long lastSparkle = 0;
  static unsigned long lastUpdate = 0;
  
  unsigned long currentTime = millis();
  
  // Периодически добавляем новые искры
  // Звёзды чаще появляются вверху 
  if (currentTime - lastSparkle > 80) {
    lastSparkle = currentTime;
    
    // Добавляем несколько новых искр
    for (int i = 0; i < 6; i++) {
      int led;
      // 70% шанс появиться в верхней половине 
      if (random(0, 100) < 70) {
        led = random(NUMLEDS / 2, NUMLEDS);
      } else {
        led = random(0, NUMLEDS);
      }
      
      // Случайный цвет из радуги, но чаще белый/жёлтый (как звёзды)
      int colorChoice = random(0, 100);
      if (colorChoice < 40) {
        // Белый/жёлтый (как настоящие звёзды)
        strip.set(led, mWheel(random(20, 60), random(220, 255)));
      } else {
        // Цветные звёзды
        strip.set(led, mWheel(random(0, 1530), random(200, 255)));
      }
    }
  }
  
  // Плавное затухание всех диодов
  if (currentTime - lastUpdate > 40) {
    lastUpdate = currentTime;
    
    for (int i = 0; i < NUMLEDS; i++) {
      mData currentColor = strip.get(i);
      if (getR(currentColor) > 0 || getG(currentColor) > 0 || getB(currentColor) > 0) {
        // Плавно затухаем (вверху медленнее - звёзды держатся дольше)
        byte fadeAmount = isTop(i) ? 12 : 15;
        strip.fade(i, fadeAmount);
      }
    }
    
    strip.show();
  }
  
  delay(30);
}

// Эффект 5: Волны
void wave() {
  static int wavePos[3] = {-50, -100, -150};  // Несколько волн с разным стартом
  static byte hue = 0;
  
  // Плавно затухаем все диоды (но не полностью, чтобы был фон)
  for (int i = 0; i < NUMLEDS; i++) {
    mData currentColor = strip.get(i);
    if (getR(currentColor) > 0 || getG(currentColor) > 0 || getB(currentColor) > 0) {
      strip.fade(i, 20);  // Медленнее затухание для фоновой подсветки
    }
  }
  
  // Создаём несколько волн одновременно для непрерывной подсветки
  int waveLength = NUMLEDS / 2;  // Увеличиваем длину волны
  int numWaves = 3;
  
  for (int w = 0; w < numWaves; w++) {
    int currentWavePos = wavePos[w];
    
    // Рисуем волну
    for (int i = 0; i < NUMLEDS; i++) {
      int dist = abs(i - currentWavePos);
      if (dist < waveLength) {
        // Вычисляем яркость в зависимости от расстояния от центра волны
        float fadeFactor = 1.0 - ((float)dist / waveLength);
        fadeFactor = fadeFactor * fadeFactor;  // Квадратичное затухание для плавности
        byte brightness = (byte)(fadeFactor * 200 + 30);  // Нежные цвета
        
        // Создаём градиент цвета, который меняется по высоте
        byte height = getHeight(i);
        int heightOffset = height * 4;  // Растянутый градиент
        // Используем пастельные цвета: начинаем с голубого диапазона
        int colorHue = (680 + hue + heightOffset + w * 100) % 1530;
        // Ограничиваем диапазон нежными цветами
        if (colorHue < 510) {
          colorHue = 680 + (510 - colorHue) / 3;
        } else if (colorHue > 1360) {
          colorHue = 680 + (colorHue - 1360) / 3;
        }
        
        // Смешиваем с уже существующим цветом для плавности
        mData existingColor = strip.get(i);
        mData newColor = mWheel(colorHue, brightness);
        // Если уже есть цвет, берём максимум яркости
        if (getR(existingColor) > 0 || getG(existingColor) > 0 || getB(existingColor) > 0) {
          byte existingBrightness = max(max(getR(existingColor), getG(existingColor)), getB(existingColor));
          brightness = max(brightness, existingBrightness);
          newColor = mWheel(colorHue, brightness);
        }
        strip.set(i, newColor);
      }
    }
    
    // Двигаем волну
    wavePos[w]++;
    
    // Когда волна уходит, запускаем новую снизу
    if (currentWavePos > NUMLEDS + waveLength) {
      wavePos[w] = -waveLength - random(0, 20);  // Небольшая случайная задержка
    }
  }
  
  // Медленно меняем общий оттенок
  static unsigned long lastHueChange = 0;
  if (millis() - lastHueChange > 2000) {
    lastHueChange = millis();
    hue = (hue + 20) % 1530;
  }
  
  strip.show();
  delay(45);
}

// Эффект 6: Заря
void chase() {
  static int chasePos[3] = {0, 0, 0};
  // Используем нежные цвета: голубой, фиолетовый, розовый (избегаем красного)
  static int hue[3] = {680, 1020, 1190};  // Голубой, синий, фиолетовый
  static int chaseSpeed[3] = {1, 1, 1};
  
  // Плавно затухаем все диоды
  for (int i = 0; i < NUMLEDS; i++) {
    mData currentColor = strip.get(i);
    if (getR(currentColor) > 0 || getG(currentColor) > 0 || getB(currentColor) > 0) {
      strip.fade(i, 20);
    }
  }
  
  // Создаём несколько бегущих огоньков, движущихся снизу вверх
  for (int c = 0; c < 3; c++) {
    int pos = chasePos[c];
    
    if (pos >= 0 && pos < NUMLEDS) {
      // Получаем цвет из нежной палитры
      int currentHue = hue[c];
      // Ограничиваем диапазон нежными цветами (избегаем красного 0-510)
      if (currentHue < 510) {
        // Если попали в красный/жёлтый диапазон, переходим в голубой
        currentHue = 680 + (510 - currentHue) / 2;
      } else if (currentHue > 1360) {
        // Если вышли за розовый, возвращаемся в голубой
        currentHue = 680 + (currentHue - 1360) / 2;
      }
      
      // Основной огонёк с уменьшенной яркостью для мягкости
      strip.set(pos, mWheel(currentHue, 220));
      
      // Хвост огонька (тянется вниз)
      int tailLength = isBottom(pos) ? 8 : 5;  // Внизу хвост длиннее
      for (int i = 1; i <= tailLength; i++) {
        int tailPos = pos - i;
        if (tailPos >= 0) {
          byte brightness = 220 - (i * 220 / tailLength);  // Мягче яркость
          if (brightness > 0) {
            strip.set(tailPos, mWheel(currentHue, brightness));
          }
        }
      }
    }
    
    // Двигаем огонёк вверх
    chasePos[c] += chaseSpeed[c];
    
    // Когда огонёк достигает верха, запускаем новый снизу
    if (chasePos[c] >= NUMLEDS) {
      chasePos[c] = random(0, NUMLEDS / 4);  // Старт в нижней четверти
      // Меняем цвет, но остаёмся в нежной палитре (голубой-синий-фиолетовый-розовый)
      int newHueOffset = random(200, 400);
      hue[c] = (hue[c] + newHueOffset) % 1530;
      // Проверяем, не попали ли в красный диапазон
      if (hue[c] < 510) {
        hue[c] = 680 + (510 - hue[c]) / 2;  // Переходим в голубой
      } else if (hue[c] > 1360) {
        hue[c] = 680 + (hue[c] - 1360) / 2;  // Возвращаемся в голубой
      }
      chaseSpeed[c] = random(1, 3);
    }
  }
  
  strip.show();
  delay(40);
}

// Эффект 7: Бегунок
void comet() {
  static int cometPos[3] = {0, 0, 0};
  static byte cometHue[3] = {0, 510, 1020};
  static int cometSpeed[3] = {1, 2, 1};
  static unsigned long lastUpdate = 0;
  
  unsigned long currentTime = millis();
  
  if (currentTime - lastUpdate > 60) {
    lastUpdate = currentTime;
    
    // Сначала заливаем всю ёлку зелёным цветом (фон ёлки)
    // Используем неяркий зелёный, чтобы кометы были хорошо видны
    for (int i = 0; i < NUMLEDS; i++) {
      // Зелёный цвет в HSV примерно 510 (из диапазона 0-1530)
      // Делаем зелёный неярким для фона
      strip.set(i, mWheel(510, 50));  // Зелёный, яркость 50
    }
    
    // Отслеживаем следы комет для плавного затухания
    static byte cometTrail[NUMLEDS] = {0};
    static byte cometTrailHue[NUMLEDS] = {0};
    
    // Плавно затухаем следы комет
    for (int i = 0; i < NUMLEDS; i++) {
      if (cometTrail[i] > 0) {
        cometTrail[i] = max(0, cometTrail[i] - 20);
        if (cometTrail[i] > 0) {
          // Показываем затухающий след кометы
          strip.set(i, mWheel(cometTrailHue[i], cometTrail[i]));
        }
        // Если след затух, остаётся зелёный фон (уже установлен выше)
      }
    }
    
    // Обновляем каждую комету
    for (int c = 0; c < 3; c++) {
      // Рисуем комету с хвостом (хвост тянется вниз, комета летит вверх)
      int tailLength = 12;
      for (int i = 0; i < tailLength; i++) {
        int pos = cometPos[c] - i;
        if (pos >= 0 && pos < NUMLEDS) {
          // Хвост становится длиннее и ярче внизу (где больше места)
          byte brightness = 255 - (i * 255 / tailLength);
          // Внизу хвост немного длиннее
          if (isBottom(pos)) {
            brightness = min(255, brightness + 20);
          }
          if (brightness > 0) {
            strip.set(pos, mWheel(cometHue[c], brightness));
            // Сохраняем информацию о следе для затухания
            cometTrail[pos] = brightness;
            cometTrailHue[pos] = cometHue[c];
          }
        }
      }
      
      // Двигаем комету вверх
      cometPos[c] += cometSpeed[c];
      
      // Когда комета достигает верха, запускаем новую снизу
      if (cometPos[c] >= NUMLEDS) {
        cometPos[c] = random(0, NUMLEDS / 4);  // Старт в нижней четверти
        cometHue[c] = (cometHue[c] + random(100, 300)) % 1530;  // Меняем цвет
        cometSpeed[c] = random(1, 3);
      }
      
      // Случайно меняем скорость
      if (random(0, 100) < 3) {
        cometSpeed[c] = random(1, 3);
      }
    }
    
    strip.show();
  }
  
  delay(40);
}

// Эффект 8: Гирлянды на елке
void aurora() {
  static byte auroraHue = 0;
  static int waveOffset = 0;
  
  // Создаём плавные волны цвета, учитывая коническую форму
  for (int i = 0; i < NUMLEDS; i++) {
    byte height = getHeight(i);
    
    // Создаём несколько наложенных волн для эффекта северного сияния
    // Волны адаптируются к высоте - внизу шире, вверху уже
    float heightFactor = 1.0 - (height / 255.0) * 0.3;  // Вверху волны уже
    float wave1 = sin((i + waveOffset) * 0.1 * heightFactor) * 0.5 + 0.5;
    float wave2 = sin((i + waveOffset * 1.3) * 0.15 * heightFactor) * 0.5 + 0.5;
    float wave3 = sin((i + waveOffset * 0.7) * 0.08 * heightFactor) * 0.5 + 0.5;
    
    // Комбинируем волны
    float combined = (wave1 + wave2 + wave3) / 3.0;
    // Внизу ярче (больше диодов), вверху тусклее
    byte baseBrightness = 200 - (height / 4);
    byte brightness = (byte)min(255, max(40, combined * baseBrightness + 30));
    
    // Создаём цвет из зелёно-сине-фиолетовой гаммы
    // Цвет меняется по высоте для более естественного вида
    int hue = (auroraHue + height / 2 + waveOffset / 2) % 1530;
    // Ограничиваем диапазон для северного сияния (зелёный-синий-фиолетовый)
    if (hue > 1020) hue = 1020 - (hue - 1020);
    if (hue < 340) hue = 340 + (340 - hue);
    
    strip.set(i, mWheel(hue, brightness));
  }
  
  waveOffset += 2;
  if (waveOffset > 1000) {
    waveOffset = 0;
    auroraHue = (auroraHue + 50) % 1530;
  }
  
  strip.show();
  delay(50);
}