/*************************************************************************************
 *
 * Skadis ring to hold a Wera 818/4/1 screw driver.
 *
 *************************************************************************************
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHOR OR COPYRIGHT
 * HOLDER BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * IT IS NOT PERMITTED TO MODIFY THIS COMMENT BLOCK.
 *
 * (c)2025, Claude "Tryphon" Theroux, Montreal, Quebec, Canada
 * http://www.ctheroux.com/
 *
 *************************************************************************************/
 
// Ring inner diameter in mm
InnerDiameter = 34;
 
// Ring height in mm
Height = 10;

// Ring thickness in mm
Thickness = 4; 

module __Customizer_Limit__ () {}

lct_MainParameters = true;


lctSFC_Debug = false;
lctSFC_Test = false;

module __Customizer_Limit__ () {}

/*************************************************************************************
 *
 * Nothing shall be modified below this line
 *
 *************************************************************************************/

/*************************************************************************************
 *
 * Library inner parameters
 *
 *************************************************************************************/

// Skadis hole length in mm
lctSFC_SkadisHoleLength = 15;

// Skadis hole radius in mm
lctSFC_SkadisHoleRadius = 2.5;

// Skadis thickness in mm
lctSFC_SkadisThickness = 5;

// Skadis distance between two vertical holes in mm
lctSFC_SkadisVerticalHoleDistance = 40;

// Back Holder thickness in mm
lctSFC_BackHolderThickness = 4;

// Holder link height in mm
lctSFC_HolderLinkHeight = 6;

// French cheat bar width in mm
lctSFC_FrontHolderWidth = 10;

// French cleat depth in mm
lctSFC_FrontHolderThickness = 4;

// General clearance in mm
lctSFC_Clearance = 0.5;
 
$fn = 60;

// Back holder width in mm
lctSFC_BackHolderWidth = lctSFC_SkadisHoleRadius * 2 - lctSFC_Clearance;

// Back holder core length in mm
lctSFC_BackHolderCoreLength = lctSFC_SkadisHoleLength - 2 * lctSFC_SkadisHoleRadius;

// French cheat bar length in mm
lctSFC_FrontHolderLength = lctSFC_BackHolderCoreLength + lctSFC_SkadisHoleRadius - lctSFC_Clearance / 2;

module lctSFC_Debug(p_EnableDebug, p_HolderNumber, p_FrontHolderLength, p_FrontHolderWidth, p_FrontHolderThickness) {
    if( p_EnableDebug ) {
        echo("lctSFC_SkadisHoleLength:", lctSFC_SkadisHoleLength);
        echo("lctSFC_SkadisHoleRadius:", lctSFC_SkadisHoleRadius);
        echo("lctSFC_SkadisThickness:", lctSFC_SkadisThickness);
        echo("lctSFC_SkadisVerticalHoleDistance:", lctSFC_SkadisVerticalHoleDistance);
        echo("p_HolderNumber:", p_HolderNumber);
        echo("p_FrontHolderLength:", p_FrontHolderLength);
        echo("p_FrontHolderWidth:", p_FrontHolderWidth);
        echo("p_FrontHolderThickness:", p_FrontHolderThickness);
        echo("lctSFC_BackHolderThickness:", lctSFC_BackHolderThickness);
        echo("lctSFC_BackHolderWidth:", lctSFC_BackHolderWidth);
        echo("lctSFC_BackHolderCoreLength:", lctSFC_BackHolderCoreLength);
        echo("lctSFC_HolderLinkHeight:", lctSFC_HolderLinkHeight);
        echo("lctSFC_Clearance:", lctSFC_Clearance);
    }
}

module lctSFC_BackHolder() {
    hull() {
        translate([ -lctSFC_BackHolderCoreLength / 2, 0, 0 ]) cylinder(d = lctSFC_BackHolderWidth, h = lctSFC_BackHolderThickness);
        translate([ lctSFC_BackHolderCoreLength / 2, 0, 0 ]) cylinder(d = lctSFC_BackHolderWidth, h = lctSFC_BackHolderThickness);
    } 
}

module lctSFC_HolderLink() {
    difference() {
        cylinder( d = lctSFC_BackHolderWidth, h = lctSFC_SkadisThickness + lctSFC_Clearance / 2);
        translate([ 0, -lctSFC_BackHolderWidth / 2, 0 ]) cube([ lctSFC_BackHolderWidth / 2, lctSFC_BackHolderWidth, lctSFC_SkadisThickness + lctSFC_Clearance / 2 ]);
    }
    translate([ 0, -lctSFC_BackHolderWidth / 2, 0 ]) cube( [ lctSFC_HolderLinkHeight - lctSFC_BackHolderWidth / 2, lctSFC_BackHolderWidth, lctSFC_SkadisThickness + lctSFC_Clearance / 2 ] );
}

module lctSFC_FrontHolder( p_FrontHolderLength, p_FrontHolderWidth, p_FrontHolderThickness)
{
    translate([ 0, p_FrontHolderWidth / 2, 0 ]) rotate([ 90, 0, 0 ])
            linear_extrude(p_FrontHolderWidth) polygon([ 
                                                    [ 0, 0 ], 
                                                    [ p_FrontHolderLength, 0 ], 
                                                    [ p_FrontHolderLength + p_FrontHolderThickness, p_FrontHolderThickness ], 
                                                    [ 0, p_FrontHolderThickness ]
                                                       ]);    
}

module lctSFC_FrenchCleatHolder( p_HolderNumber = 1, p_FrontHolderLength = lctSFC_FrontHolderLength, 
                                p_FrontHolderWidth = lctSFC_FrontHolderWidth, 
                                p_FrontHolderThickness = lctSFC_FrontHolderThickness, 
                                p_EnableDebug = false)
{
    lctSFC_Debug(p_EnableDebug, p_HolderNumber, p_FrontHolderLength, p_FrontHolderWidth, p_FrontHolderThickness);
    translate([ 0, 0, lctSFC_BackHolderWidth / 2 ]) rotate([ 90, 0, 180 ]) union() {

        for( n = [ 0 : p_HolderNumber - 1 ] ) {
            translate([ -lctSFC_SkadisVerticalHoleDistance * n, 0, 0 ]) lctSFC_BackHolder();
            translate([ lctSFC_SkadisHoleLength / 2 - lctSFC_Clearance / 2 - lctSFC_HolderLinkHeight - lctSFC_SkadisVerticalHoleDistance * n, 0, lctSFC_BackHolderThickness ]) lctSFC_HolderLink();
        }

        translate([ -p_FrontHolderLength -  + lctSFC_SkadisVerticalHoleDistance * (p_HolderNumber - 1) + lctSFC_BackHolderCoreLength / 2, p_FrontHolderWidth / 2 - lctSFC_BackHolderWidth / 2, lctSFC_BackHolderThickness + lctSFC_SkadisThickness + lctSFC_Clearance / 2 ]) lctSFC_FrontHolder(p_FrontHolderLength + lctSFC_SkadisVerticalHoleDistance * (p_HolderNumber - 1), p_FrontHolderWidth, p_FrontHolderThickness);    
    }
}

function lctSFC_FrenchCleatHolderKnockoutLength(p_HolderNumber = 1, 
                                            p_FrontHolderLength = lctSFC_FrontHolderLength, 
                                            p_FrontHolderWidth = lctSFC_FrontHolderWidth, 
                                            p_FrontHolderThickness = lctSFC_FrontHolderThickness) = p_FrontHolderLength + lctSFC_SkadisVerticalHoleDistance * (p_HolderNumber - 1) + lctSFC_Clearance / 2;

function lctSFC_FrenchCleatHolderKnockoutThickness(p_FrontHolderThickness = lctSFC_FrontHolderThickness) = p_FrontHolderThickness + lctSFC_Clearance / 2;

function lctSFC_FrenchCleatHolderKnockoutWidth(p_FrontHolderWidth = lctSFC_FrontHolderWidth) = p_FrontHolderWidth + lctSFC_Clearance / 2;

function lctFrenchCleatHolderOffset(p_InnerSkadisHoles = 0, 
                                    p_FrontHolderWidth = lctSFC_FrontHolderWidth) = (lctSFC_SkadisVerticalHoleDistance * p_InnerSkadisHoles) - p_FrontHolderWidth + lctSFC_BackHolderWidth;

module lctSFC_FrenchCleatHolderKnockout(p_HolderNumber = 1, 
                                p_FrontHolderLength = lctSFC_FrontHolderLength, 
                                p_FrontHolderWidth = lctSFC_FrontHolderWidth, 
                                p_FrontHolderThickness = lctSFC_FrontHolderThickness, 
                                p_EnableDebug = false)
{
    lctSFC_Debug(p_EnableDebug, p_HolderNumber, p_FrontHolderLength, p_FrontHolderWidth, p_FrontHolderThickness);
    
    lctSFC_FrontHolder(p_FrontHolderLength + lctSFC_SkadisVerticalHoleDistance * (p_HolderNumber - 1) + lctSFC_Clearance / 2, p_FrontHolderWidth + lctSFC_Clearance / 2, p_FrontHolderThickness + lctSFC_Clearance / 2);

}

module lctSFC_TestIkeaSkadisFrenchCleatLibrary() {
    *lctSFC_BackHolder();
    *lctSFC_HolderLink();
    *lctSFC_FrontHolder();

    lctSFC_FrenchCleatHolder(1);
    
    *lctSFC_FrenchCleatHolderKnockout(1, p_EnableDebug = lctSFC_Debug);
    
    *lctSFC_FrenchCleatHolder(2, 50);
}


if( lctSFC_Test == true ) {
    lctSFC_TestIkeaSkadisFrenchCleatLibrary();
}

// Radius of the rounded corner of the ring. 
lct_MinkowskiRadius = 0.75;

// French cleat holder height and depth clearance in mm
lct_FrenchCleatClearance = 5;

module generateRing(innerDiameter = InnerDiameter, height = Height, thickness = Thickness) {
    
    translate( [ 0, 0, lct_MinkowskiRadius ] ) minkowski() {
        
        difference() {
            cylinder(d = innerDiameter + 2 * thickness - 2 * lct_MinkowskiRadius, h = height - 2 * lct_MinkowskiRadius);
            cylinder(d = innerDiameter + 2 * lct_MinkowskiRadius, h = height - 2 * lct_MinkowskiRadius);
        }
        sphere(r = lct_MinkowskiRadius, $fn = 18);
    }
}

module generateFrenchCleats(width) {
    
    numberFrenchCleat = floor((width - lctSFC_FrenchCleatHolderKnockoutWidth() * 2) / lctSFC_SkadisVerticalHoleDistance);
    
    offset = (width - lctSFC_SkadisVerticalHoleDistance * numberFrenchCleat) / 2 - lctSFC_FrontHolderWidth / 2 + lctSFC_BackHolderWidth / 2;
    
    difference() {
        cube( [ lctSFC_FrenchCleatHolderKnockoutThickness() + lct_FrenchCleatClearance, width, lctSFC_FrenchCleatHolderKnockoutLength() + lct_FrenchCleatClearance ] );

        translate( [ 0, offset, 0 ] ) for( i = [ 0 : 1 : numberFrenchCleat ] ) {
            translate( [ 0, lctSFC_SkadisVerticalHoleDistance * i, 0] ) rotate( [ 0, -90, 180 ] ) lctSFC_FrenchCleatHolderKnockout();
        }
    }
}

if( !is_undef(lct_MainParameters) && lct_MainParameters ) {
    
    ringOffset = ( Thickness > lct_FrenchCleatClearance ? Thickness - lct_FrenchCleatClearance : 0 );
    frenchCleatOffset = InnerDiameter / 2 +  ringOffset + lctSFC_FrenchCleatHolderKnockoutThickness() + lct_FrenchCleatClearance;

    translate( [ -frenchCleatOffset, -(InnerDiameter + 2 * Thickness) / 2, -(lctSFC_FrenchCleatHolderKnockoutLength() + lct_FrenchCleatClearance) + Height ] ) generateFrenchCleats(InnerDiameter + 2 * Thickness);

    generateRing(InnerDiameter, Height, Thickness);

    translate( [ 0, InnerDiameter + 2 * Thickness, 0 ] ) lctSFC_FrenchCleatHolder(1);
}
