/*
Based on Sprocket generator v2 - Corrected
https://www.thingiverse.com/thing:3059422
*/

////////////////////////////
/* [GLOBAL VARIABLES] */
//////////////////////////

thickness = 5;
pitch = 6.35;
shafts_distance = 100;
tolerance = 0.15;

////////////////////////////
/* [SPROCKET 1 VARIABLES] */
//////////////////////////
teeth1 = 20;
hub1_height = 2; 
hub1_diameter = 10;
bore1_diameter = 8;

////////////////////////////
/* [SPROCKET 2 VARIABLES] */
//////////////////////////
teeth2 = 10;
hub2_height = 2;
hub2_diameter = 10;
bore2_diameter = 8;

//////////////////////
/* [CHAIN-PARAMETERS] */
//////////////////////
link_plate_thickness = 2;
roller_thickness = 2;
pin_diameter = 3;
clearance = 0.3;

// DERIVED VALUES
roller_diameter = pin_diameter + 3*clearance + roller_thickness;
roller_inner_diameter = pin_diameter + 3*clearance;
link_width = thickness + 1;

// Pitch radii
pitch_radius1 = pitch / (2 * sin(180 / teeth1));
pitch_radius2 = pitch / (2 * sin(180 / teeth2));

//chain values
slant_distance = sqrt((pitch_radius1-pitch_radius2)^2+(shafts_distance)^2);
slant_angle=atan((pitch_radius1-pitch_radius2)/shafts_distance);
num_chain_links = floor(((PI * (pitch_radius1 + pitch_radius2) + 2*slant_distance)/ pitch)/2)+1;
echo("*************");
echo("numbers of chain links: ", num_chain_links);
echo("*************");

//////////////////////////
/* [RENDERING QUALITY] */
///////////////////////

fs = 0.25; // Minimum size of a fragment
fa = 1;    // Minimum angle for a fragment
fn = 120;  // Fixed number of fragments
$fs = fs; 
$fa = fa; 
$fn = fn;

/////////////
/* [VIEW] */
///////////

view = "all"; // ["sprocket 1","sprocket 2","chain components","assembled link","all"]

///////////////
/* Sprocket */
///////////////

module sprocket(teeth, bore_diameter, hub_height, hub_diameter) {
    difference() {
        union() {
            sprocket_gear(teeth);

            // Hub cylinder
            translate([0, 0, thickness])
                cylinder(h=hub_height, d=hub_diameter);
        }
        // Center bore
        translate([0, 0, -1])
            cylinder(h=hub_height + thickness + 2, d=bore_diameter);
        // outer diameter
        
    }
}

module sprocket_gear(teeth) {
    roller_radius = roller_diameter / 2;
    distance_from_center = pitch / (2 * sin(180 / teeth));
    angle = 360 / teeth;

    pitch_radius = sqrt((distance_from_center * distance_from_center) - 
                        (pitch * (roller_radius + tolerance)) + 
                        ((roller_radius + tolerance) * (roller_radius + tolerance)));

    difference() {
        union() {
            // Inner cylinder
            cylinder(r=pitch_radius, h=thickness);

            // Tooth cylinders
            for (tooth = [1:teeth]) {
                intersection() {
                    rotate([0, 0, angle * (tooth + 0.5)])
                        translate([distance_from_center, 0, 0])
                            cylinder(r=pitch - roller_radius - tolerance, h=thickness);
                    rotate([0, 0, angle * (tooth - 0.5)])
                        translate([distance_from_center, 0, 0])
                            cylinder(r=pitch - roller_radius - tolerance, h=thickness);
                }
            }
        }

        // Inner groove between teeth
        for (tooth = [1:teeth]) {
            rotate([0, 0, angle * (tooth + 0.5)])
                translate([distance_from_center, 0, -1])
                    cylinder(r=roller_radius + tolerance, h=thickness + 2);
        }
        //outer diameter
        translate([0,0,-1]) 
        difference(){
            cylinder(h=thickness+2,d=pitch*(0.6+1/tan(180/teeth)+5));
            cylinder(h=thickness+2,d=pitch*(0.6+1/tan(180/teeth)));
    }}
}

////////////
/* CHAIN */
//////////

module chain_plate(type) {
    difference() {
        hull() {
            translate([-pitch/2,0,0]) cylinder(h=link_plate_thickness, d=pin_diameter*2);
            translate([pitch/2,0,0]) cylinder(h=link_plate_thickness, d=pin_diameter*2);
        }
        if (type == "press fix") {
            translate([-pitch/2,0,-1]) cylinder(h=link_plate_thickness+2, d=pin_diameter+0.25);
            translate([pitch/2,0,-1]) cylinder(h=link_plate_thickness+2, d=pin_diameter+0.25);
        }
        if (type == "inner") {
            translate([-pitch/2,0,-1]) cylinder(h=link_plate_thickness+2, d=roller_inner_diameter);
            translate([pitch/2,0,-1]) cylinder(h=link_plate_thickness+2, d=roller_inner_diameter);
        }
    }
}

module chain_roller() {
    difference() {
        cylinder(h=link_width, d=roller_diameter);
        translate([0,0,-1]) cylinder(h=link_width+2, d=roller_inner_diameter);
    }
}

module outer() {
    chain_plate();
    translate([-pitch/2,0,0]) cylinder(h=link_width + 4*link_plate_thickness + 2*clearance, d=pin_diameter);
    translate([pitch/2,0,0]) cylinder(h=link_width + 4*link_plate_thickness + 2*clearance, d=pin_diameter);
}

module inner() {
    {
    chain_plate("inner");
    translate([-pitch/2,0,link_plate_thickness]) chain_roller();
    translate([pitch/2,0,link_plate_thickness]) chain_roller();
    translate([0,0,link_plate_thickness + link_width]) chain_plate("inner");
}}

module assembled_link() {
    color("BLUE") outer();
    translate([pitch,0,link_plate_thickness + clearance/2]) inner();
    translate([-pitch,0,link_plate_thickness + clearance/2]) inner();
    translate([0,0,3*link_plate_thickness + link_width + clearance]) color("RED") chain_plate("press fix");
}

// =========================
// MULTI-LINK CHAIN GENERATOR
// =========================

module chain_run()
{
    zoff = -thickness/2 - link_plate_thickness-clearance;
    
    r1 = pitch_radius1;
    r2 = pitch_radius2;

    // angular pitch per link (radians → degrees)
    dphi1 = 2*(pitch / r1) * 180 / PI;
    dphi2 = 2*(pitch / r2) * 180 / PI;

    // wrap angles (approximate, works visually)
    wrap1 = 180 + (r1 - r2)/shafts_distance * 180/PI;
    wrap2 = 180 - (r1 - r2)/shafts_distance * 180/PI;

    // ---------- SPROCKET 1 ARC ----------
    for(i = [0 :0.5: floor(wrap1/dphi1)+0.5]){
        a = 90 + i*dphi1;
            rotate([0,0,a])
                translate([r1,0,zoff])
                    rotate([0,0,90]) {
                        if(floor(i)==i) translate([0,0,link_plate_thickness]) inner();
                        if(floor(i)!=i) {
                            color("BLUE") outer();
                            translate([0,0,3*link_plate_thickness + link_width + clearance]) color("RED") chain_plate("press fix");}}
    }
    
    // ---------- SPROCKET 1 TO 2 SLANT RUN ----------
    straight_len = shafts_distance;
    straight_links = floor(straight_len/(2*pitch));

    for(i = [0 : 0.5 : straight_links+0.5])
    {
        rotate([0,0,-slant_angle])
            translate([2*i*pitch, r1, zoff]){
                        if(floor(i)==i) translate([0,0,link_plate_thickness]) inner();
                        if(floor(i)!=i) {
                            color("BLUE") outer();
                            translate([0,0,3*link_plate_thickness + link_width + clearance]) color("RED") chain_plate("press fix");}}
    }

    // ---------- SPROCKET 2 ARC ----------
    translate([shafts_distance,0,0])
    for(i = [0 :0.5: floor(wrap2/dphi2)+0.5])
    {
        a = 90 - i*dphi2;
        rotate([0,0,a])
            translate([r2,0,zoff])
                rotate([0,0,90])
                    {
                        if(floor(i)==i) translate([0,0,link_plate_thickness]) inner();
                        if(floor(i)!=i) {
                            color("BLUE") outer();
                            translate([0,0,3*link_plate_thickness + link_width + clearance]) color("RED") chain_plate("press fix");}}
    }

    // ---------- BOTTOM SLANT RUN ----------
    for(i = [0 :0.5: straight_links+0.5])
    {
        rotate([0,0,slant_angle])
        translate([shafts_distance - 2*i*pitch, -r1, zoff])
{
                        if(floor(i)==i) translate([0,0,link_plate_thickness]) inner();
                        if(floor(i)!=i) {
                            color("BLUE") outer();
                            translate([0,0,3*link_plate_thickness + link_width + clearance]) color("RED") chain_plate("press fix");}}
                            }
}

///////////////
// DISPLAY
///////////////

if (view=="chain components") {
    color("BLUE") outer();
    translate([0,-pitch-1,0]) inner();
    translate([0,pitch+1,0]) color("RED") chain_plate("press fix");
}

if (view=="sprocket 1") 
    sprocket(teeth1, bore1_diameter, hub1_height, hub1_diameter);

if (view=="sprocket 2") 
    sprocket(teeth2, bore2_diameter, hub2_height, hub2_diameter);

if(view=="assembled link"){
    assembled_link();
    translate([2*pitch,0,0]) assembled_link();
}
    
if (view=="all") {
    color("GREEN") sprocket(teeth1, bore1_diameter, hub1_height, hub1_diameter);
    translate([shafts_distance,0,0]) color("GREEN") sprocket(teeth2, bore2_diameter, hub2_height, hub2_diameter);
    chain_run();
}