// Great Stellated Dodecahedron (GSD) - OpenSCAD generator
// By Curt McDowell <maker@fishlet.com> 2025-02-13
// Licensed under the Creative Commons - Attribution - ShareAlike.
// https://creativecommons.org/licenses/by-sa/4.0/

// When we visualize the GSD from the outside -- ignoring the way the
// faces intersect internally -- we see an outer shell that looks like
// an icosahedron with triangular pyramids glued on. While any such
// shell is called a triakis icosahedron, we're interested in the one
// whose pyramid heights matches that of the GSD outer shell.

// This program generates a GSD whole or split into four parts, where
// the GSD consists of two "crown" parts and two "middle" parts. The
// four parts are suitable for 3D printing without overhangs or
// support material, so they are accurate and easy to glue up.

// Part to generate
Part = "whole";   // [whole: Great Stellated Dodecahedron, core: Inner Icosahedron, crown: Crown; two required for printing, middle: Middle; two required for printing]

// Short edge length; circumradius is 2.2673 times this
Edge = 20;

// Clearance for 3D printer part assembly, 0 for none
Clearance = 0.2;

// Add a hole for a hook (applies to the Middle part only)
Hook_Hole = false;     // [bool]

module _end_parameters() {}

r = sqrt(5);

gsd_points = [
    [      -2,    r + 1,        0],
    [       2,    r + 1,        0],
    [      -2,   -r - 1,        0],
    [       2,   -r - 1,        0],
    [       0,       -2,    r + 1],
    [       0,        2,    r + 1],
    [       0,       -2,   -r - 1],
    [       0,        2,   -r - 1],
    [   r + 1,        0,       -2],
    [   r + 1,        0,        2],
    [  -r - 1,        0,       -2],
    [  -r - 1,        0,        2],
    [  -r - 3,    r + 3,    r + 3],
    [       0,  2*r + 4,    r + 1],
    [       0,  2*r + 4,   -r - 1],
    [  -r - 3,    r + 3,   -r - 3],
    [-2*r - 4,    r + 1,        0],
    [   r + 3,    r + 3,    r + 3],
    [  -r - 1,        0,  2*r + 4],
    [-2*r - 4,   -r - 1,        0],
    [  -r - 1,        0, -2*r - 4],
    [   r + 3,    r + 3,   -r - 3],
    [   r + 3,   -r - 3,    r + 3],
    [       0, -2*r - 4,    r + 1],
    [       0, -2*r - 4,   -r - 1],
    [   r + 3,   -r - 3,   -r - 3],
    [ 2*r + 4,   -r - 1,        0],
    [   r + 1,        0,  2*r + 4],
    [  -r - 3,   -r - 3,    r + 3],
    [  -r - 3,   -r - 3,   -r - 3],
    [   r + 1,        0, -2*r - 4],
    [ 2*r + 4,    r + 1,        0],
] / 4;  // Coordinates above are all multiplied by 4

gsd_faces = [
    [ 5, 11, 12],    [11,  0, 12],    [ 0,  5, 12],
    [ 1,  5, 13],    [ 5,  0, 13],    [ 0,  1, 13],
    [ 7,  1, 14],    [ 1,  0, 14],    [ 0,  7, 14],
    [10,  7, 15],    [ 7,  0, 15],    [ 0, 10, 15],
    [11, 10, 16],    [10,  0, 16],    [ 0, 11, 16],
    [ 9,  5, 17],    [ 5,  1, 17],    [ 1,  9, 17],
    [ 4, 11, 18],    [11,  5, 18],    [ 5,  4, 18],
    [ 2, 10, 19],    [10, 11, 19],    [11,  2, 19],
    [ 6,  7, 20],    [ 7, 10, 20],    [10,  6, 20],
    [ 8,  1, 21],    [ 1,  7, 21],    [ 7,  8, 21],
    [ 4,  9, 22],    [ 9,  3, 22],    [ 3,  4, 22],
    [ 2,  4, 23],    [ 4,  3, 23],    [ 3,  2, 23],
    [ 6,  2, 24],    [ 2,  3, 24],    [ 3,  6, 24],
    [ 8,  6, 25],    [ 6,  3, 25],    [ 3,  8, 25],
    [ 9,  8, 26],    [ 8,  3, 26],    [ 3,  9, 26],
    [ 5,  9, 27],    [ 9,  4, 27],    [ 4,  5, 27],
    [11,  4, 28],    [ 4,  2, 28],    [ 2, 11, 28],
    [10,  2, 29],    [ 2,  6, 29],    [ 6, 10, 29],
    [ 7,  6, 30],    [ 6,  8, 30],    [ 8,  7, 30],
    [ 1,  8, 31],    [ 8,  9, 31],    [ 9,  1, 31],
];

// The points of the icosahedron are simply the first 12 points of the GSD
icosahedron_faces = [
    [5, 11, 0], [1, 5, 0], [7, 1, 0], [10, 7, 0], [11, 10, 0],
    [9, 5, 1], [4, 11, 5], [2, 10, 11], [6, 7, 10], [8, 1, 7],
    [4, 9, 3], [2, 4, 3], [6, 2, 3], [8, 6, 3], [9, 8, 3],
    [5, 9, 4], [11, 4, 2], [10, 2, 6], [7, 6, 8], [1, 8, 9]
];

normalize = function(v) v / norm(v);

// face is given by an array of three 3D points
// dir = 1 for inside (of polyhedron face belongs to), -1 for outside
// scale represents the size and thickness of the cutter
module face_cutter(face, dir, scale = 100) {
    // Unit vectors pointing out from each corner of face
    corner0_v = normalize((face[0] - face[1]) + (face[0] - face[2]));
    corner1_v = normalize((face[1] - face[0]) + (face[1] - face[2]));
    corner2_v = normalize((face[2] - face[0]) + (face[2] - face[1]));

    perp_v = normalize(cross(face[1] - face[0], face[2] - face[1]));

    // Cutter face in plane of original face (expanded triangle)
    f1 = (face +
          [corner0_v, corner1_v, corner2_v] * scale -
          0.001 * [perp_v, perp_v, perp_v]);

    // Similar cutter face offset perpendicularly to form other side of prism
    f2 = (f1 +
          dir * [perp_v, perp_v, perp_v] * scale);

    // Cutter polyhedron
    points = [each f1, each f2];
    faces = dir > 0 ?
        [[0, 1, 2], [0, 3, 4, 1], [1, 4, 5, 2], [2, 5, 3, 0], [3, 5, 4]] :
        [[2, 1, 0], [1, 4, 3, 0], [2, 5, 4, 1], [0, 3, 5, 2], [4, 5, 3]];

    polyhedron(points = points, faces = faces);
}

// Rotate/translate children in such a way that a 3D coplanar triangle
// (p1, p2, p3) would be made to rest on the Z=0 plane oriented with
// p1 at the origin and p2 on the +X axis.
module lay_flat(p1, p2, p3) {
    u = (p2 - p1) / norm(p2 - p1);
    t = (p3 - p1) / norm(p3 - p1);

    w = cross(u, t);
    v = cross(w, u);

    // Construct the 4x4 Transformation Matrix
    // The rows transform the global axes to the local vectors
    M = [[u.x, u.y, u.z, 0],
         [v.x, v.y, v.z, 0],
         [w.x, w.y, w.z, 0],
         [0,   0,   0,   1]];

    multmatrix(M)
        translate(-p1)
            children();
}

// minkowski_difference technique used by BOSL2
module minkowski_difference() {
    module bounding_box(excess = 0) {
        cube([1, 1, 1] * (Edge * 10 + excess), center = true);
    }

    difference() {
        bounding_box();
        render(convexity = 10) {
            minkowski() {
                difference() {
                    bounding_box(excess = 1);
                    children(0);
                }
                for (i = [1 : 1 : $children - 1])
                    children(i);
            }
        }
    }
}

gsd_points_scaled = Edge * gsd_points;

gsd_face = function(fn) [for (pn = [0 : 2]) gsd_points_scaled[gsd_faces[fn][pn]]];

module icosahedron_trimmed() {
    ips = gsd_points_scaled;

    intersection() {
        polyhedron(points = ips, faces = icosahedron_faces);
        f1 = [ips[5], ips[0], ips[8]];
        f2 = [ips[11], ips[10], ips[3]];
        // Trim off one 5-sided corner
        face_cutter(f1, 1);
        // This would trim off the opposite 5-sided corner leaving the
        // icosahedron center disc (prism with 10 side faces)
        //face_cutter(f2, -1);
        // This trims away the entire opposite half of the icosahedron core
        face_cutter((f1 + f2) / 2, -1);
    }
}

reverse3 = function(ary3) [ary3[2], ary3[1], ary3[0]];

// icos_face is between 0 and 19
module stellation(icos_face) {
    polyhedron(points = gsd_points_scaled,
               faces = [gsd_faces[icos_face * 3 + 0],
                        gsd_faces[icos_face * 3 + 1],
                        gsd_faces[icos_face * 3 + 2],
                        reverse3(icosahedron_faces[icos_face])]);
}

module part_crown() {
    cut_face = gsd_face(15);
    lay_flat(cut_face[1], cut_face[0], cut_face[2])
        intersection() {
            polyhedron(points = gsd_points_scaled, faces = gsd_faces,
                       convexity = 4);
            face_cutter(cut_face, -1);
        }
}

module part_middle() {
    ips = gsd_points_scaled;

    lay_flat(ips[5], ips[0], ips[8]) {
        scale(1.000001)     // for a slight overlap that prevents non-mesh
            icosahedron_trimmed();

        stellation(0);
        stellation(3);
        stellation(14);
        stellation(15);
        stellation(18);

        // Stellations corresponding to opposite half of icosahedron
        // core (they are not needed because we just print two identical
        // middle parts).

        //stellation(4);
        //stellation(6);
        //stellation(8);
        //stellation(10);
        //stellation(13);
    }
}

if (Part == "crown") {
    render()    // preview bogus as of 2025-May
        if (Clearance > 0) {
            minkowski_difference() {
                part_crown();
                sphere(r = Clearance / 2);
            }
        } else
            part_crown();
}

if (Part == "middle") {
    difference() {
        render()    // preview bogus as of 2025-May
            if (Clearance > 0) {
                    minkowski_difference() {
                        part_middle();
                        sphere(r = Clearance / 2);
                    }
            } else
                part_middle();
                
        if (Hook_Hole)    // Manually positioned (terrible hack):
            translate([Edge * 0.5, -Edge * 1.2, Edge * 0.07])
                rotate([0, 90, 0])
                    rotate_extrude($fn = 36)
                        translate([Edge / 2.5, 0, 0])
                            circle(Edge / 12.0, $fn = 60);
    }
}

// Animatable
if (Part == "whole") {
    rotate(v = gsd_points[0], a = $t * 360 / 5)
        polyhedron(points = gsd_points_scaled, faces = gsd_faces);
}

if (Part == "core") {
    polyhedron(points = gsd_points_scaled, faces = icosahedron_faces);
}
