// Guscio telecomando "NEW DIGITAL"
// Versione 2.9 - corretto parametro per guide cilindriche
//
// Novità vs 2.8: Aggiunta dentini blocca scheda
// Novità vs 2.7: semplificazione: spessore pareti laterali + spessore maschera tasti
// Novità vs 2.6:
// - Introdotti 2 parametri "semplici":
//     • SIDE_WALL_THICK      = spessore pareti laterali guscio
//     • TOP_KEYPLATE_THICK   = spessore finale desiderato della maschera tasti nel TOP
// - Pocket nel TOP "auto-limitato": non può mai scavare fino a staccare i post guida PCB/gommini
// - Rimossi keep-out complicati: non servono più
// - Mantiene: vano batterie MASTER + inviti contatti + post sul TOP + notch IR sul bottom
//
// =======================
// PARAMETRI GENERALI
// =======================

remote_len     = 170;
remote_w       = 45;
remote_thick   = 16;

top_thick      = 6;
bottom_thick   = remote_thick - top_thick;

corner_r       = 4;

SIDE_WALL_THICK   = 1.8;  // pareti laterali
TOP_KEYPLATE_THICK= 1.0;  // cielo del top (zona fori)
BOT_SKIN_THICK    = 1.8;  // pelle del bottom (puoi lasciarla 1.8)


// Pocket area (tuning)
TOP_POCKET_W    = remote_w   - 2*SIDE_WALL_THICK - 2;
TOP_POCKET_L    = remote_len - 2*SIDE_WALL_THICK - 6;
TOP_POCKET_Y    = 0;

// =======================
// PCB + DENTINI BLOCCA-SCHEDA (TOP, pareti interne laterali)
// =======================

PCB_LIP_ENABLE = true;

// PCB: quota rispetto alla giunzione TOP/BOTTOM (piano Z = bottom_thick)
PCB_Z_ABOVE_JOIN = -0.5;   // mm  
PCB_THICK        = 1.6;   // mm  (tipico FR4 1.6; cambia se diversa)
PCB_CLEARANCE_Z  = 0.2;   // mm  (gioco sopra PCB, per non schiacciare)

// Dentino: geometria
PCB_LIP_PROTRUSION = 3.0;   // mm, sporge verso il centro (asse X)
PCB_LIP_THICK_Z    = 0.6;   // mm, altezza dentino (asse Z)

PCB_LIP_Y_LEN      = 140;   // mm, lunghezza lungo Y
PCB_LIP_Y_CENTER   = 14;     // mm, posizione (centro) lungo Y

// Rampa/smusso di invito (sul bordo interno INFERIORE del dentino)
PCB_LIP_RAMP_X = 0.4;   // mm, quanto “mangia” in X (<= PROTRUSION)
PCB_LIP_RAMP_Z = 0.4;   // mm, altezza rampa in Z (<= THICK_Z)


// =======================
// VANO BATTERIE — MASTER GEOMETRY (UNICA SORGENTE)
// =======================

bat_int_w      = 21;
bat_int_l      = 47;

bat_wall_thick  = 1.6;
bat_wall_h      = 8.0;
bat_floor_thick = 1.2;

bat_offset_from_bottom = 10;
bat_center_x = 0;
bat_center_y = -remote_len/2 + bat_offset_from_bottom + bat_int_l/2;

bat_open_w = bat_int_w + 2*bat_wall_thick;
bat_open_l = bat_int_l + 2*bat_wall_thick;

bat_open_clear = 0.0;

// =======================
// INVITI GUIDATI PER CONTATTI METALLICI
// =======================

bat_contact_slot_w = 5.0;
bat_contact_slot_l = 2.0;
bat_contact_slot_h = 2.0;

bat_contact_neg_y = -bat_int_l/2 + 2.5;
bat_contact_pos_y =  bat_int_l/2 - 2.5;

bat_contact_x = 0;

bat_contact_boss_r = 2.5;
bat_contact_boss_h = 1.2;

// =======================
// DIMENSIONI FORI TIPICI (TASTI)
// =======================

slot_std_w     = 8;
slot_std_h     = 6;
slot_std_r     = 1.5;

slot_num_w     = 12;
slot_num_h     = 7;
slot_num_r     = 1.5;

slot_long_w    = 9;
slot_long_h    = 19;
slot_long_r    = 2.0;

round_small_r  = 3;
round_big_r    = 4;

ok_r           = 5;

dpad_outer_r   = 15;
dpad_inner_r   = 7;
dpad_sector_w  = 20;
dpad_sector_h  = 15;

D_PAD_BRACE_LENGTH = 2*dpad_outer_r;
D_PAD_BRACE_WIDTH  = 1.2;
D_PAD_BRACE_ANGLE  = 45;

// =======================
// FINESTRA LED INFRAROSSI
// =======================

IR_WIN_W       = 8;
IR_WIN_DEPTH   = 3;
IR_WIN_X       = 0;

// Notch IR anche su BOTTOM (solo sul bordo di giunzione)
IR_WIN_DEPTH_BOT = 2.5;
IR_WIN_NOTCH_Z   = 3.0;
IR_WIN_Z0_BOT    = bottom_thick - IR_WIN_NOTCH_Z;

// =======================
// POSIZIONE TASTI (EDITABILE)
// =======================

// NUMERICO 3x4
NUM_COL_LEFT_X   = -14;
NUM_COL_MID_X    =   0;
NUM_COL_RIGHT_X  =  14;

NUM_ROW1_Y       = -25;
NUM_ROW2_Y       = -34;
NUM_ROW3_Y       = -43;
NUM_ROW4_Y       = -52;

// VOL / CH
VOL_ROW_Y          =  -7;
VOL_COL_X          = -14;
CH_COL_X           =  13;

VOLCH_CENTER_X     =   0;
VOLCH_TOP_Y        =   -1;
VOLCH_BOTTOM_Y     =  -13;

// D-PAD
DPAD_CENTER_Y      =  25;

PAD_TL_X           = -16;
PAD_TL_Y           =  41;
PAD_TR_X           =  15;
PAD_TR_Y           =  41;
PAD_BL_X           = -16;
PAD_BL_Y           =  10;
PAD_BR_X           =  15;
PAD_BR_Y           =  10;

// ZONA ALTA
POWER_ROW_Y        =  70;
POWER_LEFT_X       = -15;
POWER_RIGHT_X      =  15;

COLOR_ROW_Y        =  61;
COLOR_COL1_X       = -15;
COLOR_COL2_X       =  -5;
COLOR_COL3_X       =   5;
COLOR_COL4_X       =  15;

FUNC_ROW_Y         =  51;
FUNC_COL1_X        = COLOR_COL1_X;
FUNC_COL2_X        = COLOR_COL2_X;
FUNC_COL3_X        = COLOR_COL3_X;
FUNC_COL4_X        = COLOR_COL4_X;

// =======================
// CHIUSURA A MAGNETI
// =======================

mag_r          = 2.6;
mag_h          = 2.0;
mag_base       = 0.6;
mag_ring_thick = 2.0;

mag_corner_dx = remote_w/2  - SIDE_WALL_THICK - mag_r - 0.5;
mag_corner_dy = remote_len/2 - SIDE_WALL_THICK - mag_r - 0.5;

module magnet_rings_top(){
    for (sx = [-1, 1])
    for (sy = [-1, 1]) {
        translate([sx*mag_corner_dx, sy*mag_corner_dy, bottom_thick])
            difference(){
                cylinder(h=mag_h, r=mag_r + mag_ring_thick, $fn=48);
                cylinder(h=mag_h - mag_base + 0.1, r=mag_r+0.05, $fn=48);
            }
    }
}

module magnet_rings_bottom(){
    for (sx = [-1, 1])
    for (sy = [-1, 1]) {
        translate([sx*mag_corner_dx, sy*mag_corner_dy, bottom_thick - mag_h])
            difference(){
                cylinder(h=mag_h, r=mag_r + mag_ring_thick, $fn=48);
                cylinder(h=mag_h - mag_base + 0.1, r=mag_r+0.05, $fn=48);
            }
    }
}

module magnets_cut_top(){
    for (sx = [-1, 1])
    for (sy = [-1, 1]) {
        translate([sx*mag_corner_dx, sy*mag_corner_dy, bottom_thick])
            cylinder(h=mag_h - mag_base, r=mag_r, $fn=32);
    }
}

module magnets_cut_bottom(){
    for (sx = [-1, 1])
    for (sy = [-1, 1]) {
        translate([sx*mag_corner_dx, sy*mag_corner_dy,
                   bottom_thick - (mag_h - mag_base)])
            cylinder(h=mag_h - mag_base, r=mag_r, $fn=32);
    }
}

// =======================
// PREMI SCHEDA (TOP) — 2 CILINDRI GUIDA
// =======================

bp_post_r = 1.5;

bp_post1_x = 0;
bp_post1_y = 42;
bp_post2_x = 0;
bp_post2_y = -19.5;

bp_post1_h = 12.0;
bp_post2_h = 12.0;

bp_post_end_z = remote_thick - TOP_KEYPLATE_THICK;

module board_posts_top(){
    translate([bp_post1_x, bp_post1_y, bp_post_end_z - bp_post1_h])
        cylinder(h=bp_post1_h, r=bp_post_r, $fn=24);

    translate([bp_post2_x, bp_post2_y, bp_post_end_z - bp_post2_h])
        cylinder(h=bp_post2_h, r=bp_post_r, $fn=24);
}

// =======================
// BARRE PREMISCHEDE (BOTTOM)
// =======================

bp_bar1_y      =  65;
bp_bar1_base_z = SIDE_WALL_THICK;
bp_bar1_len_x  = 30;
bp_bar1_width  = 3;
bp_bar1_thick  = 8;

bp_bar2_y      = 10;
bp_bar2_base_z = SIDE_WALL_THICK;
bp_bar2_len_x  = 30;
bp_bar2_width  = 3;
bp_bar2_thick  = 8;

module board_bars_bottom(){
    translate([0, bp_bar1_y, bp_bar1_base_z + bp_bar1_thick/2])
        cube([bp_bar1_len_x, bp_bar1_width, bp_bar1_thick], center=true);

    translate([0, bp_bar2_y, bp_bar2_base_z + bp_bar2_thick/2])
        cube([bp_bar2_len_x, bp_bar2_width, bp_bar2_thick], center=true);
}

// =======================
// DENTINI BLOCCA SCHEDA (TOP)
// =======================

module pcb_lips_top(){
    if (PCB_LIP_ENABLE){

        pcb_bottom_z = bottom_thick + PCB_Z_ABOVE_JOIN;
        pcb_top_z    = pcb_bottom_z + PCB_THICK;
        lip_z0       = pcb_top_z + PCB_CLEARANCE_Z;

        ramp_x = min(PCB_LIP_RAMP_X, PCB_LIP_PROTRUSION);
        ramp_z = min(PCB_LIP_RAMP_Z, PCB_LIP_THICK_Z);

        inner_x = (remote_w/2) - SIDE_WALL_THICK;

        // loop: side=-1 sinistra, side=+1 destra
        for (side = [-1, 1]) {

            wall_x = side * inner_x;

            // punta verso il centro:
            // sinistra (side=-1) -> verso +X
            // destra   (side=+1) -> verso -X
            x0 = wall_x;                           // attacco parete
            x1 = wall_x - side*PCB_LIP_PROTRUSION; // punta verso centro

            difference(){
                // Corpo dentino
                translate([(x0+x1)/2, PCB_LIP_Y_CENTER, lip_z0 + PCB_LIP_THICK_Z/2])
                    cube([abs(x1-x0), PCB_LIP_Y_LEN, PCB_LIP_THICK_Z], center=true);

                // Rampa: sottraggo un cuneo sul bordo interno INFERIORE
                if (ramp_x > 0 && ramp_z > 0){
                    inner_edge_x = x1;

                    hull(){
                        translate([inner_edge_x + side*(ramp_x/2), PCB_LIP_Y_CENTER, lip_z0 + 0.01])
                            cube([ramp_x, PCB_LIP_Y_LEN, 0.02], center=true);

                        translate([inner_edge_x, PCB_LIP_Y_CENTER, lip_z0 + ramp_z])
                            cube([0.02, PCB_LIP_Y_LEN, 0.02], center=true);
                    }
                }
            }
        }
    }
}



// =======================
// FORME DI BASE
// =======================

module rounded_rect_2d(w,l,r){
    minkowski(){
        square([w - 2*r, l - 2*r], center=true);
        circle(r=r, $fn=32);
    }
}

module rounded_slot_cut(x,y,w,l,r){
    translate([x,y,bottom_thick-0.1])
        linear_extrude(height=top_thick+0.2)
            rounded_rect_2d(w,l,r);
}

module body_outer(){
    linear_extrude(height=remote_thick)
        rounded_rect_2d(remote_w, remote_len, corner_r);
}


module body_shell(){
    difference(){
        body_outer();

        translate([0,0,BOT_SKIN_THICK])
            linear_extrude(height = remote_thick - TOP_KEYPLATE_THICK - BOT_SKIN_THICK)
                rounded_rect_2d(
                    remote_w - 2*SIDE_WALL_THICK,
                    remote_len - 2*SIDE_WALL_THICK,
                    max(corner_r - SIDE_WALL_THICK, 0.1)
                );
    }
}


// =======================
// VANO BATTERIE (BOTTOM)
// =======================

module battery_compartment_master(){
    translate([bat_center_x, bat_center_y, 0])
    union(){
        translate([0,0,bat_floor_thick/2])
            cube([bat_int_w, bat_int_l, bat_floor_thick], center=true);

        translate([0,0,bat_floor_thick + bat_wall_h/2])
            difference(){
                cube([bat_open_w, bat_open_l, bat_wall_h], center=true);
                cube([bat_int_w,  bat_int_l,  bat_wall_h + 0.1], center=true);
            }
    }
}

module battery_opening_cut(){
    translate([bat_center_x, bat_center_y, 0])
        cube([bat_int_w + bat_open_clear,
              bat_int_l + bat_open_clear,
              bottom_thick + 0.2], center=true);
}

module battery_contact_guides(){
    translate([bat_center_x, bat_center_y, 0])
    union(){
        translate([bat_contact_x, bat_contact_neg_y, bat_floor_thick + bat_contact_slot_h/2])
            cube([bat_contact_slot_w, bat_contact_slot_l, bat_contact_slot_h], center=true);

        translate([bat_contact_x, bat_contact_pos_y, bat_floor_thick + bat_contact_slot_h/2])
            cube([bat_contact_slot_w, bat_contact_slot_l, bat_contact_slot_h], center=true);

        translate([bat_contact_x, bat_contact_neg_y, bat_floor_thick])
            cylinder(h=bat_contact_boss_h, r=bat_contact_boss_r, $fn=32);

        translate([bat_contact_x, bat_contact_pos_y, bat_floor_thick])
            cylinder(h=bat_contact_boss_h, r=bat_contact_boss_r, $fn=32);
    }
}

module battery_frame(){
    frame_thick = 1.0;
    translate([bat_center_x, bat_center_y, SIDE_WALL_THICK/2])
    difference(){
        cube([bat_open_w, bat_open_l, SIDE_WALL_THICK], center=true);
        cube([bat_open_w-2*frame_thick, bat_open_l-2*frame_thick, SIDE_WALL_THICK+0.02], center=true);
    }
}

// =======================
// Alleggerimento TOP: pocket auto-limitato (non stacca i post)
// =======================

module top_plate_thinning_cut(){
    // quanto vorremmo scavare
    pocket_h = max(top_thick - TOP_KEYPLATE_THICK, 0);

    if (pocket_h > 0) {

        // i post iniziano (in basso) a questa quota:
        post_start_z = min(bp_post_end_z - bp_post1_h, bp_post_end_z - bp_post2_h);

        // margine di sicurezza
        POST_SAFE_GAP = 0.8;

        // limite massimo di scavo in altezza partendo da bottom_thick
        max_pocket_top_z = post_start_z - POST_SAFE_GAP;
        max_pocket_h = max(0, max_pocket_top_z - bottom_thick);

        pocket_h_eff = min(pocket_h, max_pocket_h);

        if (pocket_h_eff > 0) {
            translate([0, TOP_POCKET_Y, bottom_thick + pocket_h_eff/2])
                cube([TOP_POCKET_W, TOP_POCKET_L, pocket_h_eff + 0.2], center=true);
        }
    }
}


// =======================
// FINESTRA LED IR
// =======================

module ir_window_cut(){
    translate([IR_WIN_X - IR_WIN_W/2,
               remote_len/2 - IR_WIN_DEPTH,
               bottom_thick])
        cube([IR_WIN_W, IR_WIN_DEPTH, top_thick], center=false);
}

module ir_window_cut_bottom(){
    translate([IR_WIN_X - IR_WIN_W/2,
               remote_len/2 - IR_WIN_DEPTH_BOT,
               IR_WIN_Z0_BOT])
        cube([IR_WIN_W, IR_WIN_DEPTH_BOT, IR_WIN_NOTCH_Z], center=false);
}

// =======================
// CROCE DI SUPPORTO D-PAD
// =======================

module dpad_brace_2d(){
    for (ang = [ D_PAD_BRACE_ANGLE, -D_PAD_BRACE_ANGLE ]) {
        rotate(ang)
            square([D_PAD_BRACE_LENGTH, D_PAD_BRACE_WIDTH], center=true);
    }
}

// =======================
// FORI TASTI
// =======================

module numeric_buttons_cut(){
    xs = [NUM_COL_LEFT_X, NUM_COL_MID_X, NUM_COL_RIGHT_X];
    ys = [NUM_ROW1_Y, NUM_ROW2_Y, NUM_ROW3_Y, NUM_ROW4_Y];
    for (i = [0:3])
        for (j = [0:2])
            rounded_slot_cut(xs[j], ys[i], slot_num_w, slot_num_h, slot_num_r);
}

module vol_ch_buttons_cut(){
    rounded_slot_cut(VOL_COL_X, VOL_ROW_Y, slot_long_w, slot_long_h, slot_long_r);
    rounded_slot_cut(CH_COL_X,  VOL_ROW_Y, slot_long_w, slot_long_h, slot_long_r);

    translate([VOLCH_CENTER_X, VOLCH_TOP_Y, bottom_thick-0.1])
        cylinder(h=top_thick+0.2, r=round_big_r, $fn=32);

    translate([VOLCH_CENTER_X, VOLCH_BOTTOM_Y, bottom_thick-0.1])
        cylinder(h=top_thick+0.2, r=round_big_r, $fn=32);
}

module dpad_ok_cut(){
    translate([0,DPAD_CENTER_Y,bottom_thick-0.1])
        cylinder(h=top_thick+0.3, r=ok_r, $fn=48);
}

module dpad_arrow_sector(direction){
    translate([0,DPAD_CENTER_Y,bottom_thick-0.1])
        linear_extrude(height=top_thick+0.3)
            difference(){
                intersection(){
                    difference(){
                        circle(r=dpad_outer_r, $fn=64);
                        circle(r=dpad_inner_r, $fn=64);
                    }
                    if (direction == "up")
                        translate([0, dpad_outer_r/2])
                            square([dpad_sector_w,dpad_sector_h], center=true);
                    else if (direction == "down")
                        translate([0,-dpad_outer_r/2])
                            square([dpad_sector_w,dpad_sector_h], center=true);
                    else if (direction == "left")
                        translate([-dpad_outer_r/2,0])
                            square([dpad_sector_h,dpad_sector_w], center=true);
                    else if (direction == "right")
                        translate([ dpad_outer_r/2,0])
                            square([dpad_sector_h,dpad_sector_w], center=true);
                }
                dpad_brace_2d();
            }
}

module dpad_buttons_cut(){
    dpad_ok_cut();
    dpad_arrow_sector("up");
    dpad_arrow_sector("down");
    dpad_arrow_sector("left");
    dpad_arrow_sector("right");
}

module pad_round_buttons_cut(){
    translate([PAD_TL_X, PAD_TL_Y, bottom_thick-0.1])
        cylinder(h=top_thick+0.2, r=round_small_r, $fn=32);
    translate([PAD_TR_X, PAD_TR_Y, bottom_thick-0.1])
        cylinder(h=top_thick+0.2, r=round_small_r, $fn=32);
    translate([PAD_BL_X, PAD_BL_Y, bottom_thick-0.1])
        cylinder(h=top_thick+0.2, r=round_small_r, $fn=32);
    translate([PAD_BR_X, PAD_BR_Y, bottom_thick-0.1])
        cylinder(h=top_thick+0.2, r=round_small_r, $fn=32);
}

module top_zone_buttons_cut(){
    rounded_slot_cut(POWER_LEFT_X,  POWER_ROW_Y, slot_std_w, slot_std_h, slot_std_r);
    rounded_slot_cut(POWER_RIGHT_X, POWER_ROW_Y, slot_std_w, slot_std_h, slot_std_r);

    rounded_slot_cut(COLOR_COL1_X, COLOR_ROW_Y, slot_std_w, slot_std_h, slot_std_r);
    rounded_slot_cut(COLOR_COL2_X, COLOR_ROW_Y, slot_std_w, slot_std_h, slot_std_r);
    rounded_slot_cut(COLOR_COL3_X, COLOR_ROW_Y, slot_std_w, slot_std_h, slot_std_r);
    rounded_slot_cut(COLOR_COL4_X, COLOR_ROW_Y, slot_std_w, slot_std_h, slot_std_r);

    rounded_slot_cut(FUNC_COL1_X, FUNC_ROW_Y, slot_std_w, slot_std_h, slot_std_r);
    rounded_slot_cut(FUNC_COL2_X, FUNC_ROW_Y, slot_std_w, slot_std_h, slot_std_r);
    rounded_slot_cut(FUNC_COL3_X, FUNC_ROW_Y, slot_std_w, slot_std_h, slot_std_r);
    rounded_slot_cut(FUNC_COL4_X, FUNC_ROW_Y, slot_std_w, slot_std_h, slot_std_r);
}

module all_buttons_cut(){
    numeric_buttons_cut();
    vol_ch_buttons_cut();
    dpad_buttons_cut();
    pad_round_buttons_cut();
    top_zone_buttons_cut();
}

// =======================
// METÀ SUPERIORE
// =======================

module top_shell(){
    difference(){
        union(){
            intersection(){
                body_shell();
                translate([0,0,bottom_thick + top_thick/2])
                    cube([remote_w*2, remote_len*2, top_thick], center=true);
            }
            magnet_rings_top();
            board_posts_top();
            pcb_lips_top();   // <-- dentini blocca scheda
        }
        all_buttons_cut();
        ir_window_cut();
        top_plate_thinning_cut();
        magnets_cut_top();
    }
}

// =======================
// METÀ INFERIORE
// =======================

module bottom_shell(){
    difference(){
        union(){
            intersection(){
                body_shell();
                translate([0,0,bottom_thick/2])
                    cube([remote_w*2, remote_len*2, bottom_thick], center=true);
            }

            battery_frame();
            battery_compartment_master();
            battery_contact_guides();

            magnet_rings_bottom();
            board_bars_bottom();
        }

        battery_opening_cut();
        ir_window_cut_bottom();
        magnets_cut_bottom();
    }
}

// =======================
// COPERCHIO VANO BATTERIE (CLIP)
// =======================

module battery_cover(){
    cover_margin   = 1.0;
    cover_thick    = 1.0;
    lip_depth      = 2.0;
    lip_clearance  = 0.4;
    lip_inner_gap  = 1.2;
    clip_hook      = 0.4;
    clip_len       = 8.0;

    translate([bat_center_x, bat_center_y, 0])
    union(){
        translate([0,0,cover_thick/2])
            cube([bat_open_w + 2*cover_margin,
                  bat_open_l + 2*cover_margin,
                  cover_thick], center=true);

        difference(){
            translate([0,0,-lip_depth/2])
                cube([bat_open_w - lip_clearance,
                      bat_open_l - lip_clearance,
                      lip_depth], center=true);
            translate([0,0,-lip_depth/2])
                cube([bat_open_w - lip_clearance - 2*lip_inner_gap,
                      bat_open_l - lip_clearance - 2*lip_inner_gap,
                      lip_depth + 0.1], center=true);
        }

        for (sx = [-1, 1]){
            translate([sx*((bat_open_w - lip_clearance)/2), 0, -lip_depth/2])
                cube([clip_hook, clip_len, lip_depth], center=true);
        }
    }
}

// =======================
// OUTPUT
// =======================
// Scommenta ciò che vuoi esportare:

top_shell();
//bottom_shell();
//battery_cover();
