/* =========================================================================================
 *                     AEROPIC's SBUS/PPM/PWM RP2040 zero smart joystick
 * =========================================================================================
 * V300 
 *
 * This is a smart radio controlled joystick for RC simulators.
 * Not only it interfaces with a RX PPM or SBUS output but it senses the signal present on GPIO 1,
 * it recognizes the emitted protocol and adapt itself.
 * supported protocols are:
 * PPM (all polarities via ppmSpy).
 * SBUS Standard (Inverted).
 * SBUS TTL (Non-inverted).
 * 6-CH PWM (GPIO 14-9).
 *
 * To change the protocol, power OFF the device to perform a reset, connect the right port of the RX then power ON the RP2040. 
 * It performs a cyclic search while nothing is output by the TX. 
 * 
 * It uses USB HID 16 bits (for a maximal accuracy on the simulator).
 *
 * hardware
 * ========
 * for a 3.3V logic RX (FS-IA6B)solder a servo wire red to +5V, black to GND, white (PPM or SBUS signal) to GPIO 1 
 * (for a 5V logic RX add a resistor bridge GND --5.6k-- GPIO1 --2.2k-- PPM/SBUS signal)
 *
 * configuration (important)
 * ========================
 * add Earle Philhower"s core "https://github.com/earlephilhower/arduino-pico/releases/download/global/package_rp2040_index.json" in preferences/additionnal boards URL
 * in board manager type RP2040 and select RP3040/RP2050 by Earle Philhower 
 * in "tools/board" select "generic RP2040"
 * USB stack : "tiny USB"
 * BOOT STAGE2 : W25Q080 QSPI/2  <==== DON'T FORGET !
 *
 * in "tools/manage libraries" include "SBUS" bolder flight, "PPM-reader" and "ADAFruit tiny USB"
 *
 * power the chip in BOOT mode, compile and upload this code
 *
 * user manual
 * ===========
 * to test the joystick on W10 or W11, just type "joy.cpl" in the search area. The device RP2040 should appear,
 * press properties, the "test" tab is activated, move some axis
 * then go to parameters and calibrate the joystick 
 * note that in PPM the number of chanels is dependning upon the used RX. 
 * FSIA6B only gets 6 channels so, in PPM, you get the 6 first propotionnal axis on the joystick. 
 * If you use the SBUS output of the RX, you'll get 8 chanels, 8 buttons
 *
 * the LED shows the status of communication:
 * - flashing blue/purple/yellow : searching protocol : no signal is received, either the RX is not connected or the radio is OFF
 * - solid purple : the PPM signal is received and valid
 * - solid cyan : SBUS signal is received and valid
 * - solid white : PWM signals (6 channels) are received and valid
 * - solid red : the SBUS TX signal is lost, failsafe is active (not implement for PPM)
 *
 * version:
 * 
 * v300 : 271225 PWM 6-CH support (GPIO 9-14)
 * v200 : 251220 SBUS support, protocol auto detect
 * V100 : PPM support
 * =============================================================================================================
*/
#include "Adafruit_TinyUSB.h"
#include "sbus.h"         
#include "PPMReader.h"     // https://github.com/dimag0g/PPM-reader
#include <Adafruit_NeoPixel.h>

// --- Configuration Pins ---
#define INPUT_PIN 1
#define PIN_NEOPIXEL 16
const uint pwm_pins[] = {14, 13, 12, 11, 10, 9}; 

// --- Objets ---
Adafruit_NeoPixel pixel(1, PIN_NEOPIXEL, NEO_GRB + NEO_KHZ800);
bfs::SbusRx sbus_rx(&Serial1);
bfs::SbusData sbusData;
PPMReader ppm(INPUT_PIN, 16);
Adafruit_USBD_HID hid;

// --- États de l'automate ---
enum Protocol { SCAN_SBUS_INV, SCAN_SBUS_NORMAL, SCAN_PPM, SCAN_PWM, LOCKED_SBUS, LOCKED_PPM, LOCKED_PWM };
Protocol currentProtocol = SCAN_SBUS_INV;

unsigned long lastSwitchTime = 0;
unsigned long lastValidFrameTime = 0; 
const unsigned long SCAN_TIMEOUT = 1000; 
bool failsafe = false;

// --- Descriptor HID ---
uint8_t const desc_hid[] = {
        0x05, 0x01,        // Usage Page (Generic Desktop)
        0x09, 0x05,        // Usage (Gamepad)
        0xA1, 0x01,        // Collection (Application)
      
        // --- 8 boutons ---
        0x05, 0x09,        // Usage Page (Button)
        0x19, 0x01,        // Usage Min
        0x29, 0x08,        // Usage Max
        0x15, 0x00,        // Logical Min
        0x25, 0x01,        // Logical Max
        0x75, 0x01,        // Report Size (1 bit)
        0x95, 0x08,        // Report Count (8 bits = 8 boutons)
        0x81, 0x02,        // Input (Data,Var,Abs)
      
        // --- 8 axes 16 bits ---
        0x05, 0x01,        // Usage Page (Generic Desktop)
        0x09, 0x30,        // X
        0x09, 0x31,        // Y
        0x09, 0x32,        // Z
        0x09, 0x33,        // Rx
        0x09, 0x34,        // Ry
        0x09, 0x35,        // Rz
        0x09, 0x36,        // Slider 1
        0x09, 0x37,        // Slider 2 
      
        0x16, 0x00, 0x80,  // Logical Min = -32768
        0x26, 0xFF, 0x7F,  // Logical Max = 32767
        0x75, 0x10,        // Report Size = 16 bits
        0x95, 0x08,        // Report Count = 8 axes
      
        0x81, 0x02,        // Input (Data,Var,Abs)
      
        0xC0               // End Collection
      };

#define ATTRIBUTE_PACKED __attribute__((packed, aligned(1)))
typedef struct ATTRIBUTE_PACKED {
    uint8_t buttons;    
    int16_t axis[8];    
} HID_GamepadReport_Data_t;

HID_GamepadReport_Data_t reportData;

void setup() {
  pixel.begin();
  pixel.setBrightness(50);  // set the led brightness in %
  
  Serial1.setRX(INPUT_PIN);
  Serial1.setTX(0);
  
  hid.setReportDescriptor(desc_hid, sizeof(desc_hid));
  hid.begin();

  ppm.minChannelValue = 980; // The minimum possible channel value. Should be smaller than maxChannelValue.
  ppm.maxChannelValue = 2020; // The maximum possible channel value. Should be greater than minChannelValue.

  for(int i=0; i<6; i++) {
    pinMode(pwm_pins[i], INPUT_PULLUP);
  }
  // init of the first state of the automaton
  lastSwitchTime = millis();
  Serial1.setInvertRX(true);
  Serial1.begin(100000, SERIAL_8E2);
  sbus_rx.Begin();
}

void updateLED() {
  if (failsafe) {
      pixel.setPixelColor(0, pixel.Color(255, 0, 0)); // Red for failsafe
  } else {
    if (currentProtocol == LOCKED_SBUS) pixel.setPixelColor(0, pixel.Color(0, 255, 255));   // "c"yan Blue = "S"Bus
    else if (currentProtocol == LOCKED_PPM) pixel.setPixelColor(0, pixel.Color(255, 0, 255));  // "P"urple "P"ink = PPm
    else if (currentProtocol == LOCKED_PWM) pixel.setPixelColor(0, pixel.Color(255, 255, 255));  // "W"hite = pWm
    else { 
      if ((millis() / 200) % 2 == 0) pixel.setPixelColor(0, pixel.Color(0, 0, 255)); // flashing blue waiting for protocol
      else pixel.setPixelColor(0, pixel.Color(0, 0, 0));
    }
  }
  pixel.show();
}

void loop() {
  updateLED();

  switch (currentProtocol) {
    case SCAN_SBUS_INV:
      if (sbus_rx.Read()) { currentProtocol = LOCKED_SBUS; lastValidFrameTime = millis(); }
      else if (millis() - lastSwitchTime > SCAN_TIMEOUT) {
        Serial1.end(); Serial1.setInvertRX(false); Serial1.begin(100000, SERIAL_8E2);
        sbus_rx.Begin(); currentProtocol = SCAN_SBUS_NORMAL; lastSwitchTime = millis();
      }
      break;

    case SCAN_SBUS_NORMAL:
      if (sbus_rx.Read()) { currentProtocol = LOCKED_SBUS; lastValidFrameTime = millis(); }
      else if (millis() - lastSwitchTime > SCAN_TIMEOUT) {
        Serial1.end(); currentProtocol = SCAN_PPM; lastSwitchTime = millis();
      }
      break;

    case SCAN_PPM:
      if (ppm.latestValidChannelValue(1, 0) != 0) { currentProtocol = LOCKED_PPM; lastValidFrameTime = millis(); }
      else if (millis() - lastSwitchTime > SCAN_TIMEOUT) {
        currentProtocol = SCAN_PWM; lastSwitchTime = millis();
      }
      break;

    case SCAN_PWM:
      {
        uint32_t p = pulseIn(pwm_pins[0], HIGH, 30000);
        if (p > 800 && p < 2200) { currentProtocol = LOCKED_PWM; lastValidFrameTime = millis(); }
        else if (millis() - lastSwitchTime > SCAN_TIMEOUT) {
          Serial1.setInvertRX(true); Serial1.begin(100000, SERIAL_8E2);
          sbus_rx.Begin(); currentProtocol = SCAN_SBUS_INV; lastSwitchTime = millis();
        }
      }
      break;

    case LOCKED_SBUS:
      if (sbus_rx.Read()) {
        lastValidFrameTime = millis();
        sbusData = sbus_rx.data();
        failsafe = sbusData.failsafe;
        if (!failsafe) {
          for(int i=0; i<8; i++) reportData.axis[i] = map(constrain(sbusData.ch[i], 172, 1811), 172, 1811, -32768, 32767);
          reportData.buttons = 0;
          for (int b = 0; b < 8; b++) if (sbusData.ch[b + 8] > 1000) reportData.buttons |= (1 << b);
          if (hid.ready()) hid.sendReport(0, &reportData, sizeof(reportData));
        } 
      }
      if (millis() - lastValidFrameTime > 2000) { currentProtocol = SCAN_SBUS_INV; lastSwitchTime = millis(); }
      break;

    case LOCKED_PPM:
      if (ppm.latestValidChannelValue(1, 0) != 0) {
        lastValidFrameTime = millis();

        // --- read 8 first channels as axis ---
        for(int i=0; i<8; i++) reportData.axis[i] = map(ppm.latestValidChannelValue(i+1, 1500), 980, 2020, -32768, 32767);

        // --- Buttons  ---        
        reportData.buttons = 0;
        for (int b = 0; b < 8; b++) if (ppm.latestValidChannelValue(b + 9, 1500) > 1500) reportData.buttons |= (1 << b);
        if (hid.ready()) hid.sendReport(0, &reportData, sizeof(reportData));
      }
      if (millis() - lastValidFrameTime > 2000) { currentProtocol = SCAN_SBUS_INV; lastSwitchTime = millis(); }
      break;

    case LOCKED_PWM:
      {
        bool anySignal = false;
        for(int i=0; i<6; i++) {
          uint32_t val = pulseIn(pwm_pins[i], HIGH, 25000);
          if (val > 0) anySignal = true; else val = 1500;
          reportData.axis[i] = map(constrain(val, 980, 2020), 980, 2020, -32768, 32767);
        }
        
        reportData.axis[6] = 0; 
        reportData.axis[7] = 0; 
        reportData.buttons = 0;
        if (hid.ready()) hid.sendReport(0, &reportData, sizeof(reportData));
        if (anySignal) lastValidFrameTime = millis();
        if (millis() - lastValidFrameTime > 2000) { currentProtocol = SCAN_SBUS_INV; lastSwitchTime = millis(); }
      }
      break;
  } // end switch
} // end loop
