/* ====================================================================================================
 * AEROPIC's  ESP32-S3 super mini RC wireless JOYSTICK : AUTO-SENSING PPM / SBUS / SBUS-NON-INV
 * =====================================================================================================
 * V101
 * 
 * This is a smart radio controlled joystick. 
 * Not only it interfaces with a RX PPM or SBUS output but it senses the signal present on GPIO 13,
 * it recognizes the emitted protocol and adapt itself.
 * supported protocols are:
 * PPM (all polarities via ppmSpy).
 * SBUS Standard (Inverted).
 * SBUS TTL (Non-inverted).
 * It performs a cyclib search if nothing is output by the TX and works too if a RX protocol hot swap is performed.
 * 
 * uses USB HID 16 bits (for a maximal accuracy on the simulator).
 * 
 * hardware :
 * ========
 * solder a servo wire red to +5V, black to GND, white (SBUS signal) to GPIO 13 (right column just under 5V, GND, 3.3V)
 * (in case your RX has a 5V logic, just add a resistor in serial to the signal pin)
 * 
 * configuration
 * =============
 * add "https://espressif.github.io/arduino-esp32/package_esp32_index.json" in preferences/additionnal boards URL
 * in board manager type ESP32 and select ESP32 expressif 
 * in "tools/board" select "ESP32S3 DEV module"
 * USB Mode : "Hardware CDC and JTAG"
 * USB CDC On Boot : "Enabled"
 * in "tools manage libraries" include "USB", "USBHID", "sbus from Bolder flight", "ESP32_PPM" and "ADAFruit Neopixel"
 * 
 * compile and upload this code. To flash the binary, unplug USB, prss and hold BOOT button, plug USB then release BOOT
 * after code upload, press the reset button or switch OFF/ON the board 
 * 
 * user manual
 * ===========
 * the LED shows the status of communication:
 *  - flashing blue/purple/yellow  : searching protocol : no signal is received, either the RX is not connected or the radio is OFF
 *  - solid cyan : the PPM signal is received and valid
 *  - solid purple : SBUS signal is received and valid
 *  - solid red : the TX signal is lost, failsafe is active
 * 
 * to test the joystick on W10 or W11, just type "joy.cpl" in the search area. The device ESP32_DEV should appear,
 * press properties, the "test" tab is activated, move some axis
 * then go to parameters and calibrate the joystick (important as the SBUS range is set to +/-150%)
 *
 * version:
 * v101 : 251220 fixes the range of PPM signal
 * =============================================================================================================
 */

#include "USB.h"
#include "USBHID.h"
#include "sbus.h"               // https://github.com/bolderflight/sbus/tree/main
#include <ESP32_ppm.h>          //https://registry.platformio.org/libraries/fanfanlatulipe26/ESP32_ppm
#include <Adafruit_NeoPixel.h>

// --- Configuration LED RGB (WS2812 intégrée) ---
#define RGB_PIN 48
#define NUM_LEDS 1
Adafruit_NeoPixel rgb(NUM_LEDS, RGB_PIN, NEO_GRB + NEO_KHZ800);

// --- Descripteur HID (8 boutons + 8 axes 16 bits) ---
class CustomGamepad : public USBHIDDevice {
public:
  uint16_t _onGetDescriptor(uint8_t* dst) {
    uint8_t const desc[] = {
        0x05, 0x01,        // Usage Page (Generic Desktop)
        0x09, 0x05,        // Usage (Gamepad)
        0xA1, 0x01,        // Collection (Application)
      
        // --- 8 boutons ---
        0x05, 0x09,        // Usage Page (Button)
        0x19, 0x01,        // Usage Min
        0x29, 0x08,        // Usage Max
        0x15, 0x00,        // Logical Min
        0x25, 0x01,        // Logical Max
        0x75, 0x01,        // Report Size (1 bit)
        0x95, 0x08,        // Report Count (8 bits = 8 boutons)
        0x81, 0x02,        // Input (Data,Var,Abs)
      
        // --- 8 axes 16 bits ---
        0x05, 0x01,        // Usage Page (Generic Desktop)
        0x09, 0x30,        // X
        0x09, 0x31,        // Y
        0x09, 0x32,        // Z
        0x09, 0x33,        // Rx
        0x09, 0x34,        // Ry
        0x09, 0x35,        // Rz
        0x09, 0x36,        // Slider 1
        0x09, 0x37,        // Slider 2 
      
        0x16, 0x00, 0x80,  // Logical Min = -32768
        0x26, 0xFF, 0x7F,  // Logical Max = 32767
        0x75, 0x10,        // Report Size = 16 bits
        0x95, 0x08,        // Report Count = 8 axes
      
        0x81, 0x02,        // Input (Data,Var,Abs)
      
        0xC0               // End Collection
      };
    memcpy(dst, desc, sizeof(desc));
    return sizeof(desc);
  }
};

CustomGamepad MyGamepad;
USBHID HID;

// --- Structure du rapport USB (8 axes + 8 boutons) ---
#define ATTRIBUTE_PACKED __attribute__((packed, aligned(1)))
typedef struct ATTRIBUTE_PACKED {
    uint8_t buttons;
    int16_t axis[8];
} HID_GamepadReport_Data_t;

HID_GamepadReport_Data_t reportData;

// --- Machine à États ---
#define RX_PIN 13
enum State { SCAN_PPM, SCAN_SBUS_INV, SCAN_SBUS_TTL, ACTIVE_PPM, ACTIVE_SBUS };
State currentState = SCAN_PPM;

unsigned long lastValidSignal = 0;
unsigned long stateTimer = 0;

// Objets PPM
int *ppmArray = NULL;
ppmReader myPPM_RX;
ppmSpy myPPM_Spy;

// Objets SBUS - On crée deux instances pour gérer les deux types d'inversion
bfs::SbusRx sbus_inv(&Serial2, RX_PIN, -1, true); 
bfs::SbusRx sbus_ttl(&Serial2, RX_PIN, -1, false);
bfs::SbusData sbusData;

void setup() {
  Serial.begin(115200);
  
  // Initialisation LED RGB
  rgb.begin();
  rgb.setBrightness(10);
  
  // Initialisation USB HID (Le port série CDC est géré par l'option "USB CDC on Boot")
  HID.addDevice(&MyGamepad, 53);
  HID.begin();
  USB.begin();

  stateTimer = millis();
}

void loop() {
  bool newFrame = false;
  bool failsafe = false;

  switch (currentState) {
    
    case SCAN_PPM:
      rgb.setPixelColor(0, rgb.Color(0, 0, 255)); rgb.show(); // Bleu
      if (millis() - stateTimer > 1000) {
        result_ppmSpy_t *res = myPPM_Spy.begin(RX_PIN);
        myPPM_Spy.start(500);
        unsigned long spyWait = millis();
        while(!myPPM_Spy.doneSpy() && (millis() - spyWait < 600)) delay(1);
        
        if (res != NULL && res->maxChan >= 4) {
          ppmArray = myPPM_RX.begin(RX_PIN);
          myPPM_RX.start();
          currentState = ACTIVE_PPM;
          lastValidSignal = millis();
          Serial.println("Locked: PPM");
        }
        myPPM_Spy.end();
        if (currentState == SCAN_PPM) {
            currentState = SCAN_SBUS_INV;
            stateTimer = millis();
        }
      }
      break;

    case SCAN_SBUS_INV:
      rgb.setPixelColor(0, rgb.Color(255, 0, 255)); rgb.show(); // Magenta
      sbus_inv.Begin(); 
      stateTimer = millis();
      while(millis() - stateTimer < 600) {
        if (sbus_inv.Read()) {
          currentState = ACTIVE_SBUS;
          lastValidSignal = millis();
          Serial.println("Locked: SBUS Inverted");
          break;
        }
        delay(10);
      }
      if (currentState != ACTIVE_SBUS) {
        Serial2.end(); // Fermeture propre du port série
        currentState = SCAN_SBUS_TTL;
        stateTimer = millis();
      }
      break;

    case SCAN_SBUS_TTL:
      rgb.setPixelColor(0, rgb.Color(255, 255, 0)); rgb.show(); // Jaune
      sbus_ttl.Begin();
      stateTimer = millis();
      while(millis() - stateTimer < 600) {
        if (sbus_ttl.Read()) {
          currentState = ACTIVE_SBUS;
          lastValidSignal = millis();
          Serial.println("Locked: SBUS TTL");
          break;
        }
        delay(10);
      }
      if (currentState != ACTIVE_SBUS) {
        Serial2.end();
        currentState = SCAN_PPM; 
        stateTimer = millis();
      }
      break;

    case ACTIVE_PPM:
      if (myPPM_RX.newFrame()) {
        newFrame = true;
        lastValidSignal = millis();
        int nbCh = ppmArray[0];
        if (nbCh < 4 || ppmArray[1] < 500) failsafe = true;
        if (!failsafe) {
          for(int i=0; i<7; i++) {
              int val = (i < nbCh) ? ppmArray[i+1] : 1500;
              reportData.axis[i] = map(val, 980, 2020, -32768, 32767);
          }
          reportData.buttons = 0;
          for(int b=0; b<8; b++) if( (b+8 <= nbCh) && ppmArray[b+8] >= 1700) reportData.buttons |= (1 << b);
        }
      }
      if (millis() - lastValidSignal > 2000) {
        myPPM_RX.end(); 
        currentState = SCAN_PPM;
        stateTimer = millis();
      }
      break;

    case ACTIVE_SBUS:
      // On teste les deux instances car on ne sait pas laquelle a verrouillé
      if (sbus_inv.Read() || sbus_ttl.Read()) {
        newFrame = true;
        lastValidSignal = millis();
        // On récupère les données de l'instance qui a répondu
        sbusData = (currentState == ACTIVE_SBUS) ? (sbus_inv.Read() ? sbus_inv.data() : sbus_ttl.data()) : sbusData; 
        // Note: sbus_inv.data() et sbus_ttl.data() sont accessibles après un Read() réussi.
        
        // Plus simple : on utilise l'instance qui a réussi le Read
        if (sbus_inv.Read()) sbusData = sbus_inv.data();
        else if (sbus_ttl.Read()) sbusData = sbus_ttl.data();

        failsafe = sbusData.failsafe;
        if (!failsafe) {
          for(int i=0; i<7; i++) reportData.axis[i] = map(sbusData.ch[i], 0, 2047, -32768, 32767);
          reportData.buttons = 0;
          for(int b=0; b<8; b++) if(sbusData.ch[b+7] > 1000) reportData.buttons |= (1 << b);
        }
      }
      if (millis() - lastValidSignal > 2000) {
        Serial2.end();
        currentState = SCAN_PPM;
        stateTimer = millis();
      }
      break;
  }

  // --- Envoi HID et LED ---
  if (newFrame) {
    if (failsafe) {
      rgb.setPixelColor(0, rgb.Color(255, 0, 0)); // Rouge pour tous les failsafes
    } else {
      // Choix de la couleur selon le protocole verrouillé
      if (currentState == ACTIVE_PPM) {
        rgb.setPixelColor(0, rgb.Color(0, 255, 255)); // Cyan pour PPM
      } 
      else if (currentState == ACTIVE_SBUS) {
        // Optionnel : différencier SBUS Inv et TTL si besoin, 
        // ou une couleur commune pour le SBUS
        rgb.setPixelColor(0, rgb.Color(255, 0, 150)); // Rose pour SBUS
      }
    }
    rgb.show();
    if (!failsafe) HID.SendReport(0, &reportData, sizeof(reportData));
  }
}
