// Multisided dice generator
// David O'Connor  15 November 2025
// davidoconnor61@gmail.com

// Type of polyhedron (number of sides)
type = 10;  // [6, 8, 10, 12, 14, 20] 

// Depth of engraved characters
char_depth = 0.8; 

// Count by 10s (10, 20, etc.)
tens_multiplier = false; 

// Start at 1 instead of 0
start_at_1 = true;

// Adjustment factor from default font size
fontsize_factor = 1.01; 

/* [Advanced] */
// Convexity for polyhedron
c1 = 10; 

// Convexity for extruded text 
c2 = 10;

// Small offset to avoid z-fighting  
eps = 0.1; 

// Font for engraved numbers
fontname = "Arial:style=Bold"; 

// Other fonts to consider:
// fontname = "Onest:style=ExtraBold"; // Nice distinction between 1 and 7
// fontname = "Arial Rounded MT Bold:style=Bold";  // Good for 3d printing in general


///////////////////////////////////////////////////////////////////////////////

// Define the points (vertices) of the polyhedron

points = 

type == 6 ?
    15 * [
            [-1, -1, -1],
            [-1, 1, -1],
            [-1, 1, 1],
            [-1, -1, 1], 
            [ 1, -1, -1], 
            [ 1, 1, -1], 
            [ 1, 1, 1], 
            [ 1, -1, 1]
        ] :

type == 8 ?
    15 * [
            [1, 1, 0], 
            [-1, 1, 0], 
            [-1, -1, 0], 
            [1, -1, 0], 
            [0, 0, sqrt(2)], 
            [0, 0, -sqrt(2)]
        ] :

type == 10 ?  
    3.2859 * [
        [0,       0,       4.565],          //  0
        [3.61803, 2.62866, -0.481939],      //  1
        [1.38197 , 4.25325 , 0.48194],      //  2
        [-1.38197 , 4.25325 , -0.48194],    //  3
        [-3.61804 , 2.62866 , 0.481939],    //  4
        [-4.47214 , 0.000001 , -0.48194],   //  5
        [-3.61803 , -2.62866 , 0.48194],    //  6
        [-1.38197, -4.25325 , -0.481939 ],  //  7
        [1.38197 , -4.25325, 0.48194 ],     //  8
        [3.61803 , -2.62866 , -0.48194 ],   //  9
        [4.47214 , 0, 0.48194 ],            // 10
        [0,        0,       -4.565]         // 11
    ] :

type == 12 ? 
    let (phi = (1 + sqrt(5)) / 2, r = 1 / phi)
    9.3*[
        // 8 cube-like vertices
        [-1, -1, -1], // 0
        [-1, -1,  1], // 1
        [-1,  1, -1], // 2
        [-1,  1,  1], // 3
        [ 1, -1, -1], // 4
        [ 1, -1,  1], // 5
        [ 1,  1, -1], // 6
        [ 1,  1,  1], // 7

        // 12 vertices with a zero coordinate and (±r, ±phi) on the other two
        [ 0, -r, -phi], // 8
        [ 0, -r,  phi], // 9
        [ 0,  r, -phi], // 10
        [ 0,  r,  phi], // 11

        [-r, -phi, 0], // 12
        [-r,  phi, 0], // 13
        [ r, -phi, 0], // 14
        [ r,  phi, 0], // 15

        [-phi, 0, -r], // 16
        [ phi, 0, -r], // 17
        [-phi, 0,  r], // 18
        [ phi, 0,  r]  // 19
    ] :

type == 14 ?
    // Choose parameters a and b to give equal areas on each face
    let(a=0.5, b=0.375 )
    30 * [
        [a, -b, 0],  //  0
        [a, 0, -b],  //  1
        [a, b, 0],   //  2
        [a, 0, b],   //  3
        [-a, 0, b],  //  4
        [-a, b, 0],  //  5
        [-a, 0, -b], //  6
        [-a, -b, 0], //  7
        [0, a, b],   //  8
        [b, a, 0],   //  9
        [0, a, -b],  // 10
        [-b, a, 0],  // 11
        [0, -a, b],  // 12
        [-b, -a, 0], // 13
        [0, -a, -b], // 14
        [b, -a, 0],  // 15
        [-b, 0, a],  // 16
        [0, -b, a],  // 17
        [b, 0, a],   // 18
        [0, b, a],   // 19
        [0, -b, -a], // 20
        [-b, 0, -a], // 21   
        [0, b, -a],  // 22
        [b, 0, -a]   // 23
    ] :

type == 20 ?
    let (phi = (1 + sqrt(5)) / 2)
    9.27 *[
            [-1,  phi,  0],
            [ 1,  phi,  0],
            [-1, -phi,  0],
            [ 1, -phi,  0],
            [ 0, -1,  phi],
            [ 0,  1,  phi],
            [ 0, -1, -phi],
            [ 0,  1, -phi],
            [ phi,  0, -1],
            [ phi,  0,  1],
            [-phi,  0, -1],
            [-phi,  0,  1]
    ] :
0;

// Define the faces of the polyhedron   

faces = 

type == 6 ? 
    [ 
        [3, 2, 1, 0], 
        [4, 5, 6, 7], 
        [0, 1, 5, 4], 
        [1, 2, 6, 5], 
        [2, 3, 7, 6], 
        [3, 0, 4, 7]
    ] :

type == 8 ? [ 
                [0, 1, 4], 
                [1, 2, 4], 
                [2, 3, 4], 
                [3, 0, 4], 
                [1, 0, 5], 
                [2, 1, 5], 
                [3, 2, 5], 
                [0, 3, 5]
            ] :

type == 10 ?
            [
                [0, 2, 3, 4],  // 0
                [0, 4, 5, 6],  // 1
                [0, 6, 7, 8],  // 2
                [0, 8, 9, 10], // 3
                [0, 10, 1, 2], // 4
                [11, 3, 2, 1], // 5
                [11, 5, 4, 3], // 6
                [11, 7, 6, 5], // 7
                [11, 9, 8, 7], // 8 
                [11, 1, 10, 9] // 9
            ] :

type == 12 ?
[
        [0,  16, 2, 10, 8],
        [4,  8, 10, 6, 17],
        [1,  9, 11, 3, 18],
        [5, 19, 7, 11, 9],

        [0, 8, 4, 14, 12],
        [2, 13, 15, 6, 10],
        [1, 12, 14, 5,  9],
        [3, 11, 7, 15, 13],

        [0, 12, 1, 18, 16],
        [4, 17, 19, 5, 14],
        [2, 16, 18, 3, 13],
        [6, 15, 7, 19, 17]
    ] :

type == 14 ? 
    [
        [16, 17, 18, 19],
        [17, 12, 15, 0, 3, 18],
        [12, 13, 14, 15],
        [16, 4, 7, 13, 12, 17 ],
        [0, 15, 14, 20, 23, 1],
        [18, 3, 2, 9, 8, 19],
        [13, 7, 6, 21, 20, 14],
        [3, 0, 1, 2],
        [8, 9, 10, 11],
        [9, 2, 1, 23, 22, 10],
        [20, 21, 22, 23],
        [4, 5, 6, 7],
        [19, 8, 11, 5, 4, 16],
        [5, 11, 10, 22, 21, 6]
    ] :
 
type == 20 ? 
    [
      [0, 11, 5],
      [0, 5, 1],
      [0, 1, 7],
      [0, 7, 10],
      [0, 10, 11],
      [1, 5, 9],
      [5, 11, 4],
      [11, 10, 2],
      [10, 7, 6],
      [7, 1, 8],
      [3, 9, 4],
      [3, 4, 2],
      [3, 2, 6],
      [3, 6, 8],
      [3, 8, 9],
      [4, 9, 5],
      [2, 4, 11],
      [6, 2, 10],
      [8, 6, 7],
      [9, 8, 1]
    ] :
0;

// Define the display numbers for each face

display_nums = 

type == 6 ?
    [1, 6, 3, 2, 4, 5] :

type == 8 ?
    [1, 7, 3, 5, 6, 4, 8, 2] :

type == 10 ?
    [1, 7, 3, 9, 5, 8, 2, 6, 10, 4] :

type == 12 ?
    [1, 8, 4, 11, 5, 10, 2, 9, 6, 3, 12, 7] :

type == 14 ?
    [1, 11, 4, 14, 6, 9, 2, 13, 5, 10, 3, 12, 7, 8] :

type == 20 ?
    [1, 13, 4, 16, 6, 11, 3, 18, 7, 10, 2, 15, 5, 14, 8, 19, 9, 12, 20, 17] :
    
0;

// Adjust whether numbering starts at 0 or 1
dn1 = 
    start_at_1 ? 
        display_nums :
        [ for (n = display_nums) n - 1 ];   

// Adjust for tens multiplier if needed
dn2 = 
    tens_multiplier ? 
        [ for (n = dn1) n * 10 ] :
        dn1;

echo("Display numbers: ", dn2);

// Determine appropriate font size

// Scale font size and adjust further if counting by tens
fontsize = fontsize_factor * (tens_multiplier ? 0.66 : 1) * (
type == 6 ? 9 :
type == 8 ? 10 :
type == 10 ? 6 :
type == 12 ? 7 :
type == 14 ? 6 :
type == 20 ? 5 :
0);


///////////////////////////////////////////////////////////////////////////////

// Determine maximum dimension of the polyhedron for reference
boxMax = maxVal(points);
boxMin = minVal(points);
boxSize = boxMax - boxMin;
maxDim = max(boxSize);
echo("Maximum dimension: ", maxDim);

render(convexity = c1)
difference()  {
    polyhedron(points, faces, convexity=c1);
    {
        for (i = [0:len(faces)-1]) {
            f = faces[i];
            verts = [ for (j = [0:len(f)-1]) points[f[j]] ];

            // centroid: average of all face vertices
            centroid = [
                sum([ for (v = verts) v[0] ]) / len(verts),
                sum([ for (v = verts) v[1] ]) / len(verts),
                sum([ for (v = verts) v[2] ]) / len(verts)
            ];

            // face normal (use first three vertices to form two edges)
            p0 = verts[0]; p1 = verts[1]; p2 = verts[2];
            normal = unit(cross(sub(p1,p0), sub(p2,p0)));
            translate(centroid) {
                rotate_to(normal)  {
                    translate([0, 0, -char_depth]) {
                        linear_extrude(char_depth+eps, convexity=c2) {
                            draw_char = dn2[i];
                            translate( draw_char == 6 || draw_char == 9 ? [0, 0.2*fontsize] : [0, 0]) {
                                text(str(draw_char), size=fontsize, font=fontname, halign="center", valign="center");
                                // Draw underscore for 6 and 9
                                if (draw_char == 6 || draw_char == 9) {
                                    translate([0, -0.65*fontsize])
                                    text("_", size=fontsize, font=fontname, halign="center", valign="center");
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}


///////////////////////////////////////////////////////////////////////////////

// A recursive function for calculating the sum of a list
function sum(list, c = 0) = 
    c < len(list) - 1 ? 
        list[c] + sum(list, c + 1) :
        list[c];
    
// Various vector helper functions
function dot(a,b) = a[0]*b[0] + a[1]*b[1] + a[2]*b[2];
function vlen(v) = sqrt(v[0]*v[0] + v[1]*v[1] + v[2]*v[2]);
function sub(a,b) = [a[0]-b[0], a[1]-b[1], a[2]-b[2]];
function add(a,b) = [a[0]+b[0], a[1]+b[1], a[2]+b[2]];
function mul(v,s) = [v[0]*s, v[1]*s, v[2]*s];
function unit(v) = let(l = vlen(v)) (l==0 ? [0,0,1] : [v[0]/l, v[1]/l, v[2]/l]);
function offset(a, v) = [for (i = [0:len(v)-1]) a + v[i] ];

// Rotates the local coordinate system so that the +Z axis 
//  aligns with the given vector n
module rotate_to(n) {
    if (n.x == 0 && n.y == 0 && n.z == -1) 
        rotate(a=180, v=[1,0,0]) { children(); }
    else {
        up = [0, 0, 1];
        rot_axis = unit(cross(n, up));
        cos_ang = dot(n, up);
        sin_ang = vlen(cross(n, up));
        angle = -atan2(sin_ang, cos_ang);
        rotate(a=angle, v=rot_axis) { children(); }
    }
}

// Functions for determining the bounding box of a polyhedron

function minVal(shapeOuter) =
    [min([ for( v = shapeOuter ) v.x ]),
     min([ for( v = shapeOuter ) v.y ]), 
     min([ for( v = shapeOuter ) v.z ])];

function maxVal(shapeOuter) =
    [max([ for( v = shapeOuter ) v.x ]),
     max([ for( v = shapeOuter ) v.y ]),
     max([ for( v = shapeOuter ) v.z ])];

// Draws a red axis along the given vector for debugging
module draw_axis(vector) {
    $fn = 32;
    color("red") {
        cylinder(10.1, 1, 1);
        translate([0, 0, 10]) cylinder(h = 4, r1 = 2, r2 = eps);
    }
}
