from math import sin, pi, cos, pow, sqrt, exp
from vector import Vector
from triangulate import ears
import struct


def connect(p1, p2):
    """Simplest way to generate facets connecting two perimeters
    with same number of points, and similar positions"""
    p11=p1[-1]
    p21=p2[-1]
    for p12,p22 in zip(p1,p2):
        if p11!=p22 and p11!=p21 and p22!=p21:
            yield [p11,p22,p21]
        if p11!=p12 and p11!=p22 and p12!=p22:
            yield [p11,p12,p22]
        p11=p12
        p21=p22


def morph(a, b, alpha):
    """Return linear interpolation of items in a and b"""
    for k,l in zip(a,b):
        if isinstance(k, list):
            yield list(morph(k,l,alpha))
        else:
            yield k+(l-k)*alpha

def writestl(facets,name="out.stl"):
    """Write a binary stl file"""
    f = open(name,'wb')
    
    f.write(struct.pack('x'*80+'I',len(facets)))
    
    for p1,p2,p3 in facets:
        n = (p2-p1).cross(p3-p1).normalize()
        f.write(struct.pack('fff', n.x,n.y,n.z))
        for b in [p1,p2,p3]:
            f.write(struct.pack('fff', b.x, b.y, b.z))
        f.write(struct.pack('H',0))
    f.close()

def turtle(turnstep, p0 = Vector(0,0)):
    # Generate points from list of [turn,step].
    p = p0
    d = 0
    for turn,step in turnstep:
        yield p
        d += turn
        p += Vector(step,0).rotate(d)

def sigmoid(x):
    # Sigmoid function
    return 1./(1.+exp(-x))
    
def center(poly):
    # Center polygon on x-y axis.
    sum = Vector(0,0)
    for p in poly:
        sum += p
    sum.z=0
    return [p-sum/len(poly) for p in poly]

####### Shapes ##########

def kochify(l,f=1):
    # One iteration of koch with adjustable angle factor f.
    for turn,step in l:
        yield [turn,step]
        # cos(a)=1/2
        # f=1 => l=step, a=60, h=step*sin(60)=step*sqrt(3)/2
        # f=0 => l=step/2, a=0, h=0
        # a=f*60
        # l=step/2/cos(a)
        
        a=f*pi/3.
        l=step/2./cos(a)
        yield [-a,l]
        yield [2.*a,l]
        yield [-a,step]

def koch(iterampl=[1,1,1], l=2.5):
    # Triangular koch (snowflake) with adjustable amplitude per iteration.
    k = [[pi/3,l],
         [2*pi/3,l],
         [2*pi/3,l]]
    for f in iterampl:
        k = kochify(k,f)
    return k

def kochnum(i):
    # Returns the number of points for a given number of iterations of koch.
    return int(pow(4,i)*3)

def square(num, l=1):
    # Square shape
    square=[]
    for i in range(4):
        n = int(num*(i+1)/4.-len(square))
        square += [[pi/2,l]]
        square += [[0,l]]*(n-1)
    return square

def circle(n,l=0.8):
    #   Circle shape
    return [[pi*2/n,l]]*n

def plus(n,l=1):
    # Plus shape
    return [[0,l]*(n/8-1),[pi/2,l],[0,l]*(n/8-1),[-pi/2,l],[0,l]*(n/8-1),[pi/2,l]]*4

def gear(n,l=1,l2=1,d=1,t=90):
    # Generate a gear shape
    dt = pi/2/n
    rt = pi/180*t
    return [[rt+dt,d],[-rt+dt,l],[-rt+dt,d],[rt+dt,l2]]*n

def bevel(poly, num=2, l=0.5):
    # For each turn, generate a curve with num steps of length l.
    out = []
    for t,s in poly:
        out += [[t/num,l]]*(num-1)
        out += [[t/num,s]]
    return out

def triangulate(poly):
    mesh = []
    t = ears(list((p.x,p.y) for p in poly))
    for tri in t:
        mesh.append([Vector(p[0],p[1],poly[0].z) for p in tri])
    return mesh
            
def gen(num = 2, polygen = turtle(circle(10)), name = "out.stl", top = False):
    # Generate a solid from a polygon generator.
    print 'Generating ' + name
    mesh = []

    for i in range(num+1):
        x = float(i)/num
        poly = polygen(x)
        if i==0:
            # Bottom surface (flip orientation of triangles)
            mesh += [reversed(p) for p in triangulate(poly)]
        else:
            # Perimeter
            mesh += connect(polyb,poly)
        polyb = poly

    if top:
        mesh += triangulate(poly)

    print 'Storing'
    writestl(mesh, name)
    print 'Done'

def ckoch1(i):
    # Circular bottom koch vase
    s = 1 #0.9+0.6*sigmoid(i/4.-5) #(1+sin(pi/2*i/15))
    z = i*100.
    a = pi/3*i
    m = sigmoid(i*20./3.-3.)
    k=koch([1.1*sin(i*2/3*pi)]*3)
    c=circle(kochnum(3),0.8)
    poly = [((p*s)+Vector(0,0,z)).rotate(a) for p in turtle(morph(c,k,m))]
    return center(poly)

def ckoch2(i):
    # Circular bottom koch vase.
    s = 0.9+0.4*sigmoid(10*i-5)
    z = i*150.
    a = pi/3*i
    m = sigmoid(i*20/3-3)
    k = koch([1,1,1])
    c=circle(kochnum(3))
    poly = [((p*s)+Vector(0,0,z)).rotate(a) for p in turtle(morph(c,k,m))]
    return center(poly)

def trikoch(i):
    # Triangluar bottom koch vase
    s = 0.9+0.6*sigmoid(10*i-5)
    z = i*100.
    a = pi/3*i
    k = koch([i,i,i])
    poly = [((p*s)+Vector(0,0,z)).rotate(a) for p in turtle(k)]
    return center(poly)

def squircle(i):
    # Square to circle
    s = 0.9+0.6*sigmoid(10*i-5)
    z = i*100.
    a = 0
    m = sigmoid((i-0.5)*8)
    c = circle(20,8)
    sq = square(20,8)
    poly = [((p*s)+Vector(0,0,z)).rotate(a) for p in turtle(morph(sq,c,m))]
    return center(poly)

def gearvase(i):
    # Gear type vase
    z = Vector(0,0,i*100.)
    a = pi/2*i
    f = sin(pi*i)+pow(i,8)
    t = bevel(gear(16,2,2+6*f,1),5,0.5)
    poly = [(p+z).rotate(a) for p in turtle(t)]
    return center(poly)

def stripe1(i):
    z = Vector(0,0,i*100.)
    s = 1.
    a = pi/2*i*2
    f = sin(pi*i)
#    t = gear(16,2,2+6*f,1+3*f,110*pow(f,1./2.))
    t = bevel(gear(16,2,2+6*f,1+5*f,110*pow(f,.5)),5,0.5)
#    t = bevel(gear(16,2+8*f,2+6*f,1+8*f,130*pow(f,.5)),5,0.5)
#    t = bevel(gear(4, 10+30*(1-i), 20+10*f, 5+20*i, 0+90*i), 5, 0.5)
    poly = [(p*s+z).rotate(a) for p in turtle(t)]
    return center(poly)

def stripe2(i):
    z = Vector(0,0,i*120.)
    a = pi/2*i*1.5*0
    f = sin(pi*i)
    v = 8*i*i
    t = bevel(gear(7,8,4+30*f*f+v,2+10*f+v,110*pow(0.01+f*0.99,.5)),5,0.7)
    poly = [(p+z).rotate(a) for p in turtle(t)]
    return center(poly)

def stripe3(i):
    z = Vector(0,0,i*120.)
    a = pi*i*0
    f = sin(pi*i)
    v = 3*pow(i,8)
    t = bevel(gear(16,1+5*f*f+v,1+14*pow(f,1.5)+v,1+2*f,130*pow(f,.5)),5,0.4)
    poly = [(p+z).rotate(a) for p in turtle(t)]
    return center(poly)

def stripe4(i):
    z = Vector(0,0,i*100.)
    s = 1.
    a = pi/2*i
    f = sin(pi*i)
    t = bevel(gear(5, 1+30*i*i, 1+20*f, 1+20*(1-i), 70*(1-pow(i,3))), 10, 1)
    poly = [(p*s+z).rotate(a) for p in turtle(t)]
    return center(poly)

def classic(i):
    z = Vector(0,0,i*100.)
    s = 0.7*(1.-i)+1.3*pow(i,3) # 2.+sin(pi*pow(i,1.5)*2)
    k=bevel(koch([1,1],5),4,1)
    t = circle(kochnum(2)*4,s)
    m = sin(i*pi)
    poly = [(p+z) for p in morph(turtle(t),turtle(k),m)]
    return center(poly)

def ribbon(i):
    z = Vector(0,0,i*100.)
    f = 150*sin(pi*i*2)
    sf = cos(f*pi/180.)
    d=4
    t = bevel(gear(8,8-d*sf+10*i*i,8-d*sf,d,f),5,0.5)
    poly = [p+z for p in turtle(t)]
    return center(poly)

def ribbon2(i):
    z = Vector(0,0,i*100.)
    a = pi/2*i
    #f = (pow(i,2)-pow(i,3))*27./4.
    #f'=2i-3i^2=0 => 2-3i=0 => i=2/3 => f=4/9-8/27=4/27
    f = 150*sin(pi*i*2)
    sf = cos(f*pi/180.)
    d=7
    t = bevel(gear(8,8-d*sf+10*i*i,8-d*sf,d,f),5,0.5)
    poly = [(p+z).rotate(a) for p in turtle(t)]
    return center(poly)

if __name__ == "__main__":
    gen(100,ckoch1,'ckoch1.stl')
    gen(100,ckoch2,'ckoch2.stl')
    gen(100,squircle,'squircle.stl')
    gen(100,trikoch,'trikoch.stl')
    gen(100,ribbon,'ribbon.stl')
    gen(100,ribbon2,'ribbon2.stl')
    gen(100,stripe1,'stripe1.stl')
    gen(100,stripe2,'stripe2.stl')
    gen(100,stripe3,'stripe3.stl')
    gen(100,stripe4,'stripe4.stl')
    gen(100,classic,'classic.stl')
    gen(100,gearvase,'gear.stl')
