use <MCAD/boxes.scad>

$fa = 1;
$fs = 0.1;

ol = 0.1;
olt = ol/2;

// The hinge
// do            - outer diameter
// di            - diameter of inner rod
// stops         - the number of parts that hold the rod (as compared to "gaps": the parts that move around the rod)
//                 there's alway one gap less than stops
//                 minimum number is 2 (obiously)
//                 Sorry for the bad explanation. Start with default an increase number of stops. That'll show it.
// slack         - clearance between stops and gaps. Also clearance for negative mask
// with_latches  - attach some primitive cubes to which you can attach the rest of your design.
//                 Setting in to false leaves just a cylinder with turning parts
// with_stops    - render the stops
// with_gaps     - render the gaps
// negative_mask - show a simplified version, slightly bigger than the actual hinge.
//                 Used to substract some slack from the parts that you attach to the hinge so they can actually move.

module hinge(do, di, stop_width, gap_width, stops = 2, slack = 0.8, with_latches=true, with_stops=true, with_gaps=true, negative_mask=false) {
    assert(stops >= 2, "You cannot have less than two steps in a hinge");
    complete_width = stops * stop_width + (stops - 1) * gap_width;
    mask_slack = negative_mask ? slack : 0;
    if (with_stops) {
        //stops
        translate([0,stop_width/2-complete_width/2,0]) {
            for (i = [1:1:stops]) {
                translate([0,(i-1)*(stop_width+gap_width),0]) {
                    rotate([90, 0, 0]) {
                        cylinder(d = do+mask_slack, h=stop_width+mask_slack, center=true);
                        if (with_latches) {
                            translate([-do/4, -do/4, 0]) {
                                cube([do/2, do/2, stop_width], center=true);
                            }
                        }
                    }
                }
            }
        }
        rotate([90,0,0]) {
            cylinder(d = di, h = complete_width-ol, center=true);
        }
    }

    if (with_gaps) {
        //gaps
        translate([0,gap_width/2-complete_width/2+stop_width,0]) {
            for (i=[1:1:stops-1]) {
                translate([0,(i-1)*(stop_width+gap_width),0]) {
                    rotate([90,0,0]) {
                        difference() {
                            union() {
                                cylinder(d=do+mask_slack, h=gap_width-slack+mask_slack, center=true);
                                if (with_latches) {
                                    translate([do/4, -do/4, 0]) {
                                        cube([do/2, do/2, gap_width-slack], center=true);
                                    }
                                }
                            }
                            if (!negative_mask) {
                                cylinder(d=di+slack, h=gap_width-slack+ol, center=true);
                            }
                        }
                    }
                }
            }
        }
    }
}

// Half a suitcase shell
// size            - vector
// radius          - radius of corners
// wall_thickness  - thickness of the walls (duh)
// handle size     - vector. leave empty if you don't want a handle
// strap_thickness - thickness of outer straps. Leave empty if you don't want straps

module suitcase_half(size, radius, wall_thickness=1.0, handle_size, strap_thickness) {
    translate([radius/2,0,0]) {
        difference() {
            union() {
                roundedCube([size[0]+radius, size[1], size[2]], r=radius, sidesonly=false, center=true);
                //straps
                if (!is_undef(strap_thickness) && strap_thickness > 0) {
                    translate([0,size[1]*0.3,0]) {
                        rotate([90,0,0]) {
                            roundedCube([size[0]+radius+2*strap_thickness, size[2], size[1]/10], r=1, sidesonly=true, center=true);
                        }
                    }
                    translate([0,-size[1]*0.3,0]) {
                        rotate([90,0,0]) {
                            roundedCube([size[0]+radius+2*strap_thickness, size[2], size[1]/10], r=1, sidesonly=true, center=true);
                        }
                    }
                }
            }
            roundedCube([size[0]+radius-2*wall_thickness, size[1]-2*wall_thickness, size[2]-2*wall_thickness], r=radius-wall_thickness, sidesonly=false, center=true);
            strap = is_undef(strap_thickness) ? 0 : strap_thickness;
            translate([size[0]/2+olt+strap/2,0,0]) {
                cube([radius+ol+strap, size[1]+ol, size[2]+ol], center=true);
            }
        }
    }
    //handle
    if (!is_undef(handle_size)) {
        translate([size[0]/2-(size[0]-radius)/2,0,size[2]/2+handle_size[2]/2-olt-0.5/2]) {
            difference() {
                roundedCube([handle_size[0], handle_size[1], handle_size[2]+0.5], r= 0.5, sidesonly= false, center=true);
                translate([0,0,-0.5-olt]) {
                    roundedCube([handle_size[0]+2*0.5, handle_size[1]-2, handle_size[2]-1+ol], r=0.2, sidesonly=false, center=true);
                }
            }
        }
    }
}

//suitcase size
ssize = [8,33,24];




intersection() {
    hinge (do = 4, di = 1.5, stop_width = 5, gap_width = 9, stops = 3, with_latches=true);
    //round outer edged so they match the suitcase
    translate([0,0,ssize[2]/2-3/2]) {
        roundedCube([2*ssize[0],ssize[1],ssize[2]], r=2, sidesonly=false, center=true);
    }
}


difference() {
    translate([-(ssize[0]/2+olt)+1,0,ssize[2]/2-3/2]) {
        suitcase_half(ssize, radius=2, handle_size=[2, 8, 4], strap_thickness=0.8);
    }
    hinge (do = 4, di = 1.5, stop_width = 5, gap_width = 9, stops = 3, with_latches=true, with_stops=false, negative_mask=true);
}

difference() {
    scale([-1,1,1]) { // mirror
        translate([-(ssize[0]/2+olt)-1,0,ssize[2]/2-3/2]) {
            suitcase_half(ssize, radius=2, strap_thickness=0.8);
        }
    }
    hinge (do = 4, di = 1.5, stop_width = 5, gap_width = 9, stops = 3, with_latches=true, with_gaps=false, negative_mask=true);
}
