#include <EEPROM.h>
#define SETTINGS_VERSION 0x08
#define SLOT_COUNT 10   // number of slots (spread wear over these)


// Libraries: FastLED
#define FASTLED_ALLOW_INTERRUPTS 1
#include <FastLED.h>

#include <IRremote.h>
// Choose your IR pin (must support external interrupts on Nano Every)
#define IR_RECEIVE_PIN 2  


// ---------------- USER SETTINGS ----------------
//#define DATA_PIN       6
#define NUM_LEDS       48
#define LED_TYPE       WS2812   // WS2815 uses WS2812 timing
#define COLOR_ORDER    GRB
#define START_BRIGHT   180       // 0–255

// === Data pin slots (compile-time constants for FastLED templates)
#define PIN0 6

// How many of the above are you using?
const uint8_t PIN_SLOTS = 1;  // e.g., using PIN0..PIN3

// ------------------------------------------------

// ---------- IR BUTTON MAP ----------
#define IR_ADDR 0xFF00

#define IR_BTN_POWER_ON   0x40
#define IR_BTN_PLAY   0x41
#define IR_BTN_BRIGTHNESS_UP   0x5C
#define IR_BTN_BRIGTHNESS_DOWN   0x5d
#define IR_BTN_FASTER   0x17
#define IR_BTN_SLOWER   0x13
#define IR_BTN_COLOR_RED_UP   0x14
#define IR_BTN_COLOR_RED_DOWN   0x10
#define IR_BTN_COLOR_GREEN_UP   0x15
#define IR_BTN_COLOR_GREEN_DOWN   0x11
#define IR_BTN_COLOR_BLUE_UP   0x16
#define IR_BTN_COLOR_BLUE_DOWN   0x12

#define IR_BTN_DIY1   0xC
#define IR_BTN_DIY2   0xD
#define IR_BTN_DIY3   0xE
#define IR_BTN_DIY4   0x8
#define IR_BTN_DIY5   0x9
#define IR_BTN_DIY6   0xA

#define IR_BTN_MODE_AUTO   0xF // USED AS MODE CHANGER
#define IR_BTN_MODE_FLASH   0xB
#define IR_BTN_MODE_JUMP3  0x4
#define IR_BTN_MODE_JUMP7  0x5
#define IR_BTN_MODE_FADE3   0x6
#define IR_BTN_MODE_FADE7   0x7

// Color Buttons
// Row 1
#define IR_BTN_RED         0x58
#define IR_BTN_GREEN       0x59
#define IR_BTN_BLUE        0x45
#define IR_BTN_WHITE       0x44

// Row 2
#define IR_BTN_DARKORANGE      0x54
#define IR_BTN_LIGHTGREEN  0x55
#define IR_BTN_LIGHTBLUE   0x49
#define IR_BTN_PINK        0x48

// Row 3
#define IR_BTN_ORANGE  0x50
#define IR_BTN_CYAN        0x51
#define IR_BTN_DARKPURPLE      0x4D
#define IR_BTN_LIGHTPINK   0x4C

// Row 4
#define IR_BTN_YELLOWORANGE 0x1C
#define IR_BTN_TEAL        0x1D
#define IR_BTN_PURPLE  0x1E
#define IR_BTN_SKYBLUE     0x1F

// Row 5
#define IR_BTN_YELLOW      0x18
#define IR_BTN_AZURE       0x19
#define IR_BTN_MAGENTA     0x1A
#define IR_BTN_LIGHTCYAN   0x1B

// ------------------------------------------------------------------------------


// Your persistent settings
struct Settings {
  uint8_t version;
  uint16_t sequence;   // increments each save
  CRGB diyColors[6];
  CRGB solidColor;
  uint8_t mode;
  uint8_t brightness;
  uint16_t speedMs;
  uint8_t  diyIndex;     // NEW: active DIY slot (0..5)
  uint8_t  powerOn;       // NEW: 1 = on, 0 = off
};

Settings settings;

const int SLOT_SIZE = sizeof(Settings);
uint32_t lastChangeMs = 0;
bool needsSave = false;

// ----- Modes -----
enum Mode : uint8_t {
  MODE_RAINBOW = 0,
  MODE_FADE    = 1,
  MODE_SWITCH  = 2,
  MODE_FLASH   = 3,
  MODE_DOT     = 4,
  MODE_SNAKE   = 5,
  MODE_COLOR   = 6,
  MODE_NOISE   = 7, 
  MODE_STARS   = 8,
  MODE_STARS_COMETS = 9,
  MODE_METEOR = 10,
  MODE_BOUNCE = 11,
  MODE_FIREFLIES = 12,
  MODE_MATRIX = 13,
  MODE_AURORA = 14,
  MODE_SPARKLES = 15,
  MODE_CONFETTI = 16,
  MODE_JUGGLE = 17,
  MODE_RIPPLE = 18,
  MODE_PLASMA = 19,
  MODE_PLASMARAINBOW = 20,
  MODE_PLASMACLOUD = 21,
  MODE_PLASMAPARTY = 22,
  MODE_PLASMAFOREST = 23,
  MODE_RAINBOW2 = 24,
  MODE_DIY     = 25,
  MODE_COUNT   = 26
  
};

Mode mode = MODE_STARS_COMETS;

//CRGB leds[NUM_LEDS];

struct Field {
  uint8_t  slot;        // which data pin (0..PIN_SLOTS-1)
  uint16_t start;       // start index within that slot’s strip
  uint16_t count;       // LEDs in this field
  uint16_t logicalZero; // which LED inside this field is the "0" for visuals
  int8_t   dir;         // +1 forward, -1 reverse
};

// EDIT these to your install (examples):
Field fields[] = {
     {0,   0,  24, 0, +1},/* Unten  */
   {0,   24,  24, 0, +1}, /* Oben  */
 
 };
const uint8_t FIELD_COUNT = sizeof(fields) / sizeof(fields[0]);

// Visual travel order across fields:
uint8_t fieldOrder[] = { 0,1 };


// Safe helpers (avoid macro max/min pitfalls)
template <typename T> inline T vmax(T a, T b) { return (a > b) ? a : b; }
template <typename T> inline T vmin(T a, T b) { return (a < b) ? a : b; }

// ----- Animation tuning -----
uint8_t  currentBrightness = START_BRIGHT;
bool     isOn = true;

// "Speed" = ms per logical step; lower = faster.
uint16_t speedMs = 20;
const uint16_t SPEED_MIN = 2;
const uint16_t SPEED_MAX = 200;

// Color Switching palette & timing
CRGB solidColor = CRGB::Black;

// Global timers
uint32_t lastStepMs = 0;
uint32_t lastSwitchMs = 0;

// 1) Per-slot totals and base offsets
uint16_t slotCount[6]   = {0};   // LEDs per slot
uint16_t slotBase[6]    = {0};   // starting offset in the global buffer

// 2) Global buffer (FastLED drives this)
const uint16_t TOTAL_LEDS_MAX = NUM_LEDS; // set a sane upper bound for safety
CRGB leds2[TOTAL_LEDS_MAX];
uint16_t TOTAL_LEDS = 0;

// 3) Logical → physical map
uint16_t path[TOTAL_LEDS_MAX];
uint16_t pathLen = 0;

static inline uint16_t wrapU(int32_t x, uint16_t m) {
  int32_t r = x % (int32_t)m; return (r < 0) ? (r + m) : (uint16_t)r;
}


struct LedView {
  CRGB* phys; uint16_t* map; uint16_t len;
  inline CRGB& operator[](uint16_t i)             { return phys[ map[i % len] ]; }
  inline const CRGB& operator[](uint16_t i) const { return phys[ map[i % len] ]; }
} V;

void initView() { V = { leds2, path, pathLen }; }

inline uint16_t NLED() { return V.len ? V.len : NUM_LEDS; }   // logical length
inline CRGB&    P(uint16_t i) { return V[i % NLED()]; }       // mapped pixel

inline void fillV(const CRGB& c) {
  const uint16_t n = NLED();
  for (uint16_t i = 0; i < n; ++i) P(i) = c;
}

inline void fadeV(uint8_t amount){
  const uint16_t n = NLED();
  const uint8_t scale = 255 - amount;   // same semantics as fadeToBlackBy(...)
  for (uint16_t i = 0; i < n; ++i) {
    CRGB &px = P(i);    // modify in-place
    px.nscale8(scale);  // NOT _video → will eventually reach 0 (black)
  }
}

// put near your other helpers
inline void clearAllPhysical() {
  // clears leds[0..TOTAL_LEDS)
  for (uint16_t i = 0; i < TOTAL_LEDS; ++i) leds2[i] = CRGB::Black;
}

void computeSlotTotals() {
  memset(slotCount, 0, sizeof(slotCount));
  for (uint8_t f=0; f<FIELD_COUNT; ++f) {
    slotCount[ fields[f].slot ] =
      (uint16_t)max<uint32_t>(slotCount[fields[f].slot], fields[f].start + fields[f].count);
  }
  // base offsets (concatenate slots)
  TOTAL_LEDS = 0;
  for (uint8_t s=0; s<PIN_SLOTS; ++s) {
    slotBase[s] = TOTAL_LEDS;
    TOTAL_LEDS += slotCount[s];
  }
}

void buildPath() {
  pathLen = 0;
  const uint8_t orderLen = sizeof(fieldOrder) / sizeof(fieldOrder[0]);
  for (uint8_t k=0; k<orderLen; ++k) {
    const Field &f = fields[fieldOrder[k]];
    for (uint16_t i=0; i<f.count; ++i) {
      if (pathLen >= TOTAL_LEDS_MAX) { Serial.println(F("Path overflow")); return; }
      int32_t inside = (int32_t)f.logicalZero + (int32_t)f.dir * (int32_t)i;
      inside = wrapU(inside, f.count);
      path[pathLen++] = slotBase[f.slot] + f.start + (uint16_t)inside;
    }
  }
}

// Lower speedMs = faster animation. These helpers convert speedMs to useful timings.
inline uint16_t spStep(uint16_t minMs=5, uint16_t scale=1) {
  // frame step interval (ms) — smaller speedMs → smaller interval
  uint32_t t = (uint32_t)speedMs / (uint32_t)scale;
  return (uint16_t)vmax<uint32_t>(t, minMs);
}
inline uint32_t spPeriod(uint16_t mult, uint16_t minMs) {
  // long period (ms) that scales with speedMs
  uint32_t p = (uint32_t)speedMs * (uint32_t)mult;
  return vmax<uint32_t>(p, (uint32_t)minMs);
}
inline uint8_t spProb(uint8_t slow=8, uint8_t fast=40) {
  // spawn probability (0..255) — faster speed => higher probability
  // map speedMs ∈ [2..200] roughly to prob ∈ [fast..slow]
  uint16_t s = vmin<uint16_t>(200, vmax<uint16_t>(2, speedMs));
  return (uint8_t)(slow + (uint16_t)(200 - s) * (uint16_t)(fast - slow) / 198);
}




void adjustSpeed(int delta) {
  int32_t s = (int32_t)speedMs + delta;
  if (s < SPEED_MIN) s = SPEED_MIN;
  if (s > SPEED_MAX) s = SPEED_MAX;
  speedMs = (uint16_t)s;
  Serial.print(F("Speed (ms/step) → "));
  Serial.println(speedMs);
  markChanged();
}

void adjustBrightness(int delta) {
  int32_t b = (int32_t)currentBrightness + delta;
  if (b < 0) b = 1;
  if (b > 255) b = 255;
  currentBrightness = (uint8_t)b;
  FastLED.setBrightness(currentBrightness);
  Serial.print(F("Brightness → "));
  Serial.println(currentBrightness);
  markChanged();
}

void handleIR() {

  // Non-blocking: only act when a complete code is available
  if (IrReceiver.decode()) {
  uint16_t cmd = IrReceiver.decodedIRData.command;
  uint16_t addr = IrReceiver.decodedIRData.address;
 
 IrReceiver.printIRResultShort(&Serial);
IrReceiver.printIRSendUsage(&Serial);

   Serial.print(F("Addr=0x"));
    Serial.print(IrReceiver.decodedIRData.address, HEX);
    Serial.print(F(" Cmd=0x"));
    Serial.println(IrReceiver.decodedIRData.command, HEX);

 // if(addr != IR_ADDR) {IrReceiver.resume(); return;
  //}

  // (Cheap debouncing for "repeat" frames)
  static uint16_t lastCmd = 0;
  static uint32_t lastAt  = 0;
  uint32_t now = millis();
  bool isRepeat = (cmd == lastCmd) && (now - lastAt < 180);  // ~NEC repeat cadence

  Serial.print(F("IR NEC  addr=0x")); Serial.print(addr, HEX);
  Serial.print(F(" cmd=0x")); Serial.print(cmd, HEX);
  Serial.print(F(" repeat=")); Serial.println(isRepeat ? F("YES") : F("NO"));

  if (!isRepeat) {
    if (cmd == IR_BTN_POWER_ON) 
    { 
      isOn = !isOn; 
      Serial.println(F("Power: Toggled"));

      if(!isOn)
      {
        saveSettingsNow();
      }
    } 
    else if (cmd == IR_BTN_BRIGTHNESS_UP) { adjustBrightness(+8);  } 
    else if (cmd == IR_BTN_BRIGTHNESS_DOWN) { adjustBrightness(-8);}
    else if (cmd == IR_BTN_FASTER) { adjustSpeed(-3); } 
    else if (cmd == IR_BTN_SLOWER) { adjustSpeed(+3); }

    // Single button to cycle modes
    else if (cmd == IR_BTN_MODE_AUTO) 
    {
      Mode next = mode;
      do {
        next = (Mode)((next + 1) % MODE_COUNT);
      } while (isAutoSkippable(next));   // keep skipping until valid
    
      setMode(next);
    }
    else if (cmd == IR_BTN_MODE_FLASH) { setMode(MODE_FLASH); }
    else if (cmd == IR_BTN_MODE_FADE7) { setMode(MODE_RAINBOW); }
    else if (cmd == IR_BTN_MODE_FADE3) { setMode(MODE_FADE); }

    // Generic solid colors
    else if (cmd == IR_BTN_RED)          { solidColor = CHSV(0,   255, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_GREEN)        { solidColor = CHSV(96,  255, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_BLUE)         { solidColor = CHSV(160, 255, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_WHITE)        { solidColor = CHSV(0,     0, 255); setMode(MODE_COLOR); }

    else if (cmd == IR_BTN_DARKORANGE)   { solidColor = CHSV(28,  255, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_LIGHTGREEN)   { solidColor = CHSV(96,  128, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_LIGHTBLUE)    { solidColor = CHSV(160, 128, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_PINK)         { solidColor = CHSV(224,  80, 255); setMode(MODE_COLOR); }

    else if (cmd == IR_BTN_ORANGE)       { solidColor = CHSV(32,  255, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_CYAN)         { solidColor = CHSV(128, 255, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_DARKPURPLE)   { solidColor = CHSV(192, 255, 180); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_LIGHTPINK)    { solidColor = CHSV(255,  40, 255); setMode(MODE_COLOR); }

    else if (cmd == IR_BTN_YELLOWORANGE) { solidColor = CHSV(40,  255, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_TEAL)         { solidColor = CHSV(100, 200, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_PURPLE)       { solidColor = CHSV(200, 255, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_SKYBLUE)      { solidColor = CHSV(140, 180, 255); setMode(MODE_COLOR); }

    else if (cmd == IR_BTN_YELLOW)       { solidColor = CHSV(64,  255, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_AZURE)        { solidColor = CHSV(150, 150, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_MAGENTA)      { solidColor = CHSV(224, 255, 255); setMode(MODE_COLOR); }
    else if (cmd == IR_BTN_LIGHTCYAN)    { solidColor = CHSV(128, 100, 255); setMode(MODE_COLOR); }

    // Adjust RGB channels
    else if (cmd == IR_BTN_COLOR_RED_UP)    { adjustDIY(0, +8); }
    else if (cmd == IR_BTN_COLOR_RED_DOWN)  { adjustDIY(0, -8); }
    else if (cmd == IR_BTN_COLOR_GREEN_UP)  { adjustDIY(1, +8); }
    else if (cmd == IR_BTN_COLOR_GREEN_DOWN){ adjustDIY(1, -8); }
    else if (cmd == IR_BTN_COLOR_BLUE_UP)   { adjustDIY(2, +8); }
    else if (cmd == IR_BTN_COLOR_BLUE_DOWN) { adjustDIY(2, -8); }

    // Save / Recall DIY slots
    else if (cmd == IR_BTN_DIY1) { loadDIY(0); }
    else if (cmd == IR_BTN_DIY2) { loadDIY(1); }
    else if (cmd == IR_BTN_DIY3) { loadDIY(2); }
    else if (cmd == IR_BTN_DIY4) { loadDIY(3); }
    else if (cmd == IR_BTN_DIY5) { loadDIY(4); }
    else if (cmd == IR_BTN_DIY6) { loadDIY(5); }

    markChanged();
  }

  lastCmd = cmd;
  lastAt  = now;
  
  
}
IrReceiver.resume();
}

bool isAutoSkippable(Mode m) {
  return (
          m == MODE_COLOR   ||
          m == MODE_DIY   ||
          m == MODE_COUNT
          );
}


//////////////////////////////
// DIY custom color storage
/////////////////////////////

CRGB diyColors[6];          // DIY1..DIY6
CRGB currentDIY = CRGB::Black;
uint8_t diyIndex = 0;       // active DIY slot (0..5)

void adjustDIY(int channel, int delta) {
  int value;
  if (channel == 0) value = currentDIY.r;
  else if (channel == 1) value = currentDIY.g;
  else value = currentDIY.b;

  value += delta;
  if (value < 0) value = 0;
  if (value > 255) value = 255;

  if (channel == 0) currentDIY.r = value;
  else if (channel == 1) currentDIY.g = value;
  else currentDIY.b = value;

   saveDIY(diyIndex);
}

void saveDIY(uint8_t idx) {
  if (idx < 6) {
    diyColors[idx] = currentDIY;
    Serial.print(F("Saved DIY")); Serial.println(idx + 1);
    markChanged();
  }
}

void loadDIY(uint8_t idx) {
  if (idx < 6) {
    currentDIY = diyColors[idx];
    diyIndex = idx;
    setMode(MODE_DIY);
    Serial.print(F("Loaded DIY")); Serial.println(idx + 1);
  }
}

// =====================================================
// DIY RENDERER
// =====================================================
void renderDIY(uint32_t now) {
  const uint16_t n = NLED();
  if (!n) return;
  fillV(currentDIY);
}

// =====================================================
// SNAKE RENDERER (Virtual view 'V' version)
// =====================================================

// ---- Config ----
const CRGB  SNAKE_HEAD_COLOR = CRGB::Red;
const CRGB  SNAKE_BODY_BASE  = CRGB(0, 180, 0); // green body
const uint8_t SNAKE_TAIL_MIN = 60;              // min brightness along tail

// ---- State (no body array!) ----
enum SnakeState : uint8_t { SNAKE_PLAY = 0, SNAKE_GAMEOVER = 1 };
static SnakeState snakeState = SNAKE_PLAY;

static uint16_t snakeHead = 0;        // head index (0..NLED-1)
static uint16_t snakeLen  = 8;        // length in pixels (1..NLED)
static int8_t   snakeDir  = +1;       // +1 forward, -1 backward
static uint16_t snakeFood = 0;        // food index
static CRGB     snakeFoodColor = CRGB::White;
static uint32_t snakeLastStep = 0;    // timing

// Game-over animation
static uint8_t  snakeGO_step   = 0;   // 0..N
static uint32_t snakeGO_nextAt = 0;

// ---- Helpers ----
static inline uint16_t wrapIndex(int32_t x, uint16_t n) {
  int32_t r = x % (int32_t)n;
  return (r < 0) ? (r + n) : (uint16_t)r;
}

// distance on ring in the + direction: (a - b) mod n is distance from b to a
static inline uint16_t distPlus(uint16_t a, uint16_t b, uint16_t n) {
  return (uint16_t)((a + n - b) % n);
}

// Is position p currently occupied by the contiguous body segment?
// Body occupies { head, head - dir, head - 2*dir, … head - (len-1)*dir }.
static bool snakeOccupies(uint16_t p, uint16_t n) {
  if (snakeDir == +1) {
    // occupied if distance from p forward to head < len  (i.e., (head - p) mod n < len)
    return distPlus(snakeHead, p, n) < snakeLen;
  } else {
    // occupied if distance from head forward to p < len  (i.e., (p - head) mod n < len)
    return distPlus(p, snakeHead, n) < snakeLen;
  }
}

static uint16_t randomFreePos(uint16_t n) {
  for (uint8_t tries = 0; tries < 40; ++tries) {
    uint16_t p = random16(n);
    if (!snakeOccupies(p, n)) return p;
  }
  for (uint16_t p = 0; p < n; ++p) if (!snakeOccupies(p, n)) return p;
  return 0;
}

static void placeFood(uint16_t n) {
  snakeFood = randomFreePos(n);
  snakeFoodColor = CHSV(random8(), 255, 255);  // vivid random color
}

static void snakeReset() {
  const uint16_t n = NLED();
  if (n == 0) return;
  snakeLen  = vmin<uint16_t>(8, n);
  snakeDir  = +1;
  snakeHead = (n / 4);                 // start around quarter point
  placeFood(n);
  snakeLastStep  = 0;
  snakeState     = SNAKE_PLAY;
  snakeGO_step   = 0;
  snakeGO_nextAt = 0;
}

// Returns true while animating; false when finished
static bool snakeGameOverAnim(uint32_t now) {
  const uint8_t flashes = 6;
  const uint16_t frameMs = 120;
  const uint16_t n = NLED();
  if (now < snakeGO_nextAt) return true;
  if (snakeGO_step < flashes) {
    bool on = ((snakeGO_step & 1) == 0);
    fillV(on ? CRGB::Red : CRGB::Black);
    snakeGO_step++;
    snakeGO_nextAt = now + frameMs;
    return true;
  }
  return false;
}

// ---- Renderer ----
void renderSnake(uint32_t now) {
  const uint16_t n = NLED();
  if (n == 0) return; // nothing to draw

  // If mapping changed and state is out-of-bounds, re-init gracefully.
  if (snakeHead >= n || snakeLen > n) snakeReset();

  // Game-over animation branch
  if (snakeState == SNAKE_GAMEOVER) {
    if (!snakeGameOverAnim(now)) snakeReset();
    return;
  }

  // Movement (tied to speedMs)
  if (now - snakeLastStep >= speedMs) {
    snakeLastStep = now;

    uint16_t nextHead = wrapIndex((int32_t)snakeHead + snakeDir, n);

    // Self bite? In contiguous 1D, the only way is when len == n (full board)
    // or if you ever try to reverse direction into your own body.
    if (snakeOccupies(nextHead, n)) {
      snakeState   = SNAKE_GAMEOVER;
      snakeGO_step = 0;
      snakeGO_nextAt = now;
      return;
    }

    // Advance head
    snakeHead = nextHead;

    // Eat?
    if (snakeHead == snakeFood) {
      if (snakeLen < n) snakeLen++;      // grow
      if (snakeLen >= n) {               // full board -> game over “win”
        snakeState   = SNAKE_GAMEOVER;
        snakeGO_step = 0;
        snakeGO_nextAt = now;
        return;
      }
      placeFood(n);
    }
  }

  // Draw on the mapped view
  fillV(CRGB::Black);

  // Food
  P(snakeFood) = snakeFoodColor;

  // Body: head red, tail green gradient
  for (uint16_t k = 0; k < snakeLen; ++k) {
    // body extends "behind" the head opposite to direction: idx = head - dir*k
    uint16_t idx = wrapIndex((int32_t)snakeHead - (int32_t)snakeDir * (int32_t)k, n);
    if (k == 0) {
      P(idx) = SNAKE_HEAD_COLOR;
    } else {
      uint8_t scale = (uint8_t)vmax<uint16_t>(SNAKE_TAIL_MIN, 255 - k * 6);
      CRGB c = SNAKE_BODY_BASE; c.nscale8(scale);
      P(idx) = c;
    }
  }
}

// =====================================================
/// STAR RENDERER
// =====================================================

#define MAX_COMETS 8   // more slots = more chances

#define MAX_SHOOTERS 5

struct ShootingStar {
  int16_t  head;        // aktuelle Kopf-LED
  int8_t   dir;         // +1 oder -1
  uint8_t  tail;        // Tail-Länge in Pixeln (2..6)
  uint8_t  stepsLeft;   // verbleibende Schritte (4..12)
  CRGB     color;       // Kopf-Farbe (leicht bläulich-weiß)
  bool     active;
};
ShootingStar shooters[MAX_SHOOTERS];

inline void resetShootingStars() {
  for (uint8_t i=0;i<MAX_SHOOTERS;i++) shooters[i].active = false;
}

// Schrittintervall für Bewegung (kleiner speedMs => schnellere Bewegung)
inline uint16_t shootStepMs() {
  return vmax<uint16_t>(6, speedMs);       // z.B. 6..200 ms
}

// Spawn-Wahrscheinlichkeit (kleiner speedMs => öfter)
inline bool shouldSpawnShooter() {
  // Probabilistisch: ~1 Treffer, wenn Zufallszahl 0 ist
  uint16_t window = vmax<uint16_t>(100, speedMs * 6);  // 250..1200
  return random16(window) == 0;
}

// ---------- Persistent Stars (slow twinkle) ----------
uint8_t  STAR_SEED         = 137;    // change to reshuffle positions
uint8_t  STAR_DENSITY_256  = 60;     // 0..255; ~Density ≈ (DENSITY/256)*NUM_LEDS
uint8_t  STAR_SPEED_SHIFT  = 3;      // larger = slower twinkle (phase uses now>>SHIFT)
uint8_t  STAR_TINT_MAX_SAT = 30;     // 0 = pure white; small values = subtle tint

// Fast, deterministic 8-bit hash from index+seed
inline uint8_t hash8_u16(uint16_t x, uint8_t seed) {
  x ^= seed;
  x ^= (x >> 4);
  x *= 0x11u;           // small odd multiplier
  x ^= (x >> 7);
  x ^= (x << 3);
  return (uint8_t)x;
}

void renderStars(uint32_t now) {
  const uint16_t n = NLED();
  if (!n) return;

  // clear through the mapped view
  fillV(CRGB::Black);

  // slow-moving time base
  const uint8_t tbase = (uint8_t)(now >> STAR_SPEED_SHIFT);

  for (uint16_t i = 0; i < n; ++i) {
    uint8_t h = hash8_u16(i, STAR_SEED);     // hash over LOGICAL index

    // Only some indices become stars
    if (h >= STAR_DENSITY_256) continue;

    // Per-star phase / brightness
    uint8_t phase = tbase + (h << 1);

    uint8_t minV = 40 + (h & 0x1F);                 // 40..71
    uint8_t maxV = 190 + ((h >> 5) & 0x3F);         // ~190..253

    uint8_t tw = sin8(phase);                       // 0..255
    uint8_t v  = lerp8by8(minV, maxV, tw);          // min..max

    uint8_t sat = STAR_TINT_MAX_SAT ? (h % (STAR_TINT_MAX_SAT + 1)) : 0;
    uint8_t hue = h;

    P(i) = CHSV(hue, sat, v);                       // write via view
  }
}

// --- Drop-in replacement for renderStarsComets() ---
void renderStarsComets(uint32_t now) {
  const uint16_t n = NLED();
  if (!n) return;

  // Hintergrund: langsames Funkeln
  fillV(CRGB::Black);
  const uint8_t tbase = (uint8_t)(now >> STAR_SPEED_SHIFT);
  for (uint16_t i = 0; i < n; ++i) {
    uint8_t h = hash8_u16(i, STAR_SEED);
    if (h < STAR_DENSITY_256) {
      uint8_t phase = tbase + (h << 1);
      uint8_t minV  = 40 + (h & 0x1F);
      uint8_t maxV  = 190 + ((h >> 5) & 0x3F);
      uint8_t tw    = sin8(phase);
      uint8_t v     = lerp8by8(minV, maxV, tw);
      uint8_t sat   = STAR_TINT_MAX_SAT ? (h % (STAR_TINT_MAX_SAT + 1)) : 0;
      P(i)         += CHSV(h, sat, v);
    }
  }

  // Sternschnuppen aktualisieren
  static uint32_t nextStepAt = 0;
  if (now >= nextStepAt) {
    nextStepAt = now + shootStepMs();

    // Bewegung & Lebenszeit
    for (uint8_t s = 0; s < MAX_SHOOTERS; ++s) {
      if (!shooters[s].active) continue;

      if (shooters[s].stepsLeft == 0) {
        shooters[s].active = false;
        continue;
      }

      shooters[s].head += shooters[s].dir;
      shooters[s].stepsLeft--;

      if (shooters[s].head < 0 || shooters[s].head >= (int16_t)n) {
        shooters[s].active = false;
      }
    }

    // Spawnen (selten, zufälliger Ort, kurze Lebensdauer)
    if (shouldSpawnShooter()) {
      for (uint8_t s = 0; s < MAX_SHOOTERS; ++s) {
        if (!shooters[s].active) {
          uint8_t tail = 2 + (random8() % 5);

          int16_t startMin = tail + 1;
          int16_t startMax = (n - 1) - (tail + 1);
          if (startMax < startMin) break; // zu kurzer Streifen

          int16_t start = (int16_t)random16((uint16_t)startMin, (uint16_t)(startMax + 1));
          int8_t  dir   = (random8() & 1) ? +1 : -1;

          shooters[s].head      = start;
          shooters[s].dir       = dir;
          shooters[s].tail      = tail;
          shooters[s].stepsLeft = 4 + (random8() % 9);  // 4..12 Schritte
          shooters[s].color     = CRGB(
            210 + random8(45),   // leicht bläuliches Weiß
            210 + random8(45),
            230 + random8(25)
          );
          shooters[s].active    = true;
          break; // nur eine pro Tick
        }
      }
    }
  }

  // Zeichnen: sehr kurze, helle Streifen
  fadeV(20); // globaler Fade für schnelle Spur-Auflösung

  for (uint8_t s = 0; s < MAX_SHOOTERS; ++s) {
    if (!shooters[s].active) continue;

    int16_t head = shooters[s].head;
    int8_t  dir  = shooters[s].dir;
    uint8_t tail = shooters[s].tail;

    for (uint8_t k = 0; k <= tail; ++k) {
      int16_t idx = head - dir * k;
      if (idx < 0 || idx >= (int16_t)n) break;

      uint8_t fall = 255 - (uint16_t)k * k * (255 / ((tail+1)*(tail+1))); // starkes Abklingen
      CRGB px = shooters[s].color;
      px.nscale8_video(fall);
      P((uint16_t)idx) += px;  // additiv über Sterne
    }
  }
}


// =====================================================
// RAINBOW RENDERER
// =====================================================
void renderRainbow(uint32_t now) {
  const uint16_t n = NLED();
  if (!n) return;

  // One full hue cycle scales with speedMs
  const uint32_t period  = vmax<uint32_t>((uint32_t)speedMs * 120UL, 300UL);
  const uint8_t  hueBase = (uint32_t)(now % period) * 255 / period;

  // Hue step per logical pixel
  const uint8_t  deltaHue = vmax<uint8_t>(1, (uint8_t)(256 / n));

  for (uint16_t i = 0; i < n; ++i) {
    P(i) = CHSV(hueBase + i * deltaHue, 255, 255);
  }
}

// =====================================================
// FADE RENDERER
// =====================================================
void renderFade(uint32_t now) {
  const uint16_t n = NLED();
  if (!n) return;

  const uint32_t period = vmax<uint32_t>((uint32_t)speedMs * 100UL, 200UL);
  const uint32_t t = now % period;
  const uint8_t  hue = (uint32_t)t * 255 / period;

  for (uint16_t i = 0; i < n; ++i) {
    P(i) = CHSV(hue, 255, 255);
  }
}

// =====================================================
// COLOR SWICHT RENDERER
// =====================================================

const CRGB switchPalette[] = {
  CRGB::Red, CRGB::Orange, CRGB::Yellow, CRGB::Green,
  CRGB::Blue, CRGB::Indigo, CRGB::Violet, CRGB::White
};
const uint8_t SWITCH_COUNT = sizeof(switchPalette) / sizeof(switchPalette[0]);
uint8_t switchIndex = 0;

void renderSwitch(uint32_t now) {
  const uint16_t n = NLED();
  if (!n) return;

  if (now - lastSwitchMs >= (uint32_t)vmax<uint32_t>(100, (uint32_t)speedMs * 20U)) {
    lastSwitchMs = now;
    switchIndex = (switchIndex + 1) % SWITCH_COUNT;
  }

  fillV(switchPalette[switchIndex]);
}


// =====================================================
// FLASH RENDERER
// =====================================================
// Flash mode state
uint8_t  flashStep     = 0;
uint32_t nextFlashAt   = 0;
CRGB     flashColor    = CRGB::White;

void renderFlash(uint32_t now) 
{
  const uint16_t n = NLED();
  if (!n) return;

  switch (flashStep) {
    case 0: // schedule/start bright flash
      if (now >= nextFlashAt) {
        // bluish-white tint
        flashColor = CRGB(
          200 + random8(56), // R: 200–255
          200 + random8(56), // G: 200–255
          230 + random8(26)  // B: 230–255 (slightly more blue)
        );
        fillV(flashColor);
        nextFlashAt = now + random(30, 80);  // short burst
        flashStep = 1;
      }
      break;

    case 1: // fade to dim
      if (now >= nextFlashAt) {
        CRGB dim = flashColor;  // make a copy, then dim it
        dim.nscale8(60);
        fillV(dim);
        nextFlashAt = now + random(40, 120);
        flashStep = (random8() < 100) ? 2 : 3; // sometimes a second burst
      }
      break;

    case 2: // optional second burst
      if (now >= nextFlashAt) {
        fillV(flashColor);
        nextFlashAt = now + random(20, 60);
        flashStep = 3;
      }
      break;

    case 3: // back to dark, then wait a while before the next flash
      if (now >= nextFlashAt) {
        fillV(CRGB::Black);
        nextFlashAt = now + random(1000, 6000); // 0.5–4 s
        flashStep = 0; // next cycle will start when the wait elapses
      }
      break;
  }
}

// =====================================================
// RUNNING DOT RENDERER
// =====================================================
// Running Dot state
uint16_t dotPos = 0;

void renderDot(uint32_t now) {
  static uint32_t lastStep = 0;
  const uint16_t n = NLED();
  if (!n) return;

  // faster travel: step size grows as speedMs gets smaller
  uint16_t stepSize = vmax<uint16_t>(1, (200 / vmax<uint16_t>(2, speedMs)));

  if (now - lastStep >= spStep(/*minMs*/5, /*scale*/1)) {
    lastStep = now;
    dotPos = (dotPos + stepSize) % n;
  }

  // hard clear the mapped view so no colors linger
  fillV(CRGB::Black);

  // draw the white dot
  P(dotPos) = CRGB::White;
}

// =====================================================
//  NOISE RENDERER
// =====================================================

// ---------- Tuline-style Perlin noise palette state ----------
CRGBPalette16 currentPalette(CRGB::Black);
CRGBPalette16 targetPalette(OceanColors_p);

// How quickly the color field moves; ties to your speedMs for consistency.
// Larger = finer noise scale (smaller “blobs”).
uint8_t  NOISE_SCALE = 30;

// Helpers for Render noise
// Create a new random-ish target palette around a base hue (Tuline style)
static void chooseRandomTargetPalette() {
  uint8_t baseC = random8();
  targetPalette = CRGBPalette16(
    CHSV(baseC + random8(32), 255, random8(128,255)),
    CHSV(baseC + random8(64), 255, random8(128,255)),
    CHSV(baseC + random8(96), 192, random8(128,255)),
    CHSV(baseC + random8(16), 255, random8(128,255))
  );
}

// Fill the strip from 2D Perlin noise with palette lookup
static void fillnoise8_render(uint32_t now) {
  const uint16_t n = NLED();
  if (!n) return;

  // Map time to a slower/faster scroll using speedMs.
  const uint16_t timeDiv = vmax<uint16_t>(4, speedMs);   // avoid div-by-zero, feel free to tweak
  const uint16_t t = now / timeDiv;

  for (uint16_t i = 0; i < n; ++i) {
    // Two-dimensional noise: one axis = position, the other = time
    uint8_t index = inoise8(i * NOISE_SCALE, t + i * NOISE_SCALE);
    P(i) = ColorFromPalette(currentPalette, index, 255, LINEARBLEND);
  }
}

void renderNoise(uint32_t now) {
  const uint16_t n = NLED();
  if (!n) return;

  // Map speedMs (2..200) to a scroll divisor (small = faster movement)
  const uint16_t timeDiv = vmax<uint16_t>(1, speedMs / 2);
  const uint16_t t = now / timeDiv;

  for (uint16_t i = 0; i < n; ++i) {
    // 2D noise: X axis = pixel position, Y axis = time
    uint8_t index = inoise8(i * NOISE_SCALE, t + i * NOISE_SCALE);
    P(i) = ColorFromPalette(currentPalette, index, 255, LINEARBLEND);
  }

  // Palette blending / retargeting
  EVERY_N_MILLIS(10) { nblendPaletteTowardPalette(currentPalette, targetPalette, 48); }
  EVERY_N_SECONDS(5) { chooseRandomTargetPalette(); }
}

// =====================================================
// SOLID COLOR RENDERER
// =====================================================
void setSolidColor(const CRGB& c) { solidColor = c; setMode(MODE_COLOR); }

void renderColor(uint32_t now) {
  (void)now;
  const uint16_t n = NLED();
  if (!n) return;
  fillV(solidColor);
}

// =====================================================
// Rainbow2 (already have, but clean version) Renderer
// =====================================================
void renderRainbow2(uint32_t now) {
  const uint16_t n = NLED();
  if (!n) return;

  const uint32_t period  = spPeriod(120, 300); // full hue cycle
  const uint8_t  hueBase = (uint32_t)(now % period) * 255 / period;
  const uint8_t  deltaHue = vmax<uint8_t>(1, (uint8_t)(256 / n));

  for (uint16_t i = 0; i < n; ++i) {
    P(i) = CHSV(hueBase + i * deltaHue, 255, 255);
  }
}

// =====================================================
// Meteor / Comet Renderer
// =====================================================
void renderMeteor(uint32_t now) {
  static uint16_t pos = 0;
  static uint32_t nextAt = 0;

  const uint16_t n = NLED();
  if (!n) return;
  if (pos >= n) pos = 0; // guard in case mapping/length changed

  if (now >= nextAt) {
    nextAt = now + spStep(/*minMs*/4, /*scale*/1); // faster when speedMs is small
    fadeV(40);                                     // trail fade via mapped view
    pos = (pos + 1) % n;
    P(pos) = CRGB::White;                          // draw meteor head
  }
}

// =====================================================
// Bouncing Balls Renderer
// =====================================================
struct Ball { int16_t pos; int8_t vel; CRGB color; };
#ifndef BALLS
#define BALLS 3
#endif
static Ball balls[BALLS] = {
  { NUM_LEDS/6,   +1, CRGB::Red   },
  { NUM_LEDS/2,   +1, CRGB::Green },
  { NUM_LEDS*5/6, -1, CRGB::Blue  }
};

void renderBounce(uint32_t now) {
  static uint32_t nextAt = 0;

  const uint16_t n = NLED();
  if (!n) return;

  if (now >= nextAt) {
    nextAt = now + spStep(/*minMs*/6, /*scale*/1);

    // trail fade
    fadeV(45);

    for (uint8_t i = 0; i < BALLS; i++) {
      // guard against mapping/length changes
      if (balls[i].pos < 0) balls[i].pos = 0;
      if (balls[i].pos >= (int16_t)n) balls[i].pos = (int16_t)n - 1;

      // advance
      balls[i].pos += balls[i].vel;

      // bounce at edges
      if (balls[i].pos <= 0)                  { balls[i].pos = 0;                 balls[i].vel = +1; }
      if (balls[i].pos >= (int16_t)n - 1)     { balls[i].pos = (int16_t)n - 1;    balls[i].vel = -1; }

      // draw via mapped view
      P((uint16_t)balls[i].pos) += balls[i].color;
    }
  }
}

// =====================================================
// Fireflies Renderer
// =====================================================
void renderFireflies(uint32_t now) {
   const uint16_t n = NLED();
  if (!n) return;

  // slower fade when slow speed → “linger”; faster fade when fast
  const uint8_t fade = (uint8_t)vmin<uint16_t>(40, 10 + (200 - vmin<uint16_t>(200, speedMs))/5);
  fadeV(fade);

  // spawn rate depends on speed
  if (random8() < spProb(/*slow*/6, /*fast*/35)) {
    uint16_t i = random16(n);
    P(i) += CHSV(random8(60,120), 200, 255); // yellow-green glow
  }
}

// =====================================================
// Matrix Code Rain Renderer
// =====================================================
void renderMatrix(uint32_t now) {
  const uint16_t n = NLED();
  if (!n) return;

  // trail persistence (mapped)
  uint8_t fadeAmt = (uint8_t)map(speedMs, SPEED_MIN, SPEED_MAX, 10, 80);
  fadeAmt = vmin<uint8_t>(80, vmax<uint8_t>(10, fadeAmt));
  fadeV(fadeAmt);

  // spawn probability scales with speed
  uint8_t spawnChance = (uint8_t)map(speedMs, SPEED_MIN, SPEED_MAX, 120, 20);
  spawnChance = vmin<uint8_t>(120, vmax<uint8_t>(20, spawnChance));

  if (random8() < spawnChance) {
    uint16_t pos = random16(n);
    uint8_t  dropLen = random8(4, 15); // 4–14 LEDs

    // head bright, tail dimmer, going "down" the index space
    for (uint8_t i = 0; i < dropLen; ++i) {
      int32_t idx = (int32_t)pos - (int32_t)i;
      if (idx < 0) break;

      uint8_t brightness = 255 - (uint16_t)i * (200 / dropLen);
      P((uint16_t)idx) += CHSV(96, 255, brightness); // green-ish
    }
  }
}

// =====================================================
// Aurora Renderer TODO: Wdoppelt mit PLASMA?
// =====================================================

DEFINE_GRADIENT_PALETTE(aurora_gp) {
  0,   0,   0,   0,   // black
  64,  0,  55,   0,   // deep green
  128, 0, 200, 128,   // teal
  192, 0,  64, 200,   // blue
  255, 100, 0, 200    // violet
};
CRGBPalette16 auroraPalette = aurora_gp;

void renderAurora(uint32_t now) {
  const uint16_t n = NLED();
  if (!n) return;

  const uint16_t t = now / vmax<uint16_t>(4, speedMs); // speed-controlled
  for (uint16_t i = 0; i < n; ++i) {
    uint8_t nv  = inoise8(i * 40, t);          // noise field
    uint8_t bri = sin8(nv + (i * 3));          // subtle shimmer
    P(i) = ColorFromPalette(auroraPalette, nv, bri, LINEARBLEND);
  }
}

// =====================================================
// Rainbow Sparkles Renderer 
// =====================================================
void renderSparkles(uint32_t now) {
  const uint16_t n = NLED();
  if (!n) return;

  const uint32_t period = spPeriod(100, 200);
  const uint8_t  baseHue = (uint32_t)(now % period) * 255 / period;

  // base rainbow background
  const uint8_t deltaHue = 4; // keep tight bands like your original
  for (uint16_t i = 0; i < n; ++i) {
    P(i) = CHSV(baseHue + i * deltaHue, 255, 255);
  }

  // occasional white sparkles; rate scales with speed
  if (random8() < spProb(/*slow*/5, /*fast*/35)) {
    P(random16(n)) = CRGB::White;
  }
}

// =====================================================
// Confetti Renderer
// =====================================================
void renderConfetti(uint32_t now) {
  (void)now;
  const uint16_t n = NLED();
  if (!n) return;

  // fade scales with speed
  const uint8_t fade = (uint8_t)vmin<uint16_t>(40, 10 + (200 - vmin<uint16_t>(200, speedMs))/5);
  fadeV(fade);

  // spawn rate scales with speed
  uint8_t bursts = 1 + (spProb(/*slow*/8, /*fast*/40) / 20); // ~1..2 bursts
  while (bursts--) {
    P(random16(n)) += CHSV(random8(), 200, 255);
  }
}

// =====================================================
// Juggle Renderer
// =====================================================
void renderJuggle(uint32_t now) {
  static uint32_t nextAt = 0;
  if (now < nextAt) return;

  const uint16_t n = NLED();
  if (!n) return;

  nextAt = now + spStep(/*minMs*/6, /*scale*/1);

  // gentle trail
  fadeV(20);

  // speed factor for oscillation: smaller speedMs => faster motion
  const uint16_t oscDiv = vmax<uint16_t>(1, speedMs / 2);
  const uint16_t t = now / oscDiv;

  uint8_t hue = 0;
  for (uint8_t i = 0; i < 8; i++) {
    // manual beatsin: 0..65535 mapped across 0..n-1
    uint16_t p = (sin16(t * (i + 1)) + 32768);     // 0..65535
    uint16_t x = (uint32_t)p * (n - 1) / 65535;
    P(x) |= CHSV(hue, 200, 255);
    hue += 32;
  }
}

// =====================================================
// Ripple Renderer
// =====================================================
void renderRipple(uint32_t now) {
  static int16_t  center = -1;
  static uint16_t step   = 0;
  static uint32_t nextAt = 0;

  const uint16_t n = NLED();
  if (!n) return;

  if (now >= nextAt) {
    nextAt = now + spStep(/*minMs*/8, /*scale*/1);

    // soft trail fade
    fadeV(35);

    // (re)start a ripple if none active; chance grows with speed
    if (center < 0 && random8() < spProb(/*slow*/3, /*fast*/25)) {
      center = (int16_t)random16(n);
      step   = 0;
    }

    // draw / advance ripple
    if (center >= 0) {
      for (uint16_t i = 0; i < n; ++i) {
        uint16_t d = (i > (uint16_t)center) ? (i - (uint16_t)center) : ((uint16_t)center - i);
        if (d == step) {
          P(i) = CHSV(((uint16_t)center * 3 + step * 8) & 0xFF, 255, 255);
        }
      }
      step++;
      if (step > n / 2) center = -1; // end ripple
    }
  }
}

// =====================================================
// Plasma (color waves) Renderer
// =====================================================
void renderPlasma(uint32_t now, const CRGBPalette16& pal) {
  const uint16_t n = NLED();
  if (!n) return;

  // smaller speedMs -> faster phase evolution
  const uint16_t t1 = now / vmax<uint16_t>(1, speedMs / 2);
  const uint16_t t2 = now / vmax<uint16_t>(1, speedMs / 3);

  for (uint16_t i = 0; i < n; ++i) {
    uint8_t a = sin8((i << 3) + (t1 >> 2));
    uint8_t b = cos8((i << 3) + (t2 >> 2));
    uint8_t idx = (a + b) >> 1; // 0..255
    P(i) = ColorFromPalette(pal, idx, 255, LINEARBLEND);
  }
}




void loop() {
  handleIR();
  saveSettingsIfNeeded();

  uint32_t now = millis();
  if (!isOn) {
    fill_solid(leds2, NUM_LEDS, CRGB::Black);
    showIfIdle();
    return;
  }

  switch (mode) {
    case MODE_RAINBOW: renderRainbow(now); break;
    case MODE_FADE:    renderFade(now); break;
    case MODE_SWITCH:  renderSwitch(now); break;
    case MODE_FLASH:   renderFlash(now); break;
    case MODE_SNAKE: renderSnake(now); break;
    case MODE_DOT:     renderDot(now); break;
    case MODE_COLOR:  renderColor(now); break;
    case MODE_NOISE:  renderNoise(now); break;
    case MODE_STARS: renderStars(now); break;
    case MODE_STARS_COMETS: renderStarsComets(now); break;
    case MODE_RAINBOW2:   renderRainbow2(now); break;
    case MODE_METEOR:    renderMeteor(now); break;
    case MODE_BOUNCE:    renderBounce(now); break;
    case MODE_FIREFLIES: renderFireflies(now); break;
    case MODE_MATRIX:    renderMatrix(now); break;
    case MODE_AURORA:    renderAurora(now); break;
    case MODE_SPARKLES:  renderSparkles(now); break;
    case MODE_CONFETTI:  renderConfetti(now); break;
    case MODE_JUGGLE:    renderJuggle(now); break;
    case MODE_RIPPLE:    renderRipple(now); break;
    case MODE_PLASMARAINBOW:    renderPlasma(now, RainbowColors_p); break;
    case MODE_PLASMA:    renderPlasma(now, HeatColors_p); break;
    case MODE_PLASMACLOUD:    renderPlasma(now, CloudColors_p); break;
    case MODE_PLASMAPARTY:    renderPlasma(now, PartyColors_p); break;
    case MODE_PLASMAFOREST:    renderPlasma(now, ForestColors_p); break;
    case MODE_DIY:     renderDIY(now); break; 
    default:           renderRainbow(now);
  }

  // 3) Push pixels, but only when the IR receiver is idle
   showIfIdle();
}


void setup() {
 Serial.begin(115200);
 delay(200);


  // Start IR (choose a valid external or pin-change interrupt pin)
  IrReceiver.begin(IR_RECEIVE_PIN, ENABLE_LED_FEEDBACK);  
  Serial.println(F("IRremote ready"));

  setupLedsWithPins();
  initView();

  //FastLED.addLeds<LED_TYPE, DATA_PIN, COLOR_ORDER>(leds, NUM_LEDS).setCorrection(TypicalLEDStrip);
  //FastLED.addLeds<LED_TYPE, DATA_PIN, COLOR_ORDER>(leds, NUM_LEDS).setCorrection(TypicalLEDStrip);

  FastLED.setBrightness(currentBrightness);
  FastLED.setDither(false); // a tiny CPU win
  loadSettings();

}

void setMode(Mode m) {
  mode = m;
  lastStepMs = 0;
  lastSwitchMs = 0;
  dotPos = 0;
 

  //if (m == MODE_SNAKE) snakeReset();

  if (m == MODE_FLASH) { flashStep = 0; nextFlashAt = 0; }
  else if (m == MODE_NOISE) {
    currentPalette = CRGBPalette16(CRGB::Black);
    targetPalette  = OceanColors_p;     // pleasant starting palette
  }
  else if (m == MODE_SNAKE) snakeReset();

  clearAllPhysical();

  Serial.print(F("Mode Changed to: "));
  Serial.println(m);

  
}

void addControllerForSlot(uint8_t s, uint16_t base, uint16_t count) {
  if (count == 0) return;
  switch (s) {
    case 0: FastLED.addLeds<WS2815, PIN0, GRB>(leds2, base, count); break;
    case 1: FastLED.addLeds<WS2815, PIN1, GRB>(leds2, base, count); break;
    case 2: FastLED.addLeds<WS2815, PIN2, GRB>(leds2, base, count); break;
    case 3: FastLED.addLeds<WS2815, PIN3, GRB>(leds2, base, count); break;
    case 4: FastLED.addLeds<WS2815, PIN4, GRB>(leds2, base, count); break;
    case 5: FastLED.addLeds<WS2815, PIN5, GRB>(leds2, base, count); break;
  }
}

void setupLedsWithPins() {
  computeSlotTotals();               // fills slotCount[] and slotBase[]
  for (uint8_t s=0; s<PIN_SLOTS; ++s) {
    addControllerForSlot(s, slotBase[s], slotCount[s]);
  }
  buildPath();                       // fills path[] in your visual travel order
  FastLED.clear(true);
}
void markChanged() {
  lastChangeMs = millis();
  needsSave = true;
}

void loadSettings() {
  Settings temp;
  uint16_t bestSeq = 0;
  int bestSlot = -1;

  for (int slot = 0; slot < SLOT_COUNT; slot++) {
    int addr = slot * SLOT_SIZE;
    EEPROM.get(addr, temp);
    if (temp.version == SETTINGS_VERSION && temp.sequence >= bestSeq) {
      bestSeq = temp.sequence;
      bestSlot = slot;
      settings = temp;
    }
  }

  if (bestSlot == -1) {
    Serial.println(F("No valid EEPROM settings → using defaults"));
    for (int i=0; i<6; i++) diyColors[i] = CRGB::Black;
    solidColor = CRGB::White;
    mode = MODE_PLASMAPARTY;
    currentBrightness = START_BRIGHT;
    speedMs = 20;
    diyIndex = 0;                                    // NEW default
    currentDIY = diyColors[diyIndex];                // keep currentDIY consistent
    isOn = true;                       // choose your preferred default

    settings.version    = SETTINGS_VERSION;
    settings.sequence   = 1;                // start sequence
    for (int i=0; i<6; i++) settings.diyColors[i] = diyColors[i];
    settings.solidColor = solidColor;
    settings.mode       = (uint8_t)mode;
    settings.brightness = currentBrightness;
    settings.speedMs    = speedMs;
    settings.diyIndex   = diyIndex;
    settings.powerOn    = isOn ? 1 : 0;     // NEW

    EEPROM.put(0, settings);                // write first slot
  } else {
    Serial.print(F("Loaded settings from slot ")); Serial.println(bestSlot);
    for (int i=0; i<6; i++) diyColors[i] = settings.diyColors[i];
    solidColor          = settings.solidColor;
    mode                = (Mode)settings.mode;
    currentBrightness   = settings.brightness;
    speedMs             = settings.speedMs;
    diyIndex = (settings.diyIndex < 6) ? settings.diyIndex : 0;
    currentDIY = diyColors[diyIndex];
    isOn              = (settings.powerOn != 0);   // NEW
    settings.sequence   = bestSeq;          // seed in-RAM sequence
  }

  FastLED.setBrightness(currentBrightness);
  
  if (mode == MODE_DIY) setMode(MODE_DIY);

}

void saveSettingsNow() {
   // bump sequence kept inside the struct
  settings.version  = SETTINGS_VERSION;
  settings.sequence = (uint16_t)(settings.sequence + 1);

  for (int i=0; i<6; i++) settings.diyColors[i] = diyColors[i];
  settings.solidColor = solidColor;
  settings.mode       = (uint8_t)mode;
  settings.brightness = currentBrightness;
  settings.speedMs    = speedMs;
  settings.diyIndex   = diyIndex;                 
   settings.powerOn    = isOn ? 1 : 0;   

  int slot = settings.sequence % SLOT_COUNT;
  int addr = slot * SLOT_SIZE;

  EEPROM.put(addr, settings);
  Serial.print(F("Saved settings at slot ")); Serial.println(slot);

  needsSave = false;
}

void saveSettingsIfNeeded() {
  if (needsSave && millis() - lastChangeMs > 30000UL) 
  {
    saveSettingsNow();  
  }
}





// ---------- Frame pushing that cooperates with IR ----------

void showIfIdle() {
  static uint32_t last = 0;
  const uint16_t minIntervalMs = 25; // ~60 FPS cap
  uint32_t now = millis();

  // Only push a frame if the IR receiver isn't currently capturing a frame
  if ((now - last) >= minIntervalMs && IrReceiver.isIdle()) {
    FastLED.show();
    last = now;
  }
}


