// ====================================
// GoPro Tripod Leg - Parametric Version
// ====================================
// Adjustable tripod leg with customizable number of sections
// for easier printing on smaller print beds

/* [Leg Configuration] */
// Number of middle sections (1-3 recommended)
n_sections = 2; // [1:5]

/* [Hidden] */
// Internal dimensions from original model
ORIGINAL_SECTIONS = 3;
SECTION_HEIGHT = 52.5;
TOP_SECTION_Z = 195;
TOP_SECTION_HEIGHT = 30;
BOUNDING_BOX_SIZE = 30;

// Import original STL model
module original_leg() {
    import("leg.STL");
}

// Extract top section (with GoPro mount)
module top_section() {
    intersection() {
        original_leg();
        translate([0, 0, TOP_SECTION_Z])
            cube([BOUNDING_BOX_SIZE, BOUNDING_BOX_SIZE, TOP_SECTION_HEIGHT]);
    }
}

// Extract one repeatable middle section
module middle_section() {
    intersection() {
        original_leg();
        translate([0, 0, TOP_SECTION_Z - SECTION_HEIGHT])
            cube([BOUNDING_BOX_SIZE, BOUNDING_BOX_SIZE, SECTION_HEIGHT]);
    }
}

// Extract bottom section (with foot)
module bottom_section() {
    intersection() {
        original_leg();
        cube([BOUNDING_BOX_SIZE, BOUNDING_BOX_SIZE, TOP_SECTION_Z - ORIGINAL_SECTIONS * SECTION_HEIGHT]);
    }
}

// Assemble parametric leg with desired number of sections
module parametric_leg() {
    // Adjust vertical position based on number of sections
    translate([0, 0, -(ORIGINAL_SECTIONS - n_sections) * SECTION_HEIGHT]) {
        top_section();
        // Stack middle sections
        for (i = [0:n_sections-1])
            translate([0, 0, -i * SECTION_HEIGHT])
                middle_section();
    }
    bottom_section();
}

// Rotate for optimal printing orientation
rotate([90, 0, 90])
    parametric_leg();