/* BamaKey TP-II case - designed by DF5WC */

/* Which object to generate:
 * 0 = lower case
 * 1 = lid
 * 2 = call (if defined)
 * 3 = lid with call (for debugging only, not printable)
 */
what = 0;

/* Call sign. An empty string excludes call the sign. */
call = "";
call_z = 0.4;

/* Case wall width */
wall = 3.0;

/* Size of the key compartment */
key_x = 91.0;
key_y = 71.0;
key_z = 34.0;

/* Size of the paddle compartment */
paddle_x = 29.0;
paddle_y = 20.0;
paddle_z = key_z;

/* Size of the cable compartment */
cable_x = 50.0;
cable_y = key_y;
cable_z = key_z;

/* Breakout for the cable */
breakout_x = wall;
breakout_y = 25.0;
breakout_z = key_z;

/* Outer dimensions of the case */
lower_x = paddle_x + key_x + cable_x + 3*wall;
lower_y = key_y + 2*wall;
lower_z = key_z + wall;

/* Gap on each side between lower case and lid */
gap = 0.4;

/* Lid wall width */
lid_wall = 2.0;

/* Outer dimensions of the lid */
lid_x = lower_x + 2*gap + 2*lid_wall;
lid_y = lower_y + 2*gap + 2*lid_wall;
lid_z = lower_z + lid_wall;     /* No gap here */

/* Closure */
closure_x = 5.0;
closure_y = 1.0;
closure_z = 3.0;

/* Hidden */
$fn  =  100;
eps  =  0.005;

module roundedplate(x, y, z, rad = 0.75) {
    linear_extrude(height = z) {
        hull() {
            translate([rad, rad, 0]) circle(r = rad);
            translate([x - rad, rad, 0]) circle(r = rad);
            translate([x - rad, y - rad, 0]) circle(r = rad);
            translate([rad, y - rad, 0]) circle(r = rad);
        }
    }
}

module callsign(height) {
    translate([0, 0, height]) {
        rotate([180, 0, 0]) {
            linear_extrude(height) {
                text(call, size=30.0, font="Nimbus Mono L:style=Bold", halign="center", valign="center");
            }
        }
    }
}

module grip_opening() {
    rad = 7.5;
    rotate([90, 0, 180]) {
        linear_extrude(height = lid_wall + 2*eps) {
            scale([2.0, 1.0]) {
                circle(r = rad);
            }
        }
    }
}

module closure() {
    l = closure_x - gap;
    translate([-l/2, 0, 0]) {
        rotate([90, 0, 90]) {
            linear_extrude(height = l) {
                polygon(points=[
                    [-eps, 0.0],
                    [closure_y, 0.5],
                    [closure_y, closure_z - 0.5],
                    [-eps, closure_z],
                    [-eps, 0.0],
                ]);
            }
        }
    }
}

module lower() {
    difference() {
        /* Lower case corpus */
        roundedplate(lower_x, lower_y, lower_z, 5);
        /* Paddle compartment */
        translate([wall, wall + (key_y - paddle_y)/2, wall]) {
            cube([paddle_x + eps, paddle_y, paddle_z + eps]);
        }
        /* Key compartment */
        translate([wall + paddle_x, wall, wall]) {
            roundedplate(key_x, key_y, key_z + eps, 5);
        }
        /* Cable compartment */
        translate([wall + paddle_x + key_x + wall, wall, wall]) {
            roundedplate(cable_x, cable_y, cable_z + eps, 5);
        }
        /* Breakout between cable and key compartment */
        translate([wall + paddle_x + key_x - eps, wall + (key_y - breakout_y)/2, wall]) {
            cube([breakout_x + 2*eps, breakout_y, breakout_z + eps]);
        }
        /* Closure holes */
        translate([lower_x / 2 - 2.5, -eps, lower_z - 15]) {
            cube([closure_x, closure_y + eps, closure_z]);
        }
        translate([lower_x / 2 - 2.5, lower_y - 1.0, lower_z - 15]) {
            cube([closure_x, closure_y + eps, closure_z]);
        }
    }
}

module upper() {
    union() {
        difference() {
            /* Lid corpus */
            roundedplate(lid_x, lid_y, lid_z, 5);
            /* Inner */
            translate([lid_wall, lid_wall, lid_wall]) {
                roundedplate(lid_x - 2*lid_wall, lid_y - 2*lid_wall, lid_z - lid_wall + eps, 5);
            }
            /* Grip openings */
            translate([lid_x / 2, -eps, lid_z]) {
                grip_opening();
            }
            translate([lid_x / 2, lid_y - lid_wall - eps, lid_z]) {
                grip_opening();
            }
            /* Call sign cavity */
            if (len(call) > 0) {
                translate([lid_x / 2, lid_y / 2, - eps]) {
                    callsign(call_z + eps);
                }
            }
        }
        /* Support */
        translate([lid_wall + wall + 2*gap + paddle_x, lid_wall + wall + 2*gap, lid_wall - eps]) {
            roundedplate(key_x - 2*gap, key_y - 2*gap, 2 - gap + eps, 5);
        }
        /* Closures */
        translate([lid_x / 2, lid_wall, lid_wall + 15 - closure_z]) {
            closure();
        }
        translate([lid_x / 2, lid_y - lid_wall, lid_wall + 15 - closure_z]) {
            rotate([0, 0, 180]) {
                closure();
            }
        }
        /* Call sign */
        *translate([lid_x / 2, lid_y / 2, call_z - eps]) {
            label(call);
        }
    }
}

if (what == 0) {
    lower();
}
if (what == 1 || what == 3) {
    upper();
}

if (what == 2 || what == 3) {
    translate([lid_x / 2, lid_y / 2, 0]) {
        color("orange") {
            callsign(call_z);
        }
    }
}


