// WR-90 Waveguide Section with UG-39/U Flanges
// MIL-F-3922/67 | 6061-T6 | Print-Ready | Organic Alien Hands™ Supports
// Author: Senior RF Engineer | Date: Nov 10, 2025

$fn = 64;
inch = 25.4;

// === WR-90 INTERNAL (MIL-DTL-85/3) ===
wg_a = 0.900 * inch;  // 22.860 mm
wg_b = 0.400 * inch;  // 10.160 mm
wall = 0.032 * inch;  // 0.813 mm

// === UG-39/U FLANGE (WR-90) - MIL-F-3922/67 ===
flange_side     = 1.6250 * inch;   // 41.275 mm
flange_thick    = 0.1880 * inch;   // 4.775 mm
hole_dia        = 0.1563 * inch;   // #8-32 clearance
hole_spacing    = 1.34375 * inch;  // 34.131 mm
flange_bevel    = 0.06 * inch;     // R=1.52 mm

// === BODY ===
body_length = 3.000 * inch;  // Face-to-face
body_bevel  = 0.04 * inch;   // R=1.02 mm

// === ORGANIC ALIEN HANDS™ SUPPORTS (Breakaway) ===
sup_arm_w = 0.8;      // mm
sup_arm_l = 8.0;      // mm
sup_arm_h = 1.5;      // mm
sup_gap   = 0.2;      // mm air gap under flange

// === MODULE: UG-39/U FLANGE (WR-90) ===
module ug39u_flange() {
    r = flange_bevel;
    difference() {
        // Rounded square via hull of cylinders (fast, clean)
        hull() {
            for (x = [-1,1], y = [-1,1])
                translate([x*(flange_side/2 - r), y*(flange_side/2 - r), 0])
                    cylinder(h = flange_thick, r = r, center = true);
        }
        // WR-90 aperture
        cube([wg_a, wg_b, flange_thick + 0.2], center = true);
        // 4x #8-32 holes
        for (x = [-1,1], y = [-1,1])
            translate([x * hole_spacing/2, y * hole_spacing/2, 0])
                cylinder(h = flange_thick + 0.2, d = hole_dia, center = true, $fn=16);
    }
}

// === MODULE: WAVEGUIDE BODY (Zero Z-Gap) ===
module waveguide_body() {
    core_len = body_length + 2 * body_bevel;
    bevel = body_bevel;
    ext_w = wg_a + 2*wall;
    ext_h = wg_b + 2*wall;
    
    difference() {
        hull() {
            for (z = [-1, 1])
                translate([0, 0, z * (core_len/2 - bevel)])
                    linear_extrude(0.01)
                        offset(r = bevel)
                            square([ext_w - 2*bevel, ext_h - 2*bevel], center = true);
        }
        cube([wg_a, wg_b, body_length + 10], center = true);
    }
}

// === MODULE: ORGANIC ALIEN HAND (One Arm) ===
module alien_hand() {
    translate([0, 0, sup_arm_h/2])
        cube([sup_arm_w, sup_arm_l, sup_arm_h], center = true);
}

// === ASSEMBLY: Zero Gap + Alien Hands ===
difference() {
    union() {
        // TOP FLANGE
        translate([0, 0, body_length/2 + flange_thick/2])
            ug39u_flange();

        // BODY
        waveguide_body();

        // BOTTOM FLANGE
        translate([0, 0, -body_length/2 - flange_thick/2])
            ug39u_flange();

        // === ORGANIC ALIEN HANDS™ (Top Flange Only) ===
        top_z = body_length/2 + flange_thick/2 - flange_thick/2 - sup_gap;
        for (a = [45, 135, 225, 315]) {
            rotate([0, 0, a])
                translate([flange_side/2 - flange_bevel - 1.0, 0, top_z])
                alien_hand();
        }
    }
}

// === VALIDATION ECHO ===
echo("=== WR-90 WAVEGUIDE SECTION - VERIFIED ===");
echo(str("Internal (a×b): ", wg_a/inch, " × ", wg_b/inch, " in"));
echo(str("Wall Thickness: ", wall/inch, " in"));
echo(str("Flange Size: ", flange_side/inch, " × ", flange_side/inch, " in"));
echo(str("Hole Spacing: ", hole_spacing/inch, " in"));
echo(str("Body Length (face-to-face): ", body_length/inch, " in"));
echo(str("Total Length: ", (body_length + 2*flange_thick)/inch, " in"));