// you can optionally use this model to replace the wheel produced by the other file.
// the other file produces engraved numbers which can be filled with paint.
// this model is a two-pass method of embedding colored text in the print without the need for multimaterial printing.

// the idea here is that the digits are roughly a single layer thick. 
// print the numbers by themselves with one color, then print the wheel directly over them.
// the overlap isn't enough to cause problems (at least not on a flimsy machine like mine).
// use a heavily-pigmented (e.g. black) filament for the text, otherwise the it may be faint.

// the STL files should be such that the two objects are coaxial and oriented face-down on the bed.
// set orientforprint = true and then use whichtoprint to export each object separately.

// clearance on wheel
clwheel = 0.45; // dia

// height of text object to be embedded (approx 1 layer height)
textd = 0.2;

// view controls
orientforprint = false;
whichtoprint = "digits"; // "wheel" or "digits"
zoffset = 10;
$fn = 100;

// ////////////////////////////////////////////////////

// draw everything
if (orientforprint) {
	if (whichtoprint == "wheel"){
		translate([0,0,3])
			rotate([0,180,0])
			wheel();
	}
	else if (whichtoprint == "digits"){
		translate([0,0,3])
			rotate([0,180,0])
			justnumbers();
	}
}
else{
	// for visualization
	translate([0,0,zoffset])
		justnumbers();
	translate([0,0,0])
		wheel();
}



// n _must_ be an integer multiple of the number of digits on the wheel
// 20 or 30 works fine for 50% DC; changing DC requires tweaking offset angle
// but the original 25 means only every other number will line up with the window.
pcr = 16.7; // pitch circle radius
n = 20; // number of cycles
p = [1,1]*1; // order of each half cycle [outer inner]
s = [1,1]*0.5; // magnitude of each half cycle [outer inner]
dc = 0.5; // duty cycle
nph = 20; // points per cycle
wheelh = 3;

module wheel(){
	difference(){
		translate([0,0,wheelh/2])
			rotate([0,0,360/n/4])
			splilinder(wheelh,pcr,n,p,s,dc,nph);
		cylinder(d = 8+clwheel,h = 10,center=true); // cut a new hole with specified clearance
	}
}

module justnumbers(){
	translate([0,0,3])
		scale([1,1,textd/1.154])
		translate([0,0,-3])
		difference(){
			cylinder(r = 16,h = 3,center=false);
			unfuckedwheel();
		}
}

module unfuckedwheel(){
	// unfuck the wheel
	// why does everyone throw parts in an STL with random location and rotation?
	// the bottom of the wheel isn't even flat or positioned on z=0 for some reason, and it's not a clearance offset either.  
	// the top is flat and positioned at z=3.  
	// the bottom z-position values are uniformly-distributed random numbers between 6.7E-15 and 1.5E-14.
	// it's curious even for float rounding error.  
	union(){
		intersection(){
			rotate([0,0,-21.0095]) // fix the random rotation
				translate([0,0,3]) // shift back up
				scale([1,1,1.001]) // stretch it a bit so that the bottom gets cut off
				translate([20.0415,44.9895,-3]) // center and orient on top face
				import("../original_model_4867466/gc_wheel.stl",convexity = 5); // ugh
			cylinder(r = 16,h = 10,center=false); // chop off the wrong knurling and the irregular bottom
		}
		cylinder(d = 9,h = 3,center=false); // fill the oversize center hole
	}
}








// creates a piecewise power function/sinusoid composition
// around a circular path (sort of like a gear or spline shape)
// pcr: pitch circle radius
// n: number of cycles
// p: order of each half cycle (2-element vector)
// s: magnitude of each half cycle (2-element vector)
// dc: duty cycle (0-1)
// nph: points per half cycle
module splinusoid(pcr, n, p=[1,1], s=[1,1], dc=0.5, nph=$fn) { 
	// calculate some things
	np = 2*nph;

	// generate the radius for one cycle (uniformly-spaced, equal-length half cycles)
	R1 = [for (thh1 = linsegment(0,180,nph)) pcr + s[0]*pow(abs(sin(thh1)),1/p[0]) ];
	R2 = [for (thh2 = linsegment(180,360,nph)) pcr - s[1]*pow(abs(sin(thh2)),1/p[1]) ];
	R = concat(R1,R2);

	// generate a sequence of nonuniformly-spaced angles covering one cycle
	th1 = [for (th = linsegment(0,360/n*dc,nph)) th ];
	th2 = [for (th = linsegment(360/n*dc,360/n,nph)) th ];
	th = concat(th1,th2);
	
	// generate the pointlist and create the pgon
	pointlist = 	[for (k = [0:np*n-1]) 
					getpoint(R[k%np],floor(k/np)*360/n + th[k%np]) ];
	closedpointlist = concat(pointlist,[pointlist[0]]);
	polygon(closedpointlist);

	function getpoint(r,th) = [r*cos(th),r*sin(th)];
}

module splilinder(h, pcr, n, p=[1,1], s=[1,1], dc=0.5, np=360/$fa) { 
	linear_extrude(height = h, center = true, convexity = 10, twist = 0)
		splinusoid(pcr, n, p, s, dc, np);
}

// generate a linear sequence of N values
// starting at A and ending one step size prior to B
// useful for generating adjacent sequence vectors without redundant elements
function linsegment(a,b,n) = 
	[a : (b-a)/n : (a + (b-a)/n*(n-0.5))];

// generate a linear sequence of N values
// starting at A and ending at B
function linspace(a,b,n) = 
	[a : (b-a)/(n-1):  (a + (b-a)/(n-1)*(n-0.5))];


























