/*************************************************************************************
 *
 * Drawer module
 *
 *************************************************************************************
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHOR OR COPYRIGHT
 * HOLDER BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * IT IS NOT PERMITTED TO MODIFY THIS COMMENT BLOCK.
 *
 * (c)2025, Claude "Tryphon" Theroux, Montreal, Quebec, Canada
 * http://www.ctheroux.com/
 *
 *************************************************************************************/

merged_code = true;

// Module width in mm
ModuleWidth = 130.5;

// Module depth in mm
ModuleDepth = 155;

// Module height in mm
ModuleHeight = 95.5;

LCTFilamentAttachmentDumpEnabled = false;

// an array of [ id, [ internal width, internal height, internal depth ] ]
// All measurements are in mm
// SideModels available are: 0, 1
// BackModels available are: 0, 1
DrawerModule = [
    [   0,
        [
            // Empty intentionnaly
        ]
    ]
    ,
    [   1,
        [
            [ "width", 119.1 ],
            [ "height", 12.6 ], 
            [ "depth", 150.9 ],
            [ "XOffset", 0 ],
            [ "YOffset", 0 ],
            [ "SideModel", 1 ],
            [ "BackModel", 0 ],
            [ "SeparatorOnWidth", true ],
            [ "SeparatorNumber", 4 ],
            [ "TopPlate", true ],
            [ "TransversalSeparators", 
                [
                    [
                        [ "LeftCount", 0 ],
                        [ "LeftUnitCount", 0 ],
                        [ "RightCount", 2 ],
                        [ "RightUnitCount", 1 ]
                    ]
                ]
            ]
                                      
        ]
    ]
    ,
    [   2,
        [
            [ "width", 119.1 ],
            [ "height", 32.6 ], 
            [ "depth", 150.9 ],
            [ "XOffset", 0 ],
            [ "YOffset", 15.1 ],
            [ "SideModel", 2 ],
            [ "BackModel", 0 ],
            [ "SeparatorOnWidth", true ],
            [ "SeparatorNumber", 4 ],
            [ "TransversalSeparators", 
                [
                    [
                        [ "LeftCount", 1 ],
                        [ "LeftUnitCount", 2 ],
                        [ "RightCount", 2 ],
                        [ "RightUnitCount", 1 ]
                    ],
                    [
                        [ "LeftCount", 0 ],
                        [ "LeftUnitCount", 2 ],
                        [ "RightCount", 3 ],
                        [ "RightUnitCount", 1 ]
                    ],
                    [
                        [ "LeftCount", 0 ],
                        [ "LeftUnitCount", 2 ],
                        [ "RightCount", 2 ],
                        [ "RightUnitCount", 1 ]
                    ]
                ]
            ]
        ]
    ],
    [   3,
        [
            [ "width", 119.1 ],
            [ "height", 37.5 ], 
            [ "depth", 150.9 ],
            [ "XOffset", 0 ],
            [ "YOffset", 50.2 ],
            [ "SideModel", 2 ],
            [ "BackModel", 0 ],
            [ "SeparatorOnWidth", true ],
            [ "SeparatorNumber", 4 ],
            [ "TransversalSeparators", 
                [ 
                    [
                        [ "LeftCount", 1 ],
                        [ "LeftUnitCount", 2 ],
                        [ "RightCount", 0 ],
                        [ "RightUnitCount", 2 ]
                    ],
                    [
                        [ "LeftCount", 0 ],
                        [ "LeftUnitCount", 1 ],
                        [ "RightCount", 0 ],
                        [ "RightUnitCount", 2 ]
                    ]
                ]
            ]
        ]
    ]
];

/*************************************************************************************
 *
 * Nothing shall be modified below this line
 *
 *************************************************************************************/

function lct_VectorSum(p_Vector, p_UpperLimit = undef, p_Index = 0, p_CurrentSum = 0) = (len(p_Vector) == 0 ? undef : (p_UpperLimit == undef || p_UpperLimit + 1 > len(p_Vector) ? lct_VectorSumInternal(p_Vector, len(p_Vector) - 1, p_Index, p_CurrentSum) : lct_VectorSumInternal(p_Vector, p_UpperLimit, p_Index, p_CurrentSum)));

function lct_VectorSumInternal(p_Vector, p_UpperLimitp_Index, p_Index = 0, p_CurrentSum = 0) = p_Index <= p_UpperLimitp_Index ? lct_VectorSumInternal(p_Vector, p_UpperLimitp_Index, p_Index + 1, p_CurrentSum + p_Vector[p_Index]) : p_CurrentSum;

function lct_VectorMaximum(p_Vector, p_UpperLimit = undef, p_Index = 0, currentMaximum = 0) = (len(p_Vector) == 0 ? undef : (p_UpperLimit == undef || p_UpperLimit + 1 > len(p_Vector) ? lct_VectorMaximumInternal(p_Vector, len(p_Vector) - 1, p_Index, currentMaximum) : lct_VectorMaximumInternal(p_Vector, p_UpperLimit, p_Index, currentMaximum)));

function lct_VectorMaximumInternal(p_Vector, p_UpperLimitp_Index, p_Index = 0, currentMaximum = 0) = p_Index <= p_UpperLimitp_Index ? lct_VectorMaximumInternal(p_Vector, p_UpperLimitp_Index, p_Index + 1, (currentMaximum > p_Vector[p_Index] ? currentMaximum : p_Vector[p_Index] )) : currentMaximum;

/*************************************************************************************
 *
 * Library inner parameters
 *
 *************************************************************************************/

// Clearance between a male and a female connector in mm
lct_MaleFemaleConnectorClearance = 0.1;

// Male connector length in mm
lct_MaleConnectorLength = 22;

// Male connector width in mm
lct_MaleConnectorWidth = 12;

// Male connector height in mm
lct_MaleConnectorHeight = 3.5;

// Filament hole diameter in mm
lct_FilamentHoleDiameter = 2.2;


/*************************************************************************************
 *
 * Nothing shall be modified below this line
 *
 *************************************************************************************/
 
$fn = 60;

function ratio(p_value, p_height) = (p_value - 2 * p_height) / p_value;

// Female connector length in mm
function lct_FemaleConnectorLength(p_length = lct_MaleConnectorLength) = p_length + lct_MaleFemaleConnectorClearance;

// Female connector width in mm
function lct_FemaleConnectorWidth(p_width = lct_MaleConnectorWidth) = p_width + lct_MaleFemaleConnectorClearance;

// Female connector height in mm
function lct_FemaleConnectorHeight(p_height = lct_MaleConnectorHeight) = p_height + lct_MaleFemaleConnectorClearance;

module lct_FilamentAttachmentDump(p_EnableDebug = false, p_length = lct_MaleConnectorLength, p_width = lct_MaleConnectorWidth, p_height = lct_MaleConnectorHeight) {
    if( p_EnableDebug ) {
        echo("vvv LCTFilamentAttachment:lct_FilamentAttachmentDump");
        echo("    lct_FilamentHoleDiameter:", lct_FilamentHoleDiameter);
        echo("    Male connector length:", p_length);
        echo("    Male connector width:", p_width);
        echo("    Male connector height:", p_height);
        echo("    Female connector length with clearance:", p_length + lct_MaleFemaleConnectorClearance);
        echo("    Female connector width with clearance:", p_width + lct_MaleFemaleConnectorClearance);
        echo("    Female connector height with clearance:", p_height + lct_MaleFemaleConnectorClearance);
        echo("    lct_MaleConnectorTopLengthRatio:", ratio(p_length, p_height));
        echo("    lct_MaleConnectorTopWidthRatio:", ratio(p_width, p_height));
        echo("^^^ LCTFilamentAttachment:lct_FilamentAttachmentDump");
    }
}

module lct_MaleConnector(p_length = lct_MaleConnectorLength, p_width = lct_MaleConnectorWidth, p_height = lct_MaleConnectorHeight) {
    difference() {
        linear_extrude(height = p_height, scale = [ ratio(p_length, p_height), ratio(p_width, p_height) ]) square([ p_length, p_width ], true);
    translate([ -p_length / 2, 0, p_height / 2]) rotate([ 0, 90, 0 ]) cylinder(h = p_length, d = lct_FilamentHoleDiameter);
    }
}

module lct_FemaleConnectorKnockout(p_length = lct_MaleConnectorLength + lct_MaleFemaleConnectorClearance, p_width = lct_MaleConnectorWidth + lct_MaleFemaleConnectorClearance, p_height = lct_MaleConnectorHeight + lct_MaleFemaleConnectorClearance) {
    difference() {
        linear_extrude(height = p_height, scale = [ ratio(p_length, p_height), ratio(p_width, p_height) ]) square([ p_length, p_width ], true);
    }
}



/*************************************************************************************
*
* Library inner parameters
*
*************************************************************************************/

// Drawer holder structure thickness in mm
lct_DrawerHolderBaseThickness = 2;

// Drawer holder structure thickness in mm (DrawerHolderStructureThickness)
lct_DrawerHolderSideThickness = 2;

// Drawer holder structure width in mm
lct_DrawerHolderStructureWidth = 10;

// Drawer holder tolerance in mm for the drawers
lct_DrawerHolderToleranceForTheDrawers = 0.5;

// Drawer bottom thickness in mm
lct_DrawerBottomThickness = 1.6;

// Drawer sides thickness in mm
lct_DrawerSidesThickness = 1.2;

// Drawer handle width in mm
lct_DrawerHandleWidth = 20;

// Drawer handle length in mm
lct_DrawerHandleLength = 10;

// Drawer handle height in mm
lct_DrawerHandleHeight = 5;

// Drawer handle thickness in mm
lct_DrawerHandleThickness = 2;

// Minimum drawer handle width in mm
lct_MinimumDrawerHandleWidth = 10;

// Separator thickness in mm
lct_SeparatorThickness = 1.6;

// Separator holder thickness in mm
lct_SeparatorHolderThickness = 2;

// Separator holders distance in mm
lct_SeparatorHolderDistance = 2;

// Separator tolerance in mm
lct_SeparatorTolerance = 0.5;

// Label card tolerance in mm
lct_LabelCardTolerance = 1;

// Label card maximum width in mm
lct_LabelCardMaximumWidth = 76.2;

// Label card maximum height in mm
lct_LabelCardMaximumHeight = 127 / 3;

// Label holder thickness in mm
lct_LabelHolderThickness = 1.5;

// Label holder media thickness in mm
lct_LabelHolderMediaThickness = 0.75;

// Label holder and drawer handle clearance in mm
lct_LabelHolderDrawerHandleClearance = 1;

// Outer shell thickness in mm
lct_OuterShellThickness = 1.2;

// Outer shell tolerance in mm
lct_OuterShellClearance = 0.4;

// Distance between objects in mm
lct_DistanceBetweenObjects = 20;

// Tab clearance related to height and width in mm
lct_TabClearance = 0.2;

// Tab width in mm
lct_TabWidth = 10;

// Tab offset from side in mm
lct_TabOffset = 15;

// Set to true to display debugging information on the console.  false to disable it.
Debug = true;

/*************************************************************************************
*
* Nothing shall be modified below this line
*
*************************************************************************************/

function lctLDS_GetAttribute(p_Attributes, p_Name) = p_Attributes[search([p_Name], p_Attributes)[0]][1];


function lct_GetDrawerId(p_drawerModules, p_index) = (p_drawerModules[p_index][0]);

function lct_GetDrawerModuleSpecifications(p_drawerModules, p_drawerId) = (p_drawerModules[search(p_drawerId, p_drawerModules)[0]][1]);

function lct_GetDrawerModuleAttribute(p_drawerModules, p_drawerId, p_attributeName) = lct_GetDrawerModuleSpecifications(p_drawerModules, p_drawerId)[search([p_attributeName], lct_GetDrawerModuleSpecifications(p_drawerModules, p_drawerId))[0]][1];

function lct_DrawerInnerDimensions() = [ 1, 2, 3 ];
 
module lctLDS_GenerateXBrace(p_width, p_height, p_thickness) {
    if( p_height - lct_DrawerHolderStructureWidth * 2 > 0 && p_width - lct_DrawerHolderStructureWidth * 2 > 0 ) {
        braceLength = sqrt((p_width - lct_DrawerHolderStructureWidth * 2) ^ 2 + (p_height - lct_DrawerHolderStructureWidth * 2) ^ 2);
        braceAngle = atan((p_width - lct_DrawerHolderStructureWidth * 2) / (p_height - lct_DrawerHolderStructureWidth * 2));
        translate([ lct_DrawerHolderStructureWidth, lct_DrawerHolderStructureWidth, 0]) rotate([ 0, 0, -braceAngle ]) translate([ -lct_DrawerHolderStructureWidth / 2, 0, 0]) cube([ lct_DrawerHolderStructureWidth, braceLength + 0, p_thickness ]);
        translate([ lct_DrawerHolderStructureWidth, (p_height - lct_DrawerHolderStructureWidth), 0]) rotate([ 0, 0, -(180 - braceAngle) ]) translate([ -lct_DrawerHolderStructureWidth / 2, 0, 0]) cube([ lct_DrawerHolderStructureWidth, braceLength + 0, p_thickness ]);
    } else {
        cube( [ p_width, p_height, p_thickness ] );
    }
}

module lctLDS_GenerateVBracePlate(p_width, p_height, p_thickness) {
    
    innerWidth = p_width - 2 * lct_DrawerHolderStructureWidth;

    if( innerWidth > 0 ) {
        difference() {
            cube( [ p_width, p_height, p_thickness ] );
            linear_extrude(p_thickness) polygon( [ 
                [ lct_DrawerHolderStructureWidth, lct_DrawerHolderStructureWidth ],
                [ p_width - lct_DrawerHolderStructureWidth, lct_DrawerHolderStructureWidth ],
                [ p_width - lct_DrawerHolderStructureWidth, p_height - lct_DrawerHolderStructureWidth - innerWidth / 2 ],
                [ p_width / 2, p_height - lct_DrawerHolderStructureWidth ], 
                [ lct_DrawerHolderStructureWidth, p_height - lct_DrawerHolderStructureWidth - innerWidth / 2 ] 
            ] );
            
        }
    } else {
        cube( [ p_width, p_height, p_thickness ] );
    }
}

module lctLDS_GeneratePlate(p_width, p_height, p_thickness, p_plateModel) {
    if( p_plateModel == 0 ) {
        difference() {
            cube([p_width, p_height, p_thickness]);
            translate([ lct_DrawerHolderStructureWidth, lct_DrawerHolderStructureWidth, 0 ]) cube([p_width - lct_DrawerHolderStructureWidth * 2, p_height - lct_DrawerHolderStructureWidth * 2, p_thickness]);
        }
        lctLDS_GenerateXBrace(p_width, p_height, p_thickness);
    } else if ( p_plateModel == 1 ) {
        lctLDS_GenerateVBracePlate(p_width, p_height, p_thickness);
    } else if ( p_plateModel == 2 ) {
        cube( [ p_width, p_height, p_thickness ] );
    }
}

module lctLDS_GenerateDrawerHolderTab(p_TabWidth, p_TabHeight) {
    cube([ lct_DrawerHolderSideThickness, p_TabWidth, p_TabHeight ]);
}

module lctLDS_GenerateDrawerHolderTabs(p_width, p_TabHeight, p_depth, p_TabWidth, p_TabOffset, p_TabSize) {
    
    translate([ 0, p_TabOffset - p_TabWidth / 2, p_TabHeight ]) lctLDS_GenerateDrawerHolderTab(p_TabWidth, p_TabSize);
    
    translate([ p_width - lct_DrawerHolderSideThickness, p_TabOffset - p_TabWidth / 2, p_TabHeight ])  lctLDS_GenerateDrawerHolderTab(p_TabWidth, p_TabSize);
    
    translate([ 0, p_depth - p_TabWidth / 2 - p_TabOffset, p_TabHeight ])  lctLDS_GenerateDrawerHolderTab(p_TabWidth, p_TabSize);
    
    translate([ p_width - lct_DrawerHolderSideThickness, p_depth - p_TabWidth / 2 - p_TabOffset, p_TabHeight ]) lctLDS_GenerateDrawerHolderTab(p_TabWidth, p_TabSize);
}

module lctLDS_GenerateDrawerHolderTopPlate(p_DrawerId, p_Drawer) {
    if( lctLDS_GetAttribute(p_Drawer, "TopPlate") ) {
        
        width = lctLDS_GetAttribute(p_Drawer, "width") + 2 * lct_DrawerHolderSideThickness + lct_DrawerHolderToleranceForTheDrawers;
        depth = lctLDS_GetAttribute(p_Drawer, "depth") + lct_DrawerHolderSideThickness + lct_DrawerHolderToleranceForTheDrawers;

        echo("    ==> LCTLightDrawerSystem:lctLDS_GenerateDrawerHolderTopPlate");
        echo(str("        Top plate width in mm: ", width));
        echo(str("        Top plate depth in mm: ", depth));
        
        difference() {
            lctLDS_GeneratePlate(width, depth, lct_DrawerHolderBaseThickness, 0);
            lctLDS_GenerateDrawerHolderTabs(width, 0, depth, lct_TabWidth + lct_TabClearance, lct_TabOffset, lct_DrawerHolderBaseThickness);
        }
    }
}

module lctLDS_GenerateDrawerHolder(p_DrawerId, p_Drawer) {
    
    externalWidth = lctLDS_GetAttribute(p_Drawer, "width") + 2 * lct_DrawerHolderSideThickness + lct_DrawerHolderToleranceForTheDrawers;
    externalDepth = lctLDS_GetAttribute(p_Drawer, "depth") + lct_DrawerHolderBaseThickness + lct_DrawerHolderToleranceForTheDrawers;
    externalHeight = lctLDS_GetAttribute(p_Drawer, "height") + 2 * lct_DrawerHolderSideThickness + lct_DrawerHolderToleranceForTheDrawers;
    
    sidePlateModel = lctLDS_GetAttribute(p_Drawer, "SideModel");
    backPlateModel = lctLDS_GetAttribute(p_Drawer, "BackModel");
   
    echo("    ==> LCTLightDrawerSystem:lctLDS_GenerateDrawerHolder");
    echo(str("        Drawer holder side thickness in mm: ", lct_DrawerHolderSideThickness));
    echo(str("        Drawer holder clearance for the drawers in mm: ", lct_DrawerHolderToleranceForTheDrawers));
    echo(str("        Drawer holder external width in mm: ", externalWidth));
    echo(str("        Drawer holder external depth in mm: ", externalDepth));
    echo(str("        Drawer holder external height in mm: ", externalHeight));
    echo(str("        Drawer holder internal width in mm: ", externalWidth - 2 * lct_DrawerHolderSideThickness));    
    echo(str("        Drawer holder internal depth in mm: ", externalDepth - lct_DrawerHolderBaseThickness));
    echo(str("        Drawer holder internal height in mm: ", externalHeight - 2 * lct_DrawerHolderSideThickness));    
    echo(str("        Side plate model: ", sidePlateModel));
    echo(str("        Back plate model: ", backPlateModel));
    echo(str("        Tab width in mm: ", lct_TabWidth));
    echo(str("        Tab center offset in mm: ", lct_TabOffset));
    echo(str("        Tab clearance in mm: ", lct_TabClearance));
    
    difference() {
        union() {
            // Generate base plate
            lctLDS_GeneratePlate(externalWidth, externalDepth, lct_DrawerHolderBaseThickness, backPlateModel);
            
            // Generate sides
            translate([ lct_DrawerHolderSideThickness, lct_DrawerHolderSideThickness, lct_DrawerHolderBaseThickness ]) 
            rotate([0, 270, 0]) lctLDS_GeneratePlate(externalHeight - lct_DrawerHolderBaseThickness * 2, externalDepth - lct_DrawerHolderSideThickness,  lct_DrawerHolderSideThickness, sidePlateModel);
            
            translate([ externalWidth, lct_DrawerHolderSideThickness, lct_DrawerHolderBaseThickness ]) rotate([ 0, 270, 0 ]) lctLDS_GeneratePlate(externalHeight - lct_DrawerHolderBaseThickness * 2, externalDepth - lct_DrawerHolderSideThickness, lct_DrawerHolderSideThickness, sidePlateModel);
            
            // Generate the back plate
            translate([ 0, 0, lct_DrawerHolderBaseThickness ]) rotate([ 0, -90, -90 ]) lctLDS_GeneratePlate(externalHeight - lct_DrawerHolderBaseThickness * 2, externalWidth, lct_DrawerHolderSideThickness, sidePlateModel);
            
            // Generate top tabs
            lctLDS_GenerateDrawerHolderTabs(externalWidth, externalHeight - lct_DrawerHolderBaseThickness, externalDepth, lct_TabWidth, lct_TabOffset, lct_DrawerHolderBaseThickness - lct_TabClearance);
        }
        // Generate bottom tabs
        lctLDS_GenerateDrawerHolderTabs(externalWidth, 0, externalDepth, lct_TabWidth + lct_TabClearance, lct_TabOffset, lct_DrawerHolderBaseThickness + 0.1 );
    }
}

module lctLDS_GenerateLabelHolder(p_width, p_height) {

    effectiveWidth = (p_width > lct_LabelCardMaximumWidth + lct_LabelHolderThickness * 2 + lct_LabelCardTolerance ? lct_LabelCardMaximumWidth + lct_LabelHolderThickness * 2 + lct_LabelCardTolerance : p_width);
    effectiveHeight = (p_height > lct_LabelCardMaximumHeight + lct_LabelHolderThickness + lct_LabelCardTolerance ? lct_LabelCardMaximumHeight + lct_LabelHolderThickness + lct_LabelCardTolerance : p_height);
    
    echo("    vvv LCTLightDrawerSystem:lctLDS_GenerateLabelHolder");
    echo(str("        Label card width in mm: ", effectiveWidth - lct_LabelHolderThickness * 2 - lct_LabelCardTolerance));
    echo(str("        Label card height in mm: ", effectiveHeight - lct_LabelHolderThickness - lct_LabelCardTolerance));
    
    translate( [ (p_width - effectiveWidth) / 2, 0, p_height - effectiveHeight ] ) difference() {
        union() {
            translate( [ effectiveWidth, lct_LabelHolderThickness * 2, 0 ] ) rotate( [ 90, 0, -90 ] ) linear_extrude(effectiveWidth) polygon( [ [ 0, 0 ], [ lct_LabelHolderThickness, lct_LabelHolderThickness ], [ lct_LabelHolderThickness, lct_LabelHolderThickness * 2 ], [ 0, lct_LabelHolderThickness * 2 ], [ 0, 0] ] );
            translate( [ 0, lct_LabelHolderThickness, lct_LabelHolderThickness * 2 ] ) cube( [ effectiveWidth, lct_LabelHolderThickness, effectiveHeight - 2 * lct_LabelHolderThickness ] );
        }
        union() {
            translate( [ lct_LabelHolderThickness * 2, lct_LabelHolderThickness, lct_LabelHolderThickness * 2] ) cube([ effectiveWidth - lct_LabelHolderThickness * 4, lct_LabelHolderThickness, effectiveHeight - lct_LabelHolderThickness * 2]);
            translate( [ lct_LabelHolderThickness, lct_LabelHolderThickness * 2 - lct_LabelHolderMediaThickness, lct_LabelHolderThickness] ) cube([ effectiveWidth - lct_LabelHolderThickness * 2, lct_LabelHolderMediaThickness, effectiveHeight - lct_LabelHolderThickness]);
        }
    }
}

module lctLDS_GenerateDrawerHandle(p_width) {
    $fn = 60;
    cube([ p_width, lct_DrawerHandleLength, lct_DrawerHandleThickness]);
    if ( lct_DrawerHandleHeight - lct_DrawerHandleThickness * 2 > 0 ) {
        translate( [ 0, 0, lct_DrawerHandleThickness ] ) cube( [ p_width, lct_DrawerHandleThickness * 2, lct_DrawerHandleHeight - lct_DrawerHandleThickness * 2 ] );
    }
    translate( [ 0, lct_DrawerHandleThickness, lct_DrawerHandleHeight - lct_DrawerHandleThickness ] ) rotate( [ 0, 90, 0 ] ) cylinder(h = p_width, r = lct_DrawerHandleThickness);
}

module lctLDS_GenerateTransversalSeparator(p_Length, p_Height, p_TransversalUnitWidth, p_LeftCount, p_RightCount, p_LeftUnitCount, p_RightUnitCount) {   
    
    echo("    vvv LCTLightDrawerSystem:lct_GenerateSeparatorType_0");
    echo(str("        Drawer separator length in mm: ", p_Length));
    echo(str("        Drawer separator height in mm: ", p_Height));
    echo(str("        Drawer separator transversal unit width in mm: ", p_TransversalUnitWidth));
    echo(str("        Drawer separator clearance in mm: ", lct_SeparatorTolerance));
    echo(str("        Drawer separator thickness in mm: ", lct_SeparatorThickness));
    echo(str("        Drawer transversal separator left side count: ", p_LeftCount));
    echo(str("        Drawer transversal separator right side count: ", p_RightCount));
    echo(str("        Drawer transversal separator left side count: ", p_LeftUnitCount));
    echo(str("        Drawer transversal separator right side count: ", p_RightUnitCount));
    
        
    length = p_Length - lct_SeparatorTolerance;
    width = p_TransversalUnitWidth;
    height = p_Height - lct_SeparatorTolerance;
    leftWidth = p_TransversalUnitWidth * p_LeftUnitCount - lct_SeparatorTolerance / 2;
    rightWidth = p_TransversalUnitWidth * p_RightUnitCount - lct_SeparatorTolerance / 2;

    translate( [ -lct_SeparatorThickness / 2, 0, 0 ] ) cube( [ lct_SeparatorThickness, length, height ] );
        
    if( p_LeftCount > 0 ) {
        transversalDistance = length / (p_LeftCount + 1);
        translate( [ -leftWidth, length / (p_LeftCount + 1), 0 ] ) 
        for( i = [ 0 : p_LeftCount - 1 ] ) {
            translate( [ 0, transversalDistance * i, 0 ] ) cube( [ leftWidth, lct_SeparatorThickness,  height ] );
        }
    }
    if( p_RightCount > 0 ) {
        transversalDistance = length / (p_RightCount + 1);
        translate( [ 0, length / (p_RightCount + 1), 0 ] ) 
        for( i = [ 0 : p_RightCount - 1 ] ) {
            translate( [ 0, transversalDistance * i, 0 ] ) cube( [ rightWidth, lct_SeparatorThickness, height ] );
        }
    }
}

module lctLDS_GenerateTransversalSeparators(p_drawerId, p_drawerDefinition, p_XOffset, p_YOffset) {
    separatorNumber = p_drawerDefinition[search(["SeparatorNumber"], p_drawerDefinition)[0]][1];
    echo(separatorNumber);
    transversalSeparators = p_drawerDefinition[search(["TransversalSeparators"], p_drawerDefinition)[0]][1];
    width = p_drawerDefinition[search(["width"], p_drawerDefinition)[0]][1] - lct_DrawerSidesThickness * 2;
    height = p_drawerDefinition[search(["height"], p_drawerDefinition)[0]][1] - lct_DrawerBottomThickness;
    depth = p_drawerDefinition[search(["depth"], p_drawerDefinition)[0]][1] - lct_DrawerSidesThickness * 2;
    echo("    vvv LCTLightDrawerSystem:lctLDS_GenerateTransversalSeparators");
    echo(str("        Drawer separator width in mm: ", width));
    echo(str("        Drawer separator height in mm: ", height));
    echo(str("        Drawer separator depth in mm: ", depth));
    echo(str("        Traversal separators: ", transversalSeparators));

    if( len(transversalSeparators) > 0 ) {
        for( i = [ 0 : len(transversalSeparators) - 1 ] ) {
            leftCount = transversalSeparators[i][search(["LeftCount"], transversalSeparators[i])[0]][1];
            rightCount = transversalSeparators[i][search(["RightCount"], transversalSeparators[i])[0]][1];
            leftUnitCount = transversalSeparators[i][search(["LeftUnitCount"], transversalSeparators[i])[0]][1];
            rightUnitCount = transversalSeparators[i][search(["RightUnitCount"], transversalSeparators[i])[0]][1];
            echo(str("            Drawer transversal separator left count: ", leftCount));
            echo(str("            Drawer transversal separator right count: ", rightCount));
            echo(str("            Drawer transversal separator left unit count: ", leftUnitCount));
            echo(str("            Drawer transversal separator right unit count: ", rightUnitCount));
  
            translate( [ p_XOffset, p_YOffset * i, 0 ] ) lctLDS_GenerateTransversalSeparator(depth, height, width / (separatorNumber + 1), leftCount, rightCount, leftUnitCount, rightUnitCount);
        }
    }
}

module lct_GenerateSeparator(p_drawerModules, p_drawerId, p_innerWidth, p_innerHeight, p_innerDepth) {

    echo("    vvv LCTLightDrawerSystem:lct_GenerateSeparator");
    echo(str("        Drawer separator width in mm: ", p_innerWidth - lct_SeparatorTolerance));
    echo(str("        Drawer separator depth in mm: ", p_innerDepth - lct_SeparatorTolerance));
    echo(str("        Drawer separator height in mm: ", p_innerHeight - lct_SeparatorTolerance));
    
    
    if( lct_GetDrawerModuleAttribute(p_drawerModules, p_drawerId, "SeparatorOnWidth") ) {
        
        //translate( [ 0, -(p_innerDepth - lct_SeparatorTolerance), 0 ] ) 
        cube( [ lct_SeparatorThickness, p_innerDepth - lct_SeparatorTolerance, p_innerHeight - lct_SeparatorTolerance ] );
        
        if( lct_GetDrawerModuleAttribute(p_drawerModules, p_drawerId, "TransversalSeparatorCount") > 0 ) {
            distance = p_innerDepth / lct_GetDrawerModuleAttribute(p_drawerModules, p_drawerId, "TransversalSeparatorCount");
            echo(str("        Drawer separator transversal distance in mm: ", distance));
            transversalWidth = (p_innerWidth - lct_SeparatorTolerance) / (lct_GetDrawerModuleAttribute(p_drawerModules, p_drawerId, "SeparatorNumber") + 1);
            echo(str("        Drawer separator transversal width in mm: ", transversalWidth));
            
            for( i = [ 1 : lct_GetDrawerModuleAttribute(p_drawerModules, p_drawerId, "TransversalSeparatorCount") ] ) {
                translate( [ lct_SeparatorThickness / 2, (i - 0.5) * distance - lct_SeparatorThickness / 2, 0 ] ) cube( [ transversalWidth, lct_SeparatorThickness, p_innerHeight - lct_SeparatorTolerance ] );
            }
        }
        
    } else {
        echo("    vvv LCTLightDrawerSystem:lct_GenerateSeparator");
        echo(str("        Drawer separator width in mm: ", p_innerWidth - lct_SeparatorTolerance));
        echo(str("        Drawer separator height in mm: ", p_innerHeight - lct_SeparatorTolerance));

        translate( [ 0, -(p_innerHeight - lct_SeparatorTolerance), 0 ] ) cube( [ p_innerWidth - lct_SeparatorTolerance, lct_SeparatorThickness, p_innerHeight - lct_SeparatorTolerance ] ); 
    }
}

module lctLDS_GenerateSeparatorHolders(p_Drawer, p_InnerWidth, p_InnerHeight, p_InnerDepth) {
    
    separatorNumber = lctLDS_GetAttribute(p_Drawer, "SeparatorNumber");
    
    if( lctLDS_GetAttribute(p_Drawer, "SeparatorOnWidth") ) {
        
        distance = p_InnerWidth / (separatorNumber + 1);
        
        for( index = [ 1 : 1 : separatorNumber ] ) {
            
            translate( [ -(lct_SeparatorHolderDistance + lct_SeparatorHolderThickness * 2) / 2  + index * distance, 0, 0 ] ) union() {
                union() {
                    cube( [  lct_SeparatorHolderThickness, lct_SeparatorHolderThickness, p_InnerHeight ] );
                    translate( [ lct_SeparatorHolderDistance + lct_SeparatorHolderThickness, 0, 0 ] ) cube( [  lct_SeparatorHolderThickness, lct_SeparatorHolderThickness, p_InnerHeight ] );
                }      
                translate( [ 0, p_InnerDepth - lct_SeparatorHolderThickness + 0.1, 0 ] ) union() {
                    cube( [  lct_SeparatorHolderThickness, lct_SeparatorHolderThickness, p_InnerHeight ] );
                    translate( [ lct_SeparatorHolderDistance + lct_SeparatorHolderThickness, 0, 0 ] ) cube( [  lct_SeparatorHolderThickness, lct_SeparatorHolderThickness, p_InnerHeight ] );
                }
            }
        }
    } else {
        
        distance = p_InnerDepth / (separatorNumber + 1);
        
        for( index = [ 1 : 1 : separatorNumber ] ) {
            
            translate( [ 0, -(lct_SeparatorHolderDistance + lct_SeparatorHolderThickness * 2) / 2 + index * distance, 0 ] ) union() {
                union() {
                    cube( [  lct_SeparatorHolderThickness, lct_SeparatorHolderThickness, p_InnerHeight ] );
                    translate( [ 0, lct_SeparatorHolderDistance + lct_SeparatorHolderThickness, 0] ) cube( [  lct_SeparatorHolderThickness, lct_SeparatorHolderThickness, p_InnerHeight ] );
                }
                translate( [ p_InnerWidth - lct_SeparatorHolderThickness + 0.1, 0, 0 ] ) union() {
                    cube( [  lct_SeparatorHolderThickness, lct_SeparatorHolderThickness, p_InnerHeight ] );
                    translate( [ 0, lct_SeparatorHolderDistance + lct_SeparatorHolderThickness, 0] ) cube( [  lct_SeparatorHolderThickness, lct_SeparatorHolderThickness, p_InnerHeight ] );
                }
            }
        } 
    }
}

module lctLDS_GenerateOuterShell(p_ExternalWidth, p_ExternalDepth, p_ExternalHeight) {
    
    outerShellInternalWidth = p_ExternalWidth - lct_OuterShellThickness * 2;
    outerShellInternalHeight = p_ExternalHeight - lct_OuterShellThickness * 2;
    outerShellInternalDepth = p_ExternalDepth - lct_OuterShellThickness;
    
    echo("vvv LCTLightDrawerSystem:lct_GenerateOuterShell");

    echo(str("    Outer shell external width in mm: ", p_ExternalWidth));
    echo(str("    Outer shell external height in mm: ", p_ExternalHeight));
    echo(str("    Outer shell external depth in mm: ", p_ExternalDepth));
    
    echo(str("    Outer shell thickness in mm: ", lct_OuterShellThickness));
    echo(str("    Outer shell clearance in mm: ", lct_OuterShellClearance));

    echo(str("    Drawer holder external width in mm: ", outerShellInternalWidth - lct_OuterShellClearance));
    echo(str("    Drawer holder external height in mm: ", outerShellInternalHeight - lct_OuterShellClearance));
    echo(str("    Drawer holder external depth in mm: ", outerShellInternalDepth - lct_OuterShellClearance));
    
    difference() {
        cube( [ p_ExternalWidth, p_ExternalHeight, p_ExternalDepth ] );
        translate( [ lct_OuterShellThickness, lct_OuterShellThickness, lct_OuterShellThickness ] ) 
        cube( [ outerShellInternalWidth, outerShellInternalHeight, outerShellInternalDepth ] );
    }
}

module lctLDS_GenerateDrawer(p_DrawerId, p_Drawer) {
    
    outerWidth = lctLDS_GetAttribute(p_Drawer, "width");
    outerHeight = lctLDS_GetAttribute(p_Drawer, "height");
    outerDepth = lctLDS_GetAttribute(p_Drawer, "depth");
    effectiveHandleWidth = ( outerWidth >= lct_DrawerHandleWidth ? lct_DrawerHandleWidth : lct_MinimumDrawerHandleWidth );
    separatorNumber = lctLDS_GetAttribute(p_Drawer, "SeparatorNumber");
    
    echo("    vvv LCTLightDrawerSystem:lctLDS_GenerateDrawer");
    echo(str("        Drawer id: ", p_DrawerId));
    echo(str("        Drawer external width in mm: ", outerWidth));
    echo(str("        Drawer external height in mm: ", outerHeight));
    echo(str("        Drawer external depth in mm: ", outerDepth));
    echo(str("        Drawer internal width in mm: ", outerWidth - lct_DrawerSidesThickness * 2));
    echo(str("        Drawer internal height in mm: ", outerHeight - lct_DrawerBottomThickness));
    echo(str("        Drawer internal depth in mm: ", outerDepth - lct_DrawerSidesThickness * 2));
    echo(str("        Effective handle width in mm: ", effectiveHandleWidth));
    echo(str("        Separator number: ", separatorNumber));

    // Drawer body
    difference() {
        cube([outerWidth, outerDepth, outerHeight]);
        translate([ lct_DrawerSidesThickness, lct_DrawerSidesThickness, lct_DrawerBottomThickness ]) cube([outerWidth - lct_DrawerSidesThickness * 2, outerDepth - lct_DrawerSidesThickness * 2, outerHeight - lct_DrawerBottomThickness]);
    }
    
    // Drawer handle
    translate( [ (outerWidth - effectiveHandleWidth) / 2, -lct_DrawerHandleLength , 0 ] ) lctLDS_GenerateDrawerHandle(effectiveHandleWidth);    
    
    // Label holder
    translate( [ 0, -lct_LabelHolderThickness * 2, lct_DrawerHandleThickness + lct_LabelHolderDrawerHandleClearance] ) lctLDS_GenerateLabelHolder(outerWidth, outerHeight - lct_DrawerHandleThickness - lct_LabelHolderDrawerHandleClearance);
       
    // Separator holders
    if( separatorNumber > 0 ) {
        translate( [ lct_DrawerSidesThickness - 0.05, lct_DrawerSidesThickness - 0.05, lct_DrawerBottomThickness, ] ) lctLDS_GenerateSeparatorHolders(p_Drawer, outerWidth - lct_DrawerSidesThickness * 2, outerHeight - lct_DrawerBottomThickness, outerDepth - lct_DrawerSidesThickness * 2);        
    }    
}

module lctLDS_Generate(p_DrawerModule, p_MaxWidth, p_MaxDepth) {
    maxWidth = p_MaxWidth + 20;
    maxDepth = p_MaxDepth + 20;
    for( i = [ 1 : len(p_DrawerModule) - 1 ] ) {
        
        drawerId = p_DrawerModule[i][0];
        drawerDefinition = p_DrawerModule[i][1];
        
        translate( [ maxWidth * i, 0, 0 ] ) lctLDS_GenerateDrawer(drawerId, drawerDefinition);
        
        translate( [ maxWidth * i, maxDepth, 0 ] ) lctLDS_GenerateDrawerHolder(drawerId, drawerDefinition);
        
        translate( [ maxWidth * i, maxDepth * 2, 0 ] ) lctLDS_GenerateDrawerHolderTopPlate(drawerId, drawerDefinition);

        translate( [ maxWidth * i, maxDepth * 3, 0 ] ) lctLDS_GenerateTransversalSeparators(drawerId, drawerDefinition, maxWidth / 2, maxDepth);
           
    }
}

// Outer shell height in mm
OuterShellInnerHeight = ModuleHeight - lct_OuterShellThickness * 2 - lct_OuterShellClearance;

// Outer shell width in mm
OuterShellInnerWidth = ModuleWidth - lct_OuterShellThickness * 2 - lct_OuterShellClearance - lct_FemaleConnectorHeight();

// Outer shell depth in mm
OuterShellInnerDepth = ModuleDepth - lct_OuterShellThickness - lct_OuterShellClearance; 

echo("vvv DrawerModule");
echo(str("    Outer shell inner width INCLUDING the attachments in mm: ", OuterShellInnerWidth));
echo(str("    Outer shell inner height EXCLUDING the attachments in mm: ", OuterShellInnerHeight));
echo(str("    Outer shell inner EXCLUDING the attachments in mm: ", OuterShellInnerDepth));
echo("^^^ DrawerModule");

module GenerateOuterShell() {
    difference() {
        union() {
            lctLDS_GenerateOuterShell(ModuleWidth - lct_FemaleConnectorHeight(), ModuleDepth, ModuleHeight);
            translate( [ ModuleWidth - lct_FemaleConnectorHeight(), 0, 0 ] ) cube([ lct_FemaleConnectorHeight(),  ModuleHeight, ModuleDepth ]);
            
            translate( [ 0, ModuleHeight - 40, 30 ] ) rotate( [ 90, 0, -90 ] ) lct_MaleConnector();
            translate( [ 0, ModuleHeight - 40, ModuleDepth - 30 ] ) rotate( [ 90, 0, -90 ] ) lct_MaleConnector();
            
            translate( [ ModuleWidth / 2, 0, 30 ] ) rotate( [ 90, 90, 0 ] ) lct_MaleConnector();
            translate( [ ModuleWidth / 2, 0, ModuleDepth - 30 ] ) rotate( [ 90, 90, 0 ] ) lct_MaleConnector();
        }
        
        translate( [ ModuleWidth, ModuleHeight - 40, 30] ) rotate( [ 90, 0, -90 ] ) lct_FemaleConnectorKnockout();
        translate( [ ModuleWidth, ModuleHeight - 40, ModuleDepth - 30] ) rotate( [ 90, 0, -90 ] ) lct_FemaleConnectorKnockout();
        translate( [ ModuleWidth - lct_FemaleConnectorHeight() / 2, 0, 30 ] ) rotate([ 0, 90, 90 ]) cylinder(h = ModuleHeight, d = lct_FilamentHoleDiameter);
        translate( [ ModuleWidth - lct_FemaleConnectorHeight() / 2, 0, ModuleDepth - 30 ] ) rotate([ 0, 90, 90 ]) cylinder(h = ModuleHeight, d = lct_FilamentHoleDiameter);
        
    }
}

echo("vvv DrawerModule");
echo(str("    Module width INCLUDING the attachments in mm: ", ModuleWidth));
echo(str("    Module height EXCLUDING the attachments in mm: ", ModuleHeight));
echo(str("    Module Depth EXCLUDING the attachments in mm: ", ModuleDepth));
echo("^^^ DrawerModule");

GenerateOuterShell();

lctLDS_Generate(DrawerModule, ModuleWidth, ModuleDepth);

echo(str("*** Module Width: ", ModuleWidth));
echo(str("*** Module Depth: ", ModuleDepth));
