/*************************************************************************************
 *
 * Miscellaneous Bucket that fits on top of the drawer module
 *
 *************************************************************************************
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHOR OR COPYRIGHT
 * HOLDER BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * IT IS NOT PERMITTED TO MODIFY THIS COMMENT BLOCK.
 *
 * (c)2025, Claude "Tryphon" Theroux, Montreal, Quebec, Canada
 * http://www.ctheroux.com/
 *
 *************************************************************************************/

merged_code = true;

// Module width in mm - THIS VALUE SHALL MATCH THE ONE IN THE DrawerModule.
ModuleWidth = 130.5;

// Module depth in mm - THIS VALUE SHALL MATCH THE ONE IN THE DrawerModule.
ModuleDepth = 155;

// Module height in mm
ModuleHeight = 55;

// Bucket wall thickness in mm
BucketWallThickness = 1.6;

// Bottom thickness in mm
BucketBottomThickness = 5;

/*************************************************************************************
 *
 * Nothing shall be modified below this line
 *
 *************************************************************************************/
 
/*************************************************************************************
 *
 * Library inner parameters
 *
 *************************************************************************************/

// Clearance between a male and a female connector in mm
lct_MaleFemaleConnectorClearance = 0.1;

// Male connector length in mm
lct_MaleConnectorLength = 22;

// Male connector width in mm
lct_MaleConnectorWidth = 12;

// Male connector height in mm
lct_MaleConnectorHeight = 3.5;

// Filament hole diameter in mm
lct_FilamentHoleDiameter = 2.2;

 
$fn = 60;

function ratio(p_value, p_height) = (p_value - 2 * p_height) / p_value;

// Female connector length in mm
function lct_FemaleConnectorLength(p_length = lct_MaleConnectorLength) = p_length + lct_MaleFemaleConnectorClearance;

// Female connector width in mm
function lct_FemaleConnectorWidth(p_width = lct_MaleConnectorWidth) = p_width + lct_MaleFemaleConnectorClearance;

// Female connector height in mm
function lct_FemaleConnectorHeight(p_height = lct_MaleConnectorHeight) = p_height + lct_MaleFemaleConnectorClearance;

module lct_FilamentAttachmentDump(p_EnableDebug = false, p_length = lct_MaleConnectorLength, p_width = lct_MaleConnectorWidth, p_height = lct_MaleConnectorHeight) {
    if( p_EnableDebug ) {
        echo("vvv LCTFilamentAttachment:lct_FilamentAttachmentDump");
        echo("    lct_FilamentHoleDiameter:", lct_FilamentHoleDiameter);
        echo("    Male connector length:", p_length);
        echo("    Male connector width:", p_width);
        echo("    Male connector height:", p_height);
        echo("    Female connector length with clearance:", p_length + lct_MaleFemaleConnectorClearance);
        echo("    Female connector width with clearance:", p_width + lct_MaleFemaleConnectorClearance);
        echo("    Female connector height with clearance:", p_height + lct_MaleFemaleConnectorClearance);
        echo("    lct_MaleConnectorTopLengthRatio:", ratio(p_length, p_height));
        echo("    lct_MaleConnectorTopWidthRatio:", ratio(p_width, p_height));
        echo("^^^ LCTFilamentAttachment:lct_FilamentAttachmentDump");
    }
}

module lct_MaleConnector(p_length = lct_MaleConnectorLength, p_width = lct_MaleConnectorWidth, p_height = lct_MaleConnectorHeight) {
    difference() {
        linear_extrude(height = p_height, scale = [ ratio(p_length, p_height), ratio(p_width, p_height) ]) square([ p_length, p_width ], true);
    translate([ -p_length / 2, 0, p_height / 2]) rotate([ 0, 90, 0 ]) cylinder(h = p_length, d = lct_FilamentHoleDiameter);
    }
}

module lct_FemaleConnectorKnockout(p_length = lct_MaleConnectorLength + lct_MaleFemaleConnectorClearance, p_width = lct_MaleConnectorWidth + lct_MaleFemaleConnectorClearance, p_height = lct_MaleConnectorHeight + lct_MaleFemaleConnectorClearance) {
    difference() {
        linear_extrude(height = p_height, scale = [ ratio(p_length, p_height), ratio(p_width, p_height) ]) square([ p_length, p_width ], true);
    }
}

module GenerateMiscellaneousBucket() {
    difference() {
        cube([ ModuleWidth, ModuleDepth, ModuleHeight ]);
        translate( [ BucketWallThickness, BucketWallThickness, BucketBottomThickness ] ) cube([ ModuleWidth - BucketWallThickness * 2, ModuleDepth - BucketWallThickness * 2, ModuleHeight - BucketBottomThickness ]);
        
        translate([ ModuleWidth / 2, 30, 0]) rotate([ 0, 0, 90 ]) lct_FemaleConnectorKnockout();
        translate([ ModuleWidth / 2, ModuleDepth - 30, 0]) rotate([ 0, 0, 90 ]) lct_FemaleConnectorKnockout();
        
        translate( [ ModuleWidth / 2, 0, lct_FemaleConnectorHeight() / 2 ] ) rotate([ 0, 90, 90 ]) cylinder(h = ModuleDepth, d = lct_FilamentHoleDiameter);
    }
}

GenerateMiscellaneousBucket();

echo(str("*** Module Width: ", ModuleWidth));
echo(str("*** Module Depth: ", ModuleDepth));
