/*
    MotorController.cpp - Library for creating and controlling 2 Motors through a L298N motor drive controller.
    Created by Sam Dicker on 10.7.2024
*/

#include "Arduino.h"
#include "MotorController.h"

/**
 * @brief Constructor for procducing a motor controller object from 6 arguments. 
 * This module assumes the controller is a full h-bridge and that the motor is capable of moving both clockwise and counter clockwise.
 * 
 * @param PosPin the normally positive terminal of the motor.
 * @param NegPin the noramlly negative terminal of the motor.
 * @param MotorSpeed the speed of the motor once it is turned on.
 * @param RampTime the time is takes the motor to reach full motorspeed once it is turned on.
 * @param Name the name of the motor.
 */
MotorController::MotorController(int PosPin, int NegPin, int MotorSpeed, int RampTime, String Name){
    this->PosPin = PosPin;
    this->NegPin = NegPin;

    if(MotorSpeed>255 || MotorSpeed<0){
        Serial.println("Enter a motorspeed between 0 and 255.");
    }
    else{
        this->MotorSpeed = MotorSpeed;
    }

    this->RampTime = RampTime;
    this->Name = Name;
}

/**
 * @brief Initiates the motor controller.
 */
void MotorController::initMotor(){
    pinMode(PosPin, OUTPUT);
    pinMode(NegPin, OUTPUT);

    MotorForward();
    delay(1000);
    MotorOff();
}

// Set Parameters

/**
 * @brief Sets the normally positive pin of the motorcontroller to a new pin.
 * @param PosPin the new positive pin of the motor.
 */
void MotorController::setPosPin(int PosPin){
    this->PosPin = PosPin;
    initMotor();
}

/**
 * @brief Sets the normally negative pin of the motorcontroller to a new pin.
 * @param NegPin the new negative pin of the motor.
 */
void MotorController::setNegPin(int NegPin){
    this->NegPin = NegPin;
    initMotor();
}

/**
 * @brief Sets the motor state of the motor to on (1) or off (0).
 * @param MotorState the new motor state.
 */
void MotorController::setMotorState(int MotorState){
    this->MotorState = MotorState;
}

/**
 * @brief Sets a new motor speed ranging from 0 to 255.
 * @param MotorSpeed the new motor speed.
 */
void MotorController::setMotorSpeed(int MotorSpeed){
    if(MotorSpeed<=255 && MotorSpeed>=0){
    this->MotorSpeed = MotorSpeed;
    }
    else{
        Serial.println("Enter a speed between 0 and 255.");
    }
}

/**
 * @brief Sets a new ramp time for the motor when turning on.
 * @param RampTime the new ramp time for the motor.
 */
void MotorController::setRampTime(int RampTime){
    this->RampTime = RampTime;
}

/**
 * @brief Set a new name for the motor.
 * @param Name the new name of the motor.
 */
void MotorController::setName(String Name){
    this->Name = Name;
}

// Get Parameters
/**
 * @brief Gets the positive pin number.
 * @return the positive pin number of the motor.
 */
int MotorController::getPosPin(){
    return PosPin;
}

/**
 * @brief Gets the negative pin number.
 * @return the negative pin number of the motor.
 */
int MotorController::getNegPin(){
    return NegPin;
}

/**
 * @brief Gets the Motor state of the motor either on (1) or off(0).
 * @return the motors state.
 */
int MotorController::getMotorState(){
    return MotorState;
}

/**
 * @brief Gets the speed of the motor once turned on.
 * @return the speed of the motor if turned on.
 */
int MotorController::getMotorSpeed(){
    return MotorSpeed;
}

/**
 * @brief Gets the ramp up time of the motor when turning on.
 * @return the ramp up time of the motor.
 */
int MotorController::getRampTime(){
    return RampTime;
}

/**
 * @brief Gets the name of the Motor.
 * @return the name of the motor.
 */
String MotorController::getName(){
    return Name;
}

// Class Functions
/**
 * @brief Turns the motor on, and spins it forward.
 */
void MotorController::MotorForward(){
    //MotorOff(); #TODO This stops the fan running if it currently running.

    if(RampTime != 0){
        int val = MotorSpeed/RampTime;
        for(int i = 0; i<=RampTime; i++){
            analogWrite(PosPin,val*i);
            delay(1000);
        }
    }
    else{
        analogWrite(PosPin, MotorSpeed);
    }
    setMotorState(1);
}

/**
 * @brief Turns the motor on, and spins it backwards.
 */
void MotorController::MotorBackward(){

    //MotorOff(); #TODO This stops the fan running if it currently running.

    if(RampTime != 0){
        int val = MotorSpeed/RampTime;
        for(int i = 0; i<=RampTime; i++){
            analogWrite(NegPin,val*i);
            delay(1000);
        }
    }
    else{
        analogWrite(NegPin,MotorSpeed);
    }
    
    setMotorState(1);
}

/**
 * @brief Turns the motor off.
 */
void MotorController::MotorOff(){
    analogWrite(PosPin, 0);
    analogWrite(NegPin, 0);
    setMotorState(0);
}