/*******************************************************************************
  Cylinder Storage Modules

  Original credit to goes to konejavihannes and their design:
    https://www.thingiverse.com/thing:5226549

  This file contains all of the modules and functions, plus a demo module for
  showing off likely use cases :)

******************************************************************************/

// Debug
debug = false;

// the secret to it all...
// golden ratio
function phi() = (1+sqrt(5))/2;

// material to remove bounded by reference square and cylinder
function calc_cutout_radius(cylinder_diameter) = cylinder_diameter/2.0 * phi() / 4;

// internal reference square
function calc_ref_square_width(cylinder_diameter) = 2* (cylinder_diameter/2.0 + (2*calc_cutout_radius(cylinder_diameter))/(3*phi()));

module cylinder_array_lid(col_count, w, cylinder_height, cylinder_diameter, wall_thickness, lid_tolerance){
    lid_diameter = cylinder_diameter+lid_tolerance;
    for(col = [0:col_count-1]){
        if (true){
            x = col*(w);
            y = -w/2;
            z = 0;
            translate([x, y, z]){
                cylinder(h=cylinder_height, d=lid_diameter, center=true);
            }
        }
        if (true){
            x = col*(w);
            y = w/2;
            z = 0;
            translate([x, y, z]){
                cylinder(h=cylinder_height, d=lid_diameter, center=true);
            }
        }
    }
}

module cube_array_lid(col_count, w, cylinder_height, cylinder_diameter, wall_thickness){
    for(col = [0:col_count-2]){
        x = col*(w) + w/2;
        y = w/2;
        z = 0;
        cw = w*phi();
        cd = w*phi();
        ch = cylinder_height;
        translate([x, y, z]){
            cube([cw, cd, ch], center=true);
        }
    }
}

module cylinder_storage_divots(w, cutout_radius, lid_height,col_count, wall_thickness){
    if (true){
        x=-w+(cutout_radius*2/3);
        y=0;
        z=lid_height/2 - (cutout_radius*2/3);
        translate([x,y,z]){
            sphere(r=cutout_radius);
        }
    }
    if (true){
        x=w*(col_count-1)-(cutout_radius*2/3);
        y=0;
        z=lid_height/2 - (cutout_radius*2/3);
        translate([x,y,z]){
            sphere(r=cutout_radius);
        }
    }
}

module cylinder_storage_lid(cylinder_diameter, cylinder_height, col_count, wall_thickness, lid_tolerance){

    case_ratio = 0.8;
    overlap = 0.2;

    //smaller cylinder cutout
    cutout_radius = calc_cutout_radius(cylinder_diameter);
    
    lid_height = cylinder_height - cylinder_height * (case_ratio - overlap) + cutout_radius*2;

    //w is the width of the interior reference square
    w = calc_ref_square_width(cylinder_diameter);

    translate([0,0,lid_height/2]){
        difference(){
            union(){
                if(true){
                    x=-w/2;
                    y=0;
                    z=0;
                    translate([x,y,z]){
                        cylinder_array_lid(col_count, w, lid_height, cylinder_diameter, wall_thickness, lid_tolerance);
                    }
                }
                if(true){
                    x=-w/2;
                    y=-w/2;
                    z=0;
                    translate([x,y,z]){
                        cube_array_lid(col_count, w, lid_height, cylinder_diameter, wall_thickness);
                    }
                }
            } // union
            cylinder_storage_divots(w, cutout_radius, lid_height, col_count, wall_thickness);
        }
    }
} //cylinder_storage_lid

module cylinder_array_case(col_count, w, cylinder_height, cylinder_diameter){
    for(col = [0:col_count-1]){
        if (true){
            x = col*(w);
            y = -w/2;
            z = 0;
            translate([x, y, z]){
                cylinder(h=cylinder_height, d=cylinder_diameter, center=true);
            }
        }
        if (true){
            x = col*(w);
            y = w/2;
            z = 0;
            translate([x, y, z]){
                cylinder(h=cylinder_height, d=cylinder_diameter, center=true);
            }
        }
    }
}

module cube_array_case(col_count, w, cylinder_height){
    for(col = [0:col_count-2]){
        x = col*(w) + w/2;
        y = w/2;
        z = 0;
        cw = w*phi();
        cd = w;
        ch = cylinder_height;
        translate([x, y, z]){
            cube([cw, cd, ch], center=true);
        }
    }
}

module key(h, r, t){
    color("green"){
        cylinder(h+2, r=r-t, center=true);
        
        cw=r*2-t*2;
        cd=r*4;
        ch=h+2;
        
        cx=0;
        cy=-cd/2;
        cz=0;
        
        translate([cx,cy,cz]){
            cube([cw,cd,ch],center=true);
        }
    }
}

module key_cut(d, h, w, t, g){
    color("purple"){
        x = -w/2;
        y = 0;
        z = 0;
        translate([x,y,z]){
            cylinder(h+4, d=d, center=true);
        }
    }
    color("purple"){
        x = w/2;
        y = 0;
        z = 0;
        translate([x,y,z]){
            cylinder(h+4, d=d, center=true);
        }
    }
}

module cutout(col_count, w, cylinder_height, cylinder_diameter, cutout_radius, wall_thickness){
    cylinder_gap = w - cylinder_diameter/2;
    x1 = 0;
    y1 = -wall_thickness;
    z1 = 0;
    translate([x1,y1,z1]){
        difference(){
            if(true){
                x = 0;
                y = -cutout_radius;
                z = 0;
                translate([x,y,z]){
                    key(cylinder_height, cutout_radius, wall_thickness);
                }
            }
            if (true){
                x = 0;
                y = -w/2;
                z = 0;
                translate([x,y,z]){
                    key_cut(cylinder_diameter, cylinder_height, w, wall_thickness, cylinder_gap);
                }
            }
        }
    }
}

module cutout_array(col_count, w, cylinder_height, cylinder_diameter, cutout_radius, wall_thickness){

    for(col = [0:col_count-2]){
        x = col*w;
        y = 0;
        z = 0;
        translate([x, y, z]){
            translate([0,0,0]){
                cutout(col_count, w, cylinder_height, cylinder_diameter, cutout_radius, wall_thickness);
            }

            translate([0,0,0]){
                rotate([0,0,180]){
                    cutout(col_count, w, cylinder_height, cylinder_diameter, cutout_radius, wall_thickness);
                }
            }
        }
    }
}

module cylinder_storage_case(cylinder_diameter, cylinder_height, col_count, wall_thickness){
    case_ratio = 0.8;
    case_height = cylinder_height * case_ratio;

    //smaller cylinder cutout
    cutout_radius = calc_cutout_radius(cylinder_diameter);

    //w is the width of the interior reference square
    w = calc_ref_square_width(cylinder_diameter);

    translate([0,0,case_height/2]){
        difference(){
            union(){
                if(true){
                    x=-w/2;
                    y=0;
                    z=0;
                    translate([x,y,z]){
                        cylinder_array_case(col_count, w, case_height, cylinder_diameter);
                    }
                }
                if(true){
                    x=-w/2;
                    y=-w/2;
                    z=0;
                    translate([x,y,z]){
                        cube_array_case(col_count, w, case_height);
                    }
                }
            } // union
            cutout_array(col_count, w, cylinder_height, cylinder_diameter, cutout_radius, wall_thickness);
        } //difference

        if (true){
            x=-w+(3*cutout_radius/2);
            y=0;
            z=cylinder_height/4;
            translate([x,y,z]){
                sphere(r=cutout_radius);
            }
        }
        if (true){
            x=w*(col_count-1)-(3*cutout_radius/2);
            y=0;
            z=cylinder_height/4;
            translate([x,y,z]){
                sphere(r=cutout_radius);
            }
        }
    }
}

// reference shapes for testing the cutout radius and other values
module debug_refrerence_shapes(cylinder_diameter, cylinder_height, col_count, wall_thickness){

    //smaller cylinder cutout
    cutout_radius = calc_cutout_radius(cylinder_diameter);

    //w is the width of the interior reference square
    w = calc_ref_square_width(cylinder_diameter);

    difference(){
        color("blue"){
            cw = w+wall_thickness;
            cd = w+wall_thickness;
            ch = cylinder_height+7;
            cube([cw,cd,ch], center=true);
        }
        color("blue"){
            cw = w;
            cd = w;
            ch = cylinder_height+8;
            cube([cw,cd,ch], center=true);
        }
    } //difference

    // a quarter of the cylinder radius and a semicircle of the cutout radius
    // must fit inside this square and be tangential to each other and the
    // corners of the square.
    difference(){
        color("blue"){
            cx = -w/4;
            cy = -w/4;
            cz = 0;
            cw = w/2+wall_thickness;
            cd = w/2+wall_thickness;
            ch = cylinder_height + 7;
            translate([cx,cy,cz]){
                cube([cw,cd,ch], center=true);
            }
        }
        color("blue"){
            cx = -w/4;
            cy = -w/4;
            cz = 0;
            cw = w/2;
            cd = w/2;
            ch = cylinder_height + 8;
            translate([cx,cy,cz]){
                cube([cw,cd,ch], center=true);
            }
        }
    } //difference
    
    color("red"){
        translate([0,-cutout_radius,0]){
            difference(){
                cylinder(h=cylinder_height+5, r=cutout_radius, center=true);
                cylinder(h=cylinder_height+6, r=cutout_radius-wall_thickness/2, center=true);
            }
        }
        translate([-w/2,-w/2,0]){
            difference(){
                cylinder(h=cylinder_height+5, d=cylinder_diameter, center=true);
                cylinder(h=cylinder_height+6, d=cylinder_diameter-wall_thickness, center=true);
            }
        }
    }
} // module debug_reference_squares

module demo_set(cylinder_diameter, cylinder_height, col_count, wall_thickness, lid_tolerance, debug){
    color("blue"){
        translate([-cylinder_diameter*3/2,cylinder_diameter,0]){
            cylinder(d=cylinder_diameter,h=cylinder_height,center=true);
        }
    }
    cylinder_storage_lid(cylinder_diameter, cylinder_height, col_count, wall_thickness, lid_tolerance);
    
    x = calc_ref_square_width(cylinder_diameter) * (col_count-1) + cylinder_diameter*3/2;
    translate([x,0,0]){
        cylinder_storage_case(cylinder_diameter, cylinder_height, col_count, wall_thickness);
    }
    
    if (debug){
        translate([0,0,cylinder_height/2]){
            debug_refrerence_shapes(cylinder_diameter, cylinder_height, col_count, wall_thickness);
        }
        translate([x,0,cylinder_height/2]){
            debug_refrerence_shapes(cylinder_diameter, cylinder_height, col_count, wall_thickness);
        }
    }
}

module demo(debug){
    $fn = 40;
    // AAA cell batteries
    if(true){
        cylinder_diameter = 11;
        cylinder_height = 44;
        col_count = 4;
        wall_thickness = 0.1;
        lid_tolerance = 0.4;
        y = 0;
        translate([0,y,0]){
            demo_set(cylinder_diameter, cylinder_height, col_count, wall_thickness, lid_tolerance, debug);
        }
    }
    
    // D cell batteries
    if(true){
        cylinder_diameter = 30;
        cylinder_height = 61;
        col_count = 2;
        wall_thickness = 0.1;
        lid_tolerance = 0.3;
        y = 60;
        translate([0,y,0]){
            demo_set(cylinder_diameter, cylinder_height, col_count, wall_thickness, lid_tolerance, debug);
        }
    }
    
    // 18650 lithium cells
    if(true){
        cylinder_diameter = 19;
        cylinder_height = 68;
        col_count = 2;
        wall_thickness = 0.1;
        lid_tolerance = 0.4;
        y = 120;
        translate([0,y,0]){
            demo_set(cylinder_diameter, cylinder_height, col_count, wall_thickness, lid_tolerance, debug);
        }
    }
    
    // Crayola(r) crayons
    if(true){
        cylinder_diameter = 9;
        cylinder_height = 93;
        col_count = 8;
        wall_thickness = 0.1;
        lid_tolerance = 0.4;
        y = 160;
        translate([0,y,0]){
            demo_set(cylinder_diameter, cylinder_height, col_count, wall_thickness, lid_tolerance, debug);
        }
    }
}

//set debug to true to show reference squares and cylinders
demo(debug=debug);