$fn = 128;

thickness = 2; // Thickness in millimeters.
layerHeight = 65; // Vertical location of the separating layer, millimeters from the bottom.
lowerDiameter = 288; // Upper inner diameter of the flower pot in millimeters.
upperDiameter = 288; // Lower inner diameter of the flower pot in millimeters.
totalHeight = 295; // Total inner height of the flower pot.
pos = layerHeight / totalHeight; // Position of this layer.
layerDiameter = pos * upperDiameter + (1 - pos) * lowerDiameter; // Diameter of the created layer.
cutOutSize = 3; // Size of each cutout hole in the pattern.
outerMarginSize = 10; // Width of the solid outer margin in millimeters.

dimpleMarginSize = 5; // Width of the dimple margin in millimeters.
outerDimpleDiameter = 75; // Diameter of each dimple on the layer.
outerDimpleDiameterBottom = 30; // Diameter of each dimple at the bottom.
outerDimpleDistanceFromCenter =  (layerDiameter - outerDimpleDiameter - 20) / 2; // Distance from the center in millimeters.
outerDimpleCount = 4; // Number of cups on the outer rim.
slitCount = 6; // Number of slits on each of the cups.
centerDimple = false; // Should there be a dimple at the center.
applyPattern = true; // Should the pattern be applied to the layer.

waterPipeHoleDiameter = 43.5; // Watering pipe hole diameter.
waterPipeHoleMarginSize = dimpleMarginSize; // Width of the margin for the watering hole.
waterPipeHoleDistanceFromCenter = (layerDiameter - waterPipeHoleDiameter - 20) / 2; // Distance of the watering hole from the center.
waterPipeHoleDirectionDeg = 45; // Offset of the watering hole in degrees (to avoid overlap with the diples).
waterPipeHoleSupporHeight = 20; // Length of the support collar on the watering pipe hole.

gridSizeX = 30; // Size of the grid pattern (Custom Flower pattern). This should be large enough to cover the entire area of the separating layer.
gridSizeY = 30; // Size of the grid pattern (Custom Flower pattern). This should be large enough to cover the entire area of the separating layer.

// 4-piece segmentation for larger prints.
includeQuarter = [
    1, // -1, -1
    1, // +1, -1
    1, // -1, +1
    1  // +1, +1
];

// Pattern for the cutouts. Each number is the relative size of the cutout square, compared to cutOutSize.
shape = [
      0.0, 0.0, 0.0, 0.0, 0.7, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 0.7, 0.0, 0.0, 0.0, 0.0, 0.0,
      0.0, 0.0, 0.7, 1.0, 0.7, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.7, 1.0, 0.7, 0.0, 0.0, 0.0,
      0.0, 0.7, 0.7, 0.0, 0.0, 0.0, 0.7, 0.8, 0.8, 0.8, 0.0, 0.0, 0.0, 0.7, 0.7, 0.0, 0.0,
      0.0, 1.0, 0.0, 0.0, 0.6, 0.8, 0.8, 0.6, 0.6, 0.6, 0.8, 0.6, 0.0, 0.0, 1.0, 0.0, 0.0,
      0.0, 1.0, 0.0, 0.0, 0.8, 0.6, 0.4, 0.4, 0.4, 0.4, 0.6, 0.8, 0.0, 0.0, 1.0, 0.0, 0.0,
      0.0, 1.0, 0.0, 0.0, 0.6, 0.8, 0.8, 0.6, 0.6, 0.6, 0.8, 0.6, 0.0, 0.0, 1.0, 0.0, 0.0,
      0.0, 0.7, 0.7, 0.0, 0.0, 0.0, 0.7, 0.8, 0.8, 0.8, 0.0, 0.0, 0.0, 0.7, 0.7, 0.0, 0.0,
      0.0, 0.0, 0.7, 1.0, 0.7, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.7, 1.0, 0.7, 0.0, 0.0, 0.0,
      0.0, 0.0, 0.0, 0.0, 0.7, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 0.7, 0.0, 0.0, 0.0, 0.0, 0.0,
      0.0, 0.7, 0.7, 0.0, 0.0, 0.0, 1.0, 0.4, 0.4, 1.0, 0.0, 0.0, 0.0, 0.7, 0.7, 0.0, 0.0,
      0.7, 0.4, 0.4, 0.7, 0.7, 0.0, 1.0, 0.4, 0.4, 1.0, 0.0, 0.7, 0.7, 0.4, 0.4, 0.7, 0.0,
      0.7, 0.4, 0.6, 0.4, 0.4, 0.7, 1.0, 0.4, 0.4, 1.0, 0.7, 0.4, 0.4, 0.6, 0.4, 0.7, 0.0,
      0.7, 0.7, 0.7, 0.6, 0.4, 0.4, 0.8, 0.4, 0.4, 0.8, 0.4, 0.4, 0.6, 0.4, 0.7, 0.7, 0.0,
      0.0, 0.0, 0.0, 0.7, 0.6, 0.4, 0.6, 0.4, 0.4, 0.6, 0.4, 0.6, 0.4, 0.7, 0.0, 0.0, 0.0,
      0.0, 0.0, 0.0, 0.0, 0.7, 0.6, 0.4, 0.4, 0.4, 0.4, 0.6, 0.4, 0.7, 0.0, 0.0, 0.0, 0.0,
      0.0, 0.0, 0.0, 0.0, 0.6, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.6, 0.0, 0.0, 0.0, 0.0, 0.0,
      0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
];
shapeWid = 17; // Width of the pattern in the shape array above.
shapeHei = 17; // Height of the pattern in the shape array above.

translate([
    0,
    0,
    layerHeight
])
intersection() {
    union() {
        // Base plate with holes.
        difference() {
            
            // Base plate.
            cylinder(
                h = thickness,
                r1 = layerDiameter / 2,
                r2 = layerDiameter / 2
            );
            
            // Grid holes.
            if (applyPattern) {
                for (y = [-gridSizeY:gridSizeY]) {
                    for (x = [-gridSizeX:gridSizeX]) {
                        if (shape[
                                ((y + gridSizeY) % shapeHei) * shapeWid +
                                ((x + gridSizeX) % shapeWid)
                            ] >= 0.3) {
                            translate([
                                x * (cutOutSize + thickness) - cutOutSize * (0.5 + (shape[
                                   ((y + gridSizeY) % shapeHei) * shapeWid +
                                   ((x + gridSizeX) % shapeWid)
                               ]) / 2),
                                y * (cutOutSize + thickness) - cutOutSize * (0.5 + (shape[
                                   ((y + gridSizeY) % shapeHei) * shapeWid +
                                   ((x + gridSizeX) % shapeWid)
                               ]) / 2),
                                -1
                            ])
                            cube([
                               cutOutSize *
                               1.0*min(1, max(0.3, shape[
                                   ((y + gridSizeY) % shapeHei) * shapeWid +
                                   ((x + gridSizeX) % shapeWid)
                               ])),
                               cutOutSize *
                               1.0*min(1, max(0.3, shape[
                                   ((y + gridSizeY) % shapeHei) * shapeWid +
                                   ((x + gridSizeX) % shapeWid)
                               ])),
                               thickness + 2
                            ]);
                        }
                    }
                }
            }

            // Water pipe hole.
            {            
                translate([
                    waterPipeHoleDistanceFromCenter * cos(waterPipeHoleDirectionDeg),
                    waterPipeHoleDistanceFromCenter * sin(waterPipeHoleDirectionDeg),
                    -1
                ])
                cylinder(
                    h = thickness + 2,
                    r1 = waterPipeHoleDiameter / 2,
                    r2 = waterPipeHoleDiameter / 2
                );
            }

            // Dimple holes.
            {
                // Outer dimple holes.
                if (outerDimpleCount > 0) {
                    for (i = [0:(outerDimpleCount - 1)]) {
                        
                        translate([
                            outerDimpleDistanceFromCenter * cos(i / outerDimpleCount * 360),
                            outerDimpleDistanceFromCenter * sin(i / outerDimpleCount * 360),
                            -1
                        ])
                        cylinder(
                            h = thickness + 2,
                            r1 = outerDimpleDiameter / 2,
                            r2 = outerDimpleDiameter / 2
                        );
                    }
                } 
                if (centerDimple) {
                    // Center dimple hole.
                    translate([
                        0,
                        0,
                        -1
                    ])
                    cylinder(
                        h = thickness + 2,
                        r1 = outerDimpleDiameter / 2,
                        r2 = outerDimpleDiameter / 2
                    );
                }
            }
        }
        
        // Solid outer rim.
        difference() {
            
            // Base plate.
            cylinder(
                h = thickness,
                r1 = layerDiameter / 2,
                r2 = layerDiameter / 2
            );
            
            // Center part.
            translate([0, 0, -1])
            cylinder(
                h = thickness + 2,
                r1 = layerDiameter / 2 - outerMarginSize,
                r2 = layerDiameter / 2 - outerMarginSize
            );
        }
        
        
        // Dimple cups.
        union() {
            // Dimple rim plates.
            if (outerDimpleCount > 0) {
                for (i = [0:(outerDimpleCount - 1)]) {
                    
                    difference() {
                        translate([
                            outerDimpleDistanceFromCenter * cos(i / outerDimpleCount * 360),
                            outerDimpleDistanceFromCenter * sin(i / outerDimpleCount * 360),
                            0
                        ])
                        cylinder(
                            h = thickness,
                            r1 = outerDimpleDiameter / 2 + dimpleMarginSize,
                            r2 = outerDimpleDiameter / 2 + dimpleMarginSize
                        );
                        
                        translate([
                            outerDimpleDistanceFromCenter * cos(i / outerDimpleCount * 360),
                            outerDimpleDistanceFromCenter * sin(i / outerDimpleCount * 360),
                            -1
                        ])
                        cylinder(
                            h = thickness + 2,
                            r1 = outerDimpleDiameter / 2,
                            r2 = outerDimpleDiameter / 2
                        );
                    }
                    
                    translate([
                        outerDimpleDistanceFromCenter * cos(i / outerDimpleCount * 360),
                        outerDimpleDistanceFromCenter * sin(i / outerDimpleCount * 360),
                        -layerHeight
                    ])
                    difference() {
                        cylinder(
                            h = layerHeight,
                            r1 = outerDimpleDiameterBottom / 2 + thickness,
                            r2 = outerDimpleDiameter / 2 + thickness
                        );
                        
                        translate([
                            0,
                            0,
                            thickness
                        ])
                        cylinder(
                            h = layerHeight - thickness + 0.025,
                            r1 = outerDimpleDiameterBottom / 2,
                            r2 = outerDimpleDiameter / 2
                        );
                        
                        for (i = [0:(slitCount - 1)]) {
                            rotate([0, 0, i / slitCount * 360])
                            translate([
                                outerDimpleDiameterBottom / 4,//-(outerDimpleDiameter / 2 + thickness),
                                -thickness / 2,
                                -1 // thickness
                            ])
                            cube([
                                outerDimpleDiameter / 2 + 2 * thickness,
                                thickness,
                                layerHeight * 0.9 - thickness
                            ]);
                        }
                    }
                }
            }
            
            // Water pipe hole.
            translate([
                waterPipeHoleDistanceFromCenter * cos(waterPipeHoleDirectionDeg),
                waterPipeHoleDistanceFromCenter * sin(waterPipeHoleDirectionDeg),
                0
            ])
            difference() {
                {
                    union() {
                        cylinder(
                            h = thickness,
                            r1 = waterPipeHoleDiameter / 2 + waterPipeHoleMarginSize,
                            r2 = waterPipeHoleDiameter / 2 + waterPipeHoleMarginSize
                        );
                        
                        translate([
                            0,
                            0,
                            -(thickness + waterPipeHoleSupporHeight)
                        ])
                        cylinder(
                            h = thickness + waterPipeHoleSupporHeight,
                            r1 = waterPipeHoleDiameter / 2 + thickness,
                            r2 = waterPipeHoleDiameter / 2 + thickness
                        );
                    }
                    
                    translate([
                        0,
                        0,
                        -thickness - waterPipeHoleSupporHeight - 1
                    ])
                    cylinder(
                        h = 2 * thickness + waterPipeHoleSupporHeight + 2,
                        r1 = waterPipeHoleDiameter / 2,
                        r2 = waterPipeHoleDiameter / 2
                    );
                }
            }
            
            if (centerDimple) {
                difference() {
                    
                    cylinder(
                        h = thickness,
                        r1 = outerDimpleDiameter / 2 + dimpleMarginSize,
                        r2 = outerDimpleDiameter / 2 + dimpleMarginSize
                    );
                
                    translate([
                        0,
                        0,
                        -1
                    ])
                    cylinder(
                        h = thickness + 2,
                        r1 = outerDimpleDiameter / 2,
                        r2 = outerDimpleDiameter / 2
                    );  
                }
            
                translate([
                    0,
                    0,
                    -layerHeight
                ])
                difference() {
                    cylinder(
                        h = layerHeight,
                        r1 = outerDimpleDiameterBottom / 2 + thickness,
                        r2 = outerDimpleDiameter / 2 + thickness
                    );
                    
                    translate([
                        0,
                        0,
                        thickness
                    ])
                    cylinder(
                        h = layerHeight - thickness + 0.025,
                        r1 = outerDimpleDiameterBottom / 2,
                        r2 = outerDimpleDiameter / 2
                    );
                    
                    for (i = [0:(slitCount - 1)]) {
                        rotate([0, 0, i / slitCount * 360])
                        translate([
                            outerDimpleDiameterBottom / 4,
                            -thickness / 2,
                            -1 // thickness
                        ])
                        cube([
                            outerDimpleDiameter / 2 + 2 * thickness,
                            thickness,
                            layerHeight * 0.9 - thickness
                        ]);
                    }
                }
            }
        }
    }
    
    for (y = [0:1]) {
        for (x = [0:1]) {
            if (includeQuarter[y * 2 + x] == 1) {
                translate([
                    (x - 1) * layerDiameter / 2 - (1 - x),    
                    (y - 1) * layerDiameter / 2 - (1 - y),
                    -layerHeight - 1
                ])
                cube([
                    layerDiameter / 2 + 1,
                    layerDiameter / 2 + 1,
                    layerHeight + thickness + 2
                ]);
            }
        }
    }
}