// Copyright Thomas Kramer, 2025
// Licensed under CERN-OHL-S v2 (Open Hardware Licence)
// https://codeberg.org/tok/openscad-models

include <morphology.scad>;

$fn=64;

// out: outer part of buckle
// in: inner part of buckle

ribbon_width = 38;
ribbon_thickness = 1.2;

out_walls = 5;
tol = 0.2;
in_walls_thickness = 3.5;
in_width = ribbon_width + 2*in_walls_thickness; // Width of inner part, without stop-bolt
out_thickness = 6;
out_height = 14;
out_back_bar_width = 12;


out_width = in_width + 2*out_walls + 2*tol;

out_bar_width = 15;

in_length = 41;

smooth_r = 1.5;


in_back_bar_thickness = 6;

i_front_bar_width = 16;
i_front_bar_thickness = 4;

i_back_bar_width = 7;

// cross bar rounding
r_small = 2;
r_big = 4;

cross_bar_height = 12;
cross_bar_angle = 40;

cross_bar_pos = i_front_bar_width + ribbon_thickness*3;
cross_bar_dx = tan(cross_bar_angle) * (cross_bar_height-r_small-r_big)+ r_small + r_big;

back_bar_pos = cross_bar_pos + cross_bar_dx + ribbon_thickness*3;
//back_bar_pos = out_length-out_back_bar_width-i_back_bar_width;


hinge_r_outer = in_back_bar_thickness/2-0.5;
hinge_depth = min(2, out_walls/2);
hinge_position = [back_bar_pos+i_back_bar_width-hinge_r_outer, hinge_r_outer+out_thickness, 0];

insertion_spacing = cross_bar_height + 2*ribbon_thickness; // allow inner part to rotate during insertion.
out_length = hinge_position.x - hinge_r_outer + insertion_spacing + out_back_bar_width;
out_support_len = cross_bar_pos + cross_bar_dx + smooth_r;
out_belt_guide_len = out_back_bar_width+ribbon_thickness;

funnel_opening = out_length - out_support_len - out_belt_guide_len;
    
module crossbar_profile2d() {
    
    translate([0, out_thickness]) {
 
        // cross bar
        translate([cross_bar_pos, 0])
        hull() {
            
            translate([r_small, 0]) {  
                translate([0, r_small])
                    circle(r_small);
                
                translate([tan(cross_bar_angle) * (cross_bar_height-r_big-r_small), -r_big+cross_bar_height])
                    circle(r_big);
            }
        }
    }
}

module in_profile2d() {

    crossbar_profile2d();
    
    translate([0, out_thickness]) {
        // front bar
        smooth2d(smooth_r)
            square([i_front_bar_width, i_front_bar_thickness]);

        
        
        // back bar
        translate([back_bar_pos, 0])
            smooth2d(smooth_r)
            square([i_back_bar_width, in_back_bar_thickness]);
    }
}


module out_profile2d() {
    
    // front bar
    difference() {
        union() {
            smooth2d(smooth_r)
                square([cross_bar_pos, out_thickness]);
                
            translate([cross_bar_pos/2, 0])
                smooth2d(smooth_r)
                    square([cross_bar_pos/2, out_thickness+2]);
        }
        
        dilate2d(ribbon_thickness) {
            crossbar_profile2d();
        }
        
        minkowski() {
            in_profile2d();
            // enlarge towards the back
            square([0.5, 0.1]);
        }
    }
    
        
    
    // back bar
    translate([out_length-out_back_bar_width, 0, 0])
    smooth2d(smooth_r)
        hull() {
            translate([out_back_bar_width-1, 0])
                square([1, out_thickness]);
            translate([out_thickness/2, out_thickness/2])
                circle(out_thickness/2);
        }
}


module out_walls() {
    module profile() {
        smooth2d(smooth_r) {
            hull() { out_profile2d(); }
            
            // arcs
            r = out_length * 1;
            intersection() {
                square([out_length, out_height]);
                // Rounding
                translate([out_length/2, -r+out_height])
                    circle(r);
            }
        }
    }
    

    
    linear_extrude(out_walls)
        profile();
    
    
    translate([0, 0, out_width-out_walls])
    linear_extrude(out_walls)
        profile();
    
    // support: keeps the inner part from falling out when belt is under tension.
    module support_profile() {
        smooth2d(smooth_r) {
            square([out_support_len, out_thickness]);
        }
        
        // Guide for attached belt.
        smooth2d(smooth_r)
        intersection() {
            profile();
            translate([out_length-out_belt_guide_len, 0])
                square([out_belt_guide_len, 100]);
        }
    }
    translate([0, 0, out_walls])
        linear_extrude(in_walls_thickness) support_profile();
    translate([0, 0, out_width-out_walls-in_walls_thickness])
       linear_extrude(in_walls_thickness) support_profile();
}

module ribbon_fixed_at_out_cutout() {
    translate([0, 0, out_thickness])
    linear_extrude(out_height)
    translate([-1, -ribbon_width/2])
        square([out_length*2, ribbon_width]);
}

module out_solid_filled() {
    rotate([90, 0, 0])
    translate([0, 0, -out_width/2])
    linear_extrude(out_width)
        out_profile2d();
}

module out_solid() {
    difference() {
        out_solid_filled();
        ribbon_fixed_at_out_cutout();
    }
}

//out_solid();

module smooth2d(r) {
        dilate2d(r) erode2d(r) children();
}


module in_bars() {
    linear_extrude(in_width)
     in_profile2d();
}

module out_bars() {
    difference() {
        
        translate([0, 0, -out_width/2]) // center
        linear_extrude(out_width)
            out_profile2d();
        
        // make space for the inner part
        minkowski() {
            in(with_hinges=false, bars=false, walls=true);
            tol = 1;
            translate([0, tol/2, 0])
            cube(1, center=true);
        }
        
        // Notch for easier release of buckle
        // Important when using with thick gloves.
        r1 = ribbon_width;
        //translate([0, out_thickness/2, 0])
        translate([-r1 + r1 * 0.1, 0, 0])
        rotate([-90, 0, 0])
        cylinder(40, r1, r1, $fn=256);
    }
}

module in_walls() {
    module profile() {
        hull() { in_profile2d(); }
    }
    
    linear_extrude(in_walls_thickness)
        profile();
    
    translate([0, 0, in_width-in_walls_thickness])
    linear_extrude(in_walls_thickness)
        profile();
}

module smooth_outer_x_corners(r) {
    rotate([-90, 0, 0])
    smooth_outer_z_corners(r)
    rotate([90, 0, 0])
    children();
}

module smooth_outer_z_corners(r) {
    intersection() {
        
        children();
    
        linear_extrude(1000)
            smooth2d(r)
                hull()
                    projection()
                        children();
        
    }
}

module in(with_hinges=true, walls=true, bars=true) {
    module hinge() {
        translate(hinge_position)
            mirror([0, 0, 1])
                cylinder(hinge_depth, hinge_r_outer, hinge_r_outer/2);
    }
    
    translate([0, 0, -in_width/2]){
       
        if(bars) in_bars();
        if(walls) in_walls();
         
        if(with_hinges) {
            // hinges
            hinge();
            translate([0, 0, in_width])
                mirror([0, 0, 1])
                    hinge();
        }
       
    }
    
}


module out() {
    out_bars();
    
    translate([0, 0, -out_width/2]){
        out_walls();
    }
}

module rotate_at(rot, origin) {
    translate(origin)
        rotate(rot)
            translate(-origin)
                children();
}

module assemble_trajectory() {
    dw = 0.2;
    module node1() {square(dw, center=true);}
    module node2() {translate([0, -3]) square(dw, center=true);}
    module node3() {
        dy = -10;
        spacing = hinge_r_outer + 2;
        dx = -hinge_position.x + out_support_len+spacing/2;
        translate([dx, dy]) square([funnel_opening-spacing, dw]);
        r = funnel_opening;
        translate([0, -r-0])
            intersection() { // cut to upper right quadrant
                translate([-hinge_position.x + out_support_len, 0]) circle(r);
                square(r);
            }
    }
    
    linear_extrude(dw) {
        hull() {
            node1(); node2();
        }
        hull() {
            node2(); node3();
        }
    }
        
}

module assemble_trajectory2() {
    dw = 0.2;
    funnel_opening = 8;
    module node1() {square(dw, center=true);}
    module node2() {translate([0, -4]) square(dw, center=true);}
    module node3() {
        dy = -8;
        dx = 1;
        translate([dx, dy]) square([funnel_opening, dw], center=true);
        r = funnel_opening*0.6;
        translate([funnel_opening/2+dx-r, dy])
            intersection() { // cut to upper right quadrant
                circle(r);
                square(r);
            }
    }
    
    linear_extrude(dw) {
        hull() {
            node1(); node2();
        }
        hull() {
            node2(); node3();
        }
    }
        
}

module hinges_cutout() {
    
    module hinges() {in(with_hinges=true, walls=false, bars=false);};
   
    
    minkowski() {
        hinges();
        assemble_trajectory();
    }
}

smooth_outer_x_corners(4)
difference() {
    out();  
    hinges_cutout();
}

translate([0, 30, 0])
    in();

rotate_at([0, 0, -90], hinge_position)
%in();

//out_profile2d();
//#in_profile2d();



