// DIY Inkjet Printer Firmware
// Nihcolas C Lewis 2011
// portions of code adapted from:
//Davey, M. (2008). Nickel-O-Matic Robot. Retrieved from Parallax Inc Web site: http://www.parallax.com/tabid/769/Default.aspx
//Fried, L. ". (2009, October 20). Blog: Adafruit. Retrieved from Adafruit Industries: http://www.adafruit.com/blog/2009/10/20/example-code-for-multi-button-checker-with-debouncing/
//Fried, L. ". (2010). Character LCDs. Retrieved from Ladyada.net: http://www.ladyada.net/learn/lcd/charlcd.html
//Fried, L. ". (2010). Logger Shield. Retrieved from Ladyada.net: http://www.ladyada.net/make/logshield/design.html
//Gilliland, M. (2005). Inkjet Applications. Woodglen Press .
//neillzero. (2006, October 16). LCD4Bit v0.1 (http://abstractplain.net). Retrieved from R Cube Station: http://www.r3cube.com/Drivers-software/LCD4Bit_mod.zip

// include the library code:
#include <LiquidCrystal.h>
#include <Stepper.h>
#include <SD.h>

//**LCD
// initialize the library with the numbers of the interface pins
LiquidCrystal lcd(7, 8, 9, 10, 11, 12);

//**Buttons
#define DEBOUNCE 10  // button debouncer, how many ms to debounce, 5+ ms is usually plenty
// here is where we define the buttons that we'll use. button "0" is the first, button "5" is the 6th, etc
 // the analog 0-15 pins are also known as 54-69
byte buttons[] = {54, 55, 56, 57, 58, 59, 60, 61, 62, 63};
//                 0   1   2   3   4   5   6   7   8   9  
//                +X  +Y  Mnu Prt -X  -Y  EX TEST  Xnd Ynd END STOPS (Pressed = Not Pressed)
// This handy macro lets us determine how big the array up above is, by checking the size
#define NUMBUTTONS sizeof(buttons)
// we will track if a button is just pressed, just released, or 'currently pressed' 
byte pressed[NUMBUTTONS], justpressed[NUMBUTTONS], justreleased[NUMBUTTONS];

//**Ink
//number of time each slice is fired
int saturation = 3;
const int purgeTimes = 10;

//**Steppers
const int xStepsPerRevolution = 200;
const int yStepsPerRevolution = 400;

int xSpeed = 60;
int ySpeed = 35;

const int xStepPerInch = 136;  //125=0.461" => 136=0.5"
const int yStepPerInch = 300;  //255.2=0.75" => 340.3=1"  -  340.3 = 1.25" => 272.21=1"

const int xStepPerPass = xStepPerInch/8;
const int yStepPerStrip = yStepPerInch/96;
int yDirection = 1;

int xLoc = 0;  //steps
int yLoc = 0;  //steps

int row = 1;  //values for LCD feedback during print

boolean menuMode = false;

// initialize the stepper library on pins 42 through 49:
Stepper stepperX(xStepsPerRevolution, 42,43,44,45);            
Stepper stepperY(yStepsPerRevolution, 46,47,48,49);

//**SD
const int chipSelect = 53;

//**Machine Locations [inches]
const int xPrintMin = 2;
const int xPrintMax = 8.5;
const int yMax = 4.25;


const int xPrintMinSteps = xPrintMin*xStepPerInch;
const int xPrintMaxSteps = xPrintMax*xStepPerInch;
const int yMaxSteps = yMax*yStepPerInch;


void setup(){
  byte i;

  // set the stepper speeds [RPM]:
  stepperX.setSpeed(xSpeed);
  stepperY.setSpeed(ySpeed);
  
  // set up the LCD's number of columns and rows: 
  lcd.begin(16, 2);
  // Print a message to the LCD.
  lcd.print("Open3DP" );
  lcd.setCursor(0, 1);
  Serial.print("inkjet printer");
  
  Serial.begin(9600);
  Serial.print("Open3DP inkjet printer");

  // pin13 LED
  pinMode(13, OUTPUT);

  // initialize the ink pins.
  pinMode(22, OUTPUT);
  pinMode(23, OUTPUT);
  pinMode(24, OUTPUT);
  pinMode(25, OUTPUT);
  pinMode(26, OUTPUT);
  pinMode(27, OUTPUT);
  pinMode(28, OUTPUT);
  pinMode(29, OUTPUT);
  pinMode(30, OUTPUT);
  pinMode(31, OUTPUT);
  pinMode(32, OUTPUT);
  pinMode(33, OUTPUT);

  // Make input & enable pull-up resistors on switch pins
  for (i=0; i< NUMBUTTONS; i++) {
    pinMode(buttons[i], INPUT);
    digitalWrite(buttons[i], HIGH);
  }

  homeXY();
}

void loop()
{
  // set the cursor to column 0, line 1
  // (note: line 1 is the second row, since counting begins with 0):
  lcd.setCursor(0, 1);
  check_switches();      // when we check the switches we'll get the current state
//  for (byte i = 0; i < NUMBUTTONS; i++) {
//    if (justpressed[i]) {
//      Serial.print(i, DEC);
//      Serial.println(" Just pressed");
//      // remember, check_switches() will CLEAR the 'just pressed' flag
//    }
//    if (justreleased[i]) {
//      Serial.print(i, DEC);
//      Serial.println(" Just released");
//      // remember, check_switches() will CLEAR the 'just pressed' flag
//    }
//    if (pressed[i]) {
//      Serial.print(i, DEC);
//      Serial.println(" pressed");
//      // is the button pressed down at this moment
//    }
//  }
  if(menuMode){  //menu operations
    if (pressed[0]) {
      xSpeed++;
      stepperX.setSpeed(xSpeed);
      lcd.setCursor(1, 1);
      lcd.print("   ");
      lcd.setCursor(1, 1);
      lcd.print(xSpeed);
      delay(50);
    }
    if (pressed[1]) {
      ySpeed++;
      stepperY.setSpeed(ySpeed);
      lcd.setCursor(5, 1);
      lcd.print("   ");
      lcd.setCursor(5, 1);
      lcd.print(ySpeed);
      delay(50);
    }
	if (justpressed[2]) {
      menuMode = false;
      lcd.setCursor(0, 0);
      lcd.print("Open3DP         ");
      lcd.setCursor(0, 1);
      lcd.print("                ");
    }
    if (pressed[4] && xSpeed>0) {
      xSpeed--;
      stepperX.setSpeed(xSpeed);
      lcd.setCursor(1, 1);
      lcd.print("   ");
      lcd.setCursor(1, 1);
      lcd.print(xSpeed);
      delay(50);
    }
    if (pressed[5] && ySpeed>0) {
      ySpeed--;
      stepperY.setSpeed(ySpeed);
      lcd.setCursor(5, 1);
      lcd.print("   ");
      lcd.setCursor(5, 1);
      lcd.print(ySpeed);
      delay(50);
    }
    if (pressed[6]) {
      if(saturation<20){
        saturation++;
      } else {
        saturation=1;
      }
      lcd.setCursor(9, 0);
      lcd.print("   ");
      lcd.setCursor(9, 0);
      lcd.print(saturation);
      delay(50);
    }
  } else {  //normal operations
    if (pressed[0] && xLoc<xPrintMaxSteps) {
      stepperX.step(1);
      xLoc++;
      lcd.setCursor(0, 1);
      lcd.print("X:              ");
      lcd.setCursor(3, 1);
      lcd.print(xLoc);
    }
    if (pressed[1] && yLoc<yMaxSteps) {
      stepperY.step(1);
      yLoc++;
      lcd.setCursor(0, 1);
      lcd.print("Y:              ");
      lcd.setCursor(3, 1);
      lcd.print(yLoc);
    }
	if (justpressed[2]) {
      menuMode = true;
      lcd.setCursor(0, 0);
      lcd.print("Menu            ");
      lcd.setCursor(0, 1);
      lcd.print("X   Y   S       ");
      lcd.setCursor(1, 1);
      lcd.print(xSpeed);
      lcd.setCursor(5, 1);
      lcd.print(ySpeed);
      lcd.setCursor(9, 1);
      lcd.print(saturation);
    }
    if (justpressed[3]) {
      lcd.setCursor(0, 1);
      lcd.print("Print           ");
      printfile();
      lcd.setCursor(0, 1);
      lcd.print("                ");
    }
    if (pressed[4] && xLoc>0) {
      stepperX.step(-1);
      xLoc--;
      lcd.setCursor(0, 1);
      lcd.print("X:              ");
      lcd.setCursor(3, 1);
      lcd.print(xLoc);
    }
    if (pressed[5] && yLoc>0) {
      stepperY.step(-1);
      yLoc--;
      lcd.setCursor(0, 1);
      lcd.print("Y:              ");
      lcd.setCursor(3, 1);
      lcd.print(yLoc);
    }
    if (justpressed[7]) {
      lcd.setCursor(0, 1);
      lcd.print("Calibrate       ");
      //purge();
      calibrate();
      lcd.setCursor(0, 1);
      lcd.print("                ");
    }
  }    
    
}

void check_switches()
{
  static byte previousstate[NUMBUTTONS];
  static byte currentstate[NUMBUTTONS];
  static long lasttime;
  byte index;

  if (millis() < lasttime) {
     // we wrapped around, lets just try again
     lasttime = millis();
  }

  if ((lasttime + DEBOUNCE) > millis()) {
    // not enough time has passed to debounce
    return;
  }
  // ok we have waited DEBOUNCE milliseconds, lets reset the timer
  lasttime = millis();

  for (index = 0; index < NUMBUTTONS; index++) {
    justpressed[index] = 0;       // when we start, we clear out the "just" indicators
    justreleased[index] = 0;

    currentstate[index] = digitalRead(buttons[index]);   // read the button

    /*     
    Serial.print(index, DEC);
    Serial.print(": cstate=");
    Serial.print(currentstate[index], DEC);
    Serial.print(", pstate=");
    Serial.print(previousstate[index], DEC);
    Serial.print(", press=");
    */

    if (currentstate[index] == previousstate[index]) {
      if ((pressed[index] == LOW) && (currentstate[index] == LOW)) {
          // just pressed
          justpressed[index] = 1;
      }
      else if ((pressed[index] == HIGH) && (currentstate[index] == HIGH)) {
          // just released
          justreleased[index] = 1;
      }
      pressed[index] = !currentstate[index];  // remember, digital HIGH means NOT pressed
    }
    //Serial.println(pressed[index], DEC);
    previousstate[index] = currentstate[index];   // keep a running tally of the buttons
  }
}

void homeXY(){
  lcd.setCursor(0, 1);
  lcd.print("Homing Y..." );
//home Y
  check_switches();
  while(pressed[9]){
    stepperY.step(-1);
    check_switches();
  }
  yLoc=0;
  lcd.setCursor(0, 1);
  lcd.print("Homing X..." );
//home X
  check_switches();
  while(pressed[8]){
    stepperX.step(-1);
    check_switches();
  }
  xLoc=0;
  lcd.setCursor(0, 1);
  lcd.print("                " );
}


void printfile( )
{
  row = 1;
  //Serial.print("Initializing SD card...");
  // Print a message to the LCD.
  lcd.setCursor(0, 0);
  lcd.print("Init SD card... " );
  // make sure that the default chip select pin is set to
  // output, even if you don't use it:
  pinMode(53, OUTPUT);
  
  // see if the card is present and can be initialized:
  if (!SD.begin(chipSelect)) {
    Serial.println("Card failed, or not present");
    lcd.setCursor(0, 1);
  //lcd.print("****++++----@@@@");  //16 char - so you wipe out the line before
    lcd.print("SD Card failed  ");
    // don't do anything more:
    return;
  }
  Serial.println("card initialized");
  lcd.setCursor(0, 0);
  lcd.print("SD Card ready   ");
  
  // open the file. note that only one file can be open at a time,
  // so you have to close this one before opening another.
  File dataFile = SD.open("print.dat");

  // if the file is available, write to it:
  if (dataFile) {
    lcd.setCursor(0, 0);
    lcd.print("Printing...     " );
    lcd.setCursor(0, 1);
    lcd.print("                " );
    lcd.setCursor(0, 1);
    lcd.print(row );
    stepperX.step(xPrintMinSteps-xLoc);
    xLoc = xPrintMinSteps;
    //step y
    stepperY.step(yStepPerStrip*yDirection);
    while (dataFile.position()<dataFile.size()-1) {
      //lcd.setCursor(10, 1);
      //lcd.print("      ");
      lcd.setCursor(10, 1);
      lcd.print(dataFile.position());

      //Serial.write(dataFile.read());
        byte lower = dataFile.read();
        byte upper = dataFile.read();
        step_spray_ink(lower,upper);
    }

    lcd.setCursor(0, 0);
    Serial.println("closing graphics.dat");
    lcd.print("closing file    " );
    dataFile.close();
  // if the file isn't open, pop up an error:
  } else {
    Serial.println("error opening graphics.dat");
    lcd.setCursor(0, 0);
    lcd.print("file error      " );
  } 
}

void calibrate() { // I think there is a error here need to check the loops
  //***********home X & Y**********
  while(yLoc>0){
    stepperY.step(-1);
    yLoc--;
  }
  while(xLoc>0){
    stepperX.step(-1);
    xLoc--;
  }
  for(int i = 0;i<8;i++){
    for(int j = 0;j<11;j++){
      step_spray_ink(0xAA,0x0A);
    }
    step_spray_ink(0xFF,0x0F);
  }
  step_spray_ink(0x00,0xF0);  //new line
  for(int i = 0;i<12;i++){
    step_spray_ink(0x10,0x00);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x20,0x00);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x40,0x00);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x80,0x00);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x01,0x01);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x02,0x02);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x04,0x04);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x08,0x08);
  }
  step_spray_ink(0x00,0xF0);  //new line
  for(int i = 0;i<12;i++){
    step_spray_ink(0x80,0x00);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x40,0x00);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x20,0x00);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x10,0x00);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x08,0x08);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x04,0x04);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x02,0x02);
  }
  for(int i = 0;i<12;i++){
    step_spray_ink(0x01,0x01);
  }
  step_spray_ink(0x00,0xF0);  //new line
  for(int i = 0;i<8;i++){
    for(int j = 0;j<11;j++){
      step_spray_ink(0x55,0x05);
    }
    step_spray_ink(0xFF,0x0F);
  }
  step_spray_ink(0x00,0xF0);  //new line
}

void purge() {
  //***********home X & Y**********
  while(yLoc>0){
    stepperY.step(-1);
    yLoc--;
  }
  while(xLoc>0){
    stepperX.step(-1);
    xLoc--;
  }
  for(int i = 0;i<purgeTimes;i++){
    spray_ink(0xFF,0xFF);
  }
}

void step_spray_ink( byte lower, byte upper) {
  if (upper == 0xf0){
    //step x, reverse y, step y
    stepperX.step(xStepPerPass);
    xLoc = xLoc + xStepPerPass;
    yDirection=-yDirection;
    stepperY.step(yStepPerStrip*yDirection);
    yLoc = yLoc + yStepPerStrip*yDirection;
    row++;
    lcd.setCursor(0, 1);
    lcd.print(row );
  } else {
    for(int i = 0; i<saturation; i++){
      spray_ink(lower, upper);
    }
    stepperY.step(yStepPerStrip*yDirection);
    yLoc = yLoc + yStepPerStrip*yDirection;
  }
}

void spray_ink( byte lower_nozzles, byte upper_nozzles) {
//lower_nozzles = 1-8  - PORTA
  for(int i = 0; i <= 7; i++){
    byte mask = 1<<i;
    if(lower_nozzles & mask){
      PORTA |= mask; delayMicroseconds(5);
      PORTA &= ~mask; delayMicroseconds(1);
    }
  } 
//upper_nozzles = 9-12 - PORTC using pins 4,5,6,7 so 4-7 not 0-3
  for(int i = 4; i <= 7; i++){
    byte mask = 1<<i;
    if(upper_nozzles<<3 & mask){
      PORTC |= mask; delayMicroseconds(5);
      PORTC &= ~mask; delayMicroseconds(1);
    }
  }
//wait to be sure we don't try to fire nozzles to fast and burn them out
  delayMicroseconds(800); 
}

