function [h arclength] = arc(X1, X2, X3)



% function [h arclength] = arc(X1, X2, X3)
%
% This is a MATLAB function that draws an arc with parameters X1 X2 X3.
% This mimics the Eagle PCB (http://cadsoft.de/) clockwise arc command. 
% X1 is the coordinates of the start point of the arc X1 = [x1; y1].
% X2 is the point on the polar opposite of X1.  
% The center lies at the midpoint of the line X1-X2.
% X3 defines the endpoint of the arc.
% h is the handle to the graphics object.
% arclength is the length of the arc.
%
% Version 1.0 Initial Release - Aug 31 2011
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  
%  Copyright 2010, 2011 Matt Moses   
%  mmoses152 at gmail dot com
% 
%  This file is part of flatcoilwind. 
%
%  flatcoilwind is free software: you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation, either version 3 of the License, or
%  (at your option) any later version.
%
%  flatcoilwind is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with flatcoilwind.  If not, see <http://www.gnu.org/licenses/>.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

global fileOutputEnable
global fileKicadEnable
global fidOut
global kicadLayer
global kicadWidth
global arcResolution

% the number of line segments per circle is ceil(res*2*pi)
res = arcResolution; 

xcen = (X1(1) + X2(1)) / 2;
ycen = (X1(2) + X2(2)) / 2;
r = sqrt((X1(1) - X2(1))^2 + (X1(2) - X2(2))^2)/2;

startAngle = atan2(X1(2) - ycen, X1(1) - xcen);
endAngle = atan2(X3(2) - ycen, X3(1) - xcen);

% atan2 returns a value from -pi to pi
% there are several possibilities to choose from
% when we go to calculate the angular length of the arc.
% Remember, this is a CLOCKWISE arc, so we start from the
% more positive angle and go in the negative direction.  But things 
% get a little cumbersome when we have to cross zero.
%
% What follows is just a naive way to solve this.  We identify seven cases.
% There is an accompanying sketch that takes some of the confusion out of
% this mess.

% case 1
if (startAngle > endAngle) && (endAngle >= 0)
    n = 1 + ceil (res * (startAngle - endAngle));
    t = linspace(startAngle, endAngle, n);
    arclength = r*(startAngle - endAngle);
end

% case 2
if (endAngle > startAngle) && (startAngle >= 0)
    n = 1 + ceil (res * (2*pi + startAngle - endAngle));
    t = linspace(startAngle, endAngle-2*pi, n);
    arclength = r*(2*pi + startAngle - endAngle);
end

% case 3
if (startAngle >= 0) && (endAngle < 0)
    n = 1 + ceil (res * (startAngle - endAngle));
    t = linspace(startAngle, endAngle, n);
    arclength = r*(startAngle - endAngle);
end

% case 4
if (endAngle >= 0) && (startAngle < 0)
    n = 1 + ceil (res * (2*pi + startAngle - endAngle));
    t = linspace(startAngle, endAngle - 2*pi, n);
    arclength = r*(2*pi + startAngle - endAngle);
end

% case 5
if (startAngle < endAngle) && (endAngle < 0)
    n = 1 + ceil (res * (2*pi + startAngle - endAngle));
    t = linspace(startAngle, endAngle - 2*pi, n);
    arclength = r*(2*pi + startAngle - endAngle);
end
    
% case 6
if (endAngle < startAngle) && (startAngle < 0)
    n = 1 + ceil (res * (startAngle - endAngle));
    t = linspace(startAngle, endAngle, n);
    arclength = r*(startAngle - endAngle);
end

% case 7
if (startAngle == endAngle)
    n = 2;
    t = linspace(startAngle, endAngle, n);
    arclength = 0;
end    

xx = xcen + r*cos(t);
yy = ycen + r*sin(t);
h = line(xx,yy);

if (fileOutputEnable)
    lineout = sprintf('arc (%0.4f %0.4f) (%0.4f %0.4f) (%0.4f %0.4f);', X1(1), X1(2), X2(1), X2(2), X3(1), X3(2));
    fprintf(fidOut, '%s\n', lineout);
end

if (fileKicadEnable)
	for i = 1:length(xx)-1
		if (kicadLayer == 0)
			lineout = sprintf('DS %0.0f %0.0f %0.0f %0.0f %0.0f 0', 10000*xx(i), 10000*yy(i), 10000*xx(i+1), 10000*yy(i+1), kicadWidth);
			fprintf(fidOut, '%s\n', lineout);
		else
			lineout = sprintf('DS %0.0f %0.0f %0.0f %0.0f %0.0f 15', 10000*xx(i), 10000*yy(i), 10000*xx(i+1), 10000*yy(i+1), kicadWidth);
			fprintf(fidOut, '%s\n', lineout);
		end
	end
end
    