% flatcoilwind.m
%
% This is a MATLAB script that automatically generates flat coil patterns
% and corresponding magnet locations for a 2-phase brushless DC motor that
% can operate as a stepper motor.  The patterns can be displayed
% graphically and/or converted into a script file that can be interpreted
% by the PCB layout progam EAGLE (http://cadsoft.de/).  Alternatively,
% KiCad (http://www.lis.inpg.fr/realise_au_lis/kicad/) format can be used. 
%
% Version 1.0 Initial Release - Aug 31 2011
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  
%  Copyright 2010, 2011 Matt Moses   
%  mmoses152 at gmail dot com
% 
%  This file is part of flatcoilwind. 
%
%  flatcoilwind is free software: you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation, either version 3 of the License, or
%  (at your option) any later version.
%
%  flatcoilwind is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with flatcoilwind.  If not, see <http://www.gnu.org/licenses/>.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

close all
clear all

global fileOutputEnable
global fileKicadEnable
global fidOut
global kicadLayer
global kicadWidth
global arcResolution

%------------------------------------------------------------------------
%------------------------------------------------------------------------
% User-specified parameters.
% You can adjust these parameters to generate different coil patterns.
%
% set fileOutEnable = 1 to write an EAGLE script file, 0 otherwise
% set fileKicadEnable = 1 to write a Kicad file, don't set both to one
% at the same time. It will screw things up.
fileOutputEnable = 0;
fileKicadEnable = 0;

outputpath = 'C:\flatcoilwind\'; % PATH for output file

arcResolution = 5; % number of segments per arc
kicadWidth = 120; % width of segment
n = 15; % number of 'turns' per segment
m = 11; % number of segments (or wedges)
r1 = 1.375; % radius of outermost inner trace arc
r2 = 1.625; % radius of innermost outer trace
r2tab = 1.2*r2; % radius of innermost outer trace for wedge with entry point
rMag = 0.125; % radius of magnet
rmagHole = 0.255/2; % radius of the clearance hole for magnet
rencHole1 = 0.0625; % radius of optical encoder hole in coil plate
rencHole2 = 0.05; % radius of optical encoder hole in magnet plate
renc = 0.8; % radius to optical hole ring
rshaft = 0.2; % bolt circle for "output shaft vias"
nshaft = 8; % number of "output shaft vias"
%------------------------------------------------------------------------
%------------------------------------------------------------------------

% n = 5; % number of 'turns' per segment
% m = 3; % number of segments (or wedges)
% r1 = 1.375; % radius of outermost inner trace arc
% r2 = 1.625; % radius of innermost outer trace
% r2tab = 1.2*r2; % radius of innermost outer trace for wedge with entry point
% rMag = 0.125; % radius of magnet
% rmagHole = 0.255/2; % radius of the clearance hole for magnet
% rshaft = 0.2; % bolt circle for "output shaft vias"
% nshaft = 8; % number of "output shaft vias"


% This is the angular offset from zero (horizontal) in units of radians.  
% The offset between top and bottom phases is always a quarter cycle, 
% and is set in the main loop below.
mainOffset = -45 * pi / 180 - pi/4/m;

%------------------------------------
% COLORS
% You can set colors for the graphical plots here.  
% This is useful for making certain structures
% visible or not visible depending on how you want them to display.  
% This has no effect on the EAGLE script files.
%
black = [0 0 0];
white = [1 1 1];
red = [1 0 0];
green = [0 1 0];
blue = [0 0 1];
yellow = [1 1 0];
violet = [1 0 1];
cyan = [0 1 1];
gray = [0.5 0.5 0.5];
lightgray = [0.8 0.8 0.8];
transparent = 99;

circleColor = black;
topColor = red;
bottomColor = blue;
topPhaseColor = cyan;
bottomPhaseColor = yellow;
magnetColor = black;
shaftColor = black;
%------------------------------------

% angle between radial traces
a1 = pi / n / m;

% counter for total length 
traceLength = 0;

% turn hold on for MATLAB plotting
hold on

% plot the circles in MATLAB
if (circleColor ~= transparent)
    circ2 = circle(0,0,r1);
    set(circ2, 'Color', circleColor)
    circ3 = circle(0,0,r2);
    set(circ3, 'Color', circleColor)
end

% This next part opens the script file for EAGLE for writing, and sets up
% the layer and trace width.  NOTE: When switching between top and bottom
% layers, make sure to change both the copper (either 1 or 16) and
% silkscreen (either 21 or 22).
%
% Some important EAGLE layers:
% 1 Top (copper)
% 16 Bottom (copper) 
% 20 Dimension
% 21 tPlace (top silkscreen)
% 22 bPlace (bottom silkscreen)
%
% SET OUTPUT FILENAME AND PATH
if (fileOutputEnable)
    titleFrag = datestr(now,'dd-mmm-yyyy_HH.MM.SS');
    fileOut = [outputpath 'flatcoil_' titleFrag '.scr'];
    fidOut = fopen(fileOut,'w'); % open output file
    lineout = ['BRD:']
    fprintf(fidOut, '%s\n', lineout);
end

if (fileKicadEnable)
    titleFrag = datestr(now,'dd-mmm-yyyy_HH.MM.SS');
    fileOut = [outputpath 'flatcoil_' titleFrag '.emp'];
    fidOut = fopen(fileOut,'w'); % open output file
    line1 = ['PCBNEW-LibModule-V1'];
    line2 = ['# encoding utf-8'];
	line3 = ['$INDEX'];
	line4 = ['CoilTest2'];
	line5 = ['$EndINDEX'];
	line6 = ['$MODULE CoilTest2'];
    fprintf(fidOut, '%s\n', line1);
    fprintf(fidOut, '%s\n', line2);
    fprintf(fidOut, '%s\n', line3);
    fprintf(fidOut, '%s\n', line4);
    fprintf(fidOut, '%s\n', line5);
    fprintf(fidOut, '%s\n', line6);
end


% Start making traces.  The first i-loop does the first n-1 wedges, which are all
% pretty much just the same, except rotated by a phase angle.
% 
% The main outside loop runs twice - once for top, once for bottom.
for Lcount = 1:2
    if (Lcount == 1)
    	kicadLayer = 0;
        traceColor = topColor;
        % SET LAYER FOR TOP COPPER
        if (fileOutputEnable)
            lineout = ['layer 1;']
            fprintf(fidOut, '%s\n', lineout);
        end
    else
    	kicadLayer = 15;
        traceColor = bottomColor;
        % SET LAYER FOR BOTTOM COPPER
        if (fileOutputEnable)
            lineout = ['layer 16;'];
            fprintf(fidOut, '%s\n', lineout);
        end
    end
    %
    % SET COPPER TRACE WIDTH
    % trace dimensions are assumed to be in INCHES
    traceWidth = .015; % this used below for resistance calculation
    traceThickness = 0.0014; % for 1oz copper
    if (fileOutputEnable)
        lineout = ['change width 0.015;'];
        fprintf(fidOut, '%s\n', lineout);
    end

    segOffset = 0;
    for j = 1:m-1
        for i = 1:n
            ai1 = mainOffset + segOffset + (i-1)*a1 + a1/2;
            ai2 = mainOffset + segOffset + 2*pi/m - a1/2 - (i-1)*a1;
            aRP = mainOffset + segOffset + pi/m ;
            
            % This sets the spacing between the outside loops
            if (i==1)
                outstep = a1*r2 / 2 * 0.95;
            else
                outstep = a1*r1;
            end
            
            % The spacing is slightly different for the set of tab loops
            if (i==2)
                outstepTab = a1*r2tab / 2 * 0.95;
            else
                outstepTab = outstep;
            end
            
            % rF is circle radius for outside arc
            if (i==1)
                rF = r2 + i * outstep;
            else
                rF = r2 + (i-1) * outstep + a1*r2 / 2 * 0.95;
            end
            
            % outside radius of tab set of arcs
            if (i==2)
                rFtab = r2tab + (i-1) * outstepTab;
            else
                rFtab = r2tab + (i-2) * outstepTab + a1*r2tab / 2 * 0.95;
            end
            
            % rI is radius for inside circles
            rI = r1 * cos( (n-i)*a1 + a1/2 );
            
            % make first radial trace 
            if (traceColor ~= transparent)
                if (j == 1)
                    [trace1  lineLen] = lineL([rI*cos(ai1) r2tab*cos(ai1)],[rI*sin(ai1) r2tab*sin(ai1)]);
                else
                    [trace1  lineLen] = lineL([rI*cos(ai1) r2*cos(ai1)],[rI*sin(ai1) r2*sin(ai1)]);
                end
                set(trace1, 'Color', traceColor);
                traceLength = traceLength + lineLen;
            end
            
            % make second radial trace 
            if (traceColor ~= transparent)
                [trace2  lineLen] = lineL([rI*cos(ai2) r2*cos(ai2)],[rI*sin(ai2) r2*sin(ai2)]);
                set(trace2, 'Color', traceColor);
                traceLength = traceLength + lineLen;
            end
        
            % make inside arc 
            if (traceColor ~= transparent)
                [arcIn arcLen] = arc([rI*cos(ai1) rI*sin(ai1)], [2*r1*cos(aRP)-rI*cos(ai1) 2*r1*sin(aRP)-rI*sin(ai1)] ,  [rI*cos(ai2) rI*sin(ai2)]);
                set(arcIn, 'Color', traceColor);
                traceLength = traceLength + arcLen;
            end
            
            % make outisde arc
            if (traceColor ~= transparent)
                % filAngle is the angle the fillet extends
                filAngle = atan( ((rF/r2)^2 - 1)/2/(rF/r2));
                filAngleTab = atan( ((rFtab/r2tab)^2 - 1)/2/(rFtab/r2tab));
                
                % filRad is the radius of the fillet
                filRad = r2 * ((rF/r2)^2 - 1)/2/(rF/r2);
                filRadTab = r2tab * ((rFtab/r2tab)^2 - 1)/2/(rFtab/r2tab);
                
                % this is the outer part of the arc
                [arcOut arcLen] = arc([rF*cos(2*mainOffset + j*4*pi/m - ai2 - filAngle) rF*sin(2*mainOffset + j*4*pi/m - ai2 - filAngle)], ...
                             [0-rF*cos(2*mainOffset + j*4*pi/m - ai2 - filAngle) 0-rF*sin(2*mainOffset + j*4*pi/m - ai2 - filAngle)], ...
                             [rF*cos(ai2+filAngle) rF*sin(ai2+filAngle)]);
                set(arcOut, 'Color', traceColor);                 
                traceLength = traceLength + arcLen;
                
                % this IF block is needed because the first set of fillets
                % goes to the "tab" radius, not the normal radius
                if (j == 1)
                    lastFilPoint1 = [rFtab*cos(ai1-filAngleTab), rFtab*sin(ai1-filAngleTab)];
                    firstFilPoint1 = [r2tab*cos(ai1), r2tab*sin(ai1)];
                    cenFil1 = firstFilPoint1 + filRadTab*[cos(ai1 - pi/2) sin(ai1 - pi/2)];
                else
                    lastFilPoint1 = [rF*cos(ai1-filAngle), rF*sin(ai1-filAngle)];
                    firstFilPoint1 = [r2*cos(ai1), r2*sin(ai1)];
                    cenFil1 = firstFilPoint1 + filRad*[cos(ai1 - pi/2) sin(ai1 - pi/2)];
                end
                    
                % this IF block is just to not put a fillet on the very
                % first radial trace
                if (i > 1) || (j > 1)
                    [filOut1 arcLen] = arc(firstFilPoint1, ...
                          2*cenFil1 - firstFilPoint1, ...
                          lastFilPoint1);
                    set(filOut1, 'Color', traceColor);
                    traceLength = traceLength + arcLen;
                end
                        
                % fillet coordinates for second fillet          
                firstFilPoint2 = [rF*cos(ai2+filAngle), rF*sin(ai2+filAngle)];
                lastFilPoint2 = [r2*cos(ai2), r2*sin(ai2)];
                cenFil2 = lastFilPoint2 - filRad*[cos(ai2- pi/2) sin(ai2- pi/2)];
                
                [filOut2 arcLen] = arc(firstFilPoint2, ...
                          2*cenFil2 - firstFilPoint2, ...
                          lastFilPoint2);
                set(filOut2, 'Color', traceColor);
                traceLength = traceLength + arcLen;
            end; % (end make outisde arc)
        end
        segOffset = segOffset + 2*pi/m;
    end

    % Now make the last wedge, which has the outside arcs a little bit
    % different than in the loop above.
    for i = 1:n
        ai1 = mainOffset + segOffset + (i-1)*a1 + a1/2;
        ai2 = mainOffset + segOffset + 2*pi/m - a1/2 - (i-1)*a1;
        aRP = mainOffset + segOffset + pi/m ;
        
        if (i==1)
            outstep = a1*r2 / 2 * 0.95;
        else
            outstep = a1*r1;
        end
        
        % The spacing is slightly different for the FINAL set of tab loops
        if (i==1)
            outstepTab = a1*r2tab / 2 * 0.95;
        else
            outstepTab = outstep;
        end
        
        % rF is circle radius for outside arc
        if (i==1)
            rF = r2 + i * outstep;
        else
            rF = r2 + (i-1) * outstep + a1*r2 / 2 * 0.95;
        end
            
        % outside radius of tab set of arcs
        if (i==1)
            rFtab = r2tab + i * outstepTab;
        else
            rFtab = r2tab + (i-1) * outstepTab + a1*r2tab / 2 * 0.95;
        end
        
        rI = r1 * cos( (n-i)*a1 + a1/2 );

        % final - first radial trace
        if (traceColor ~= transparent)
            [trace1  lineLen] = lineL([rI*cos(ai1) r2*cos(ai1)],[rI*sin(ai1) r2*sin(ai1)]);
            set(trace1, 'Color', traceColor); 
            traceLength = traceLength + lineLen;
        end;
    
        % final - second radial trace 
        if (traceColor ~= transparent)
            [trace2  lineLen] = lineL([rI*cos(ai2) (r2tab)*cos(ai2)],[rI*sin(ai2) (r2tab)*sin(ai2)]);
            set(trace2, 'Color', traceColor); 
            traceLength = traceLength + lineLen;
        end
        
        % final - inside arc 
        if (traceColor ~= transparent)
            [arcIn arcLen] = arc([rI*cos(ai1) rI*sin(ai1)], [2*r1*cos(aRP)-rI*cos(ai1) 2*r1*sin(aRP)-rI*sin(ai1)] ,  [rI*cos(ai2) rI*sin(ai2)]);
            set(arcIn, 'Color', traceColor); 
            traceLength = traceLength + arcLen;
        end
             
        % final - outisde arc 
        if (traceColor ~= transparent)
            % filAngle is the angle the fillet extends
            filAngle = atan( ((rF/r2)^2 - 1)/2/(rF/r2));
                
            % filRad is the radius of the fillet
            filRad = r2 * ((rF/r2)^2 - 1)/2/(rF/r2);
                    
            % the Tab values are for the tab wedge
            filAngleTab = atan( ((rFtab/r2tab)^2 - 1)/2/(rFtab/r2tab));                
            filRadTab = r2tab * ((rFtab/r2tab)^2 - 1)/2/(rFtab/r2tab);
                    
            lastFilPoint1 = [rF*cos(ai1-filAngle), rF*sin(ai1-filAngle)];
            firstFilPoint1 = [r2*cos(ai1), r2*sin(ai1)];
            cenFil1 = firstFilPoint1 + filRad*[cos(ai1 - pi/2) sin(ai1 - pi/2)];
                
            [filOut1 arcLen] = arc(firstFilPoint1, ...
                2*cenFil1 - firstFilPoint1, ...
                lastFilPoint1);
            set(filOut1, 'Color', traceColor);
            traceLength = traceLength + arcLen;
            
            % fillet coordinates for second fillet     
            if (i<n)              
                % new code with tab offset - Aug 19
                filAngle2 = atan( (((rFtab)/r2tab)^2 - 1)/2/((rFtab)/r2tab));
                filRad2 = r2tab* (((rFtab) / r2tab)^2 - 1)/2/((rFtab)/r2tab);
                
                firstFilPoint2 = [(rFtab)*cos(ai2+filAngle2), (rFtab)*sin(ai2+filAngle2)];
                lastFilPoint2 = [r2tab*cos(ai2), r2tab*sin(ai2)];
                cenFil2 = lastFilPoint2 - filRad2*[cos(ai2- pi/2) sin(ai2- pi/2)];
                              
                [filOut2 arcLen] = arc(firstFilPoint2, ...
                    2*cenFil2 - firstFilPoint2, ...
                    lastFilPoint2);
                set(filOut2, 'Color', traceColor);
                traceLength = traceLength + arcLen;
                
                arc2Start = [(rFtab)*cos(ai1-filAngle2+a1-segOffset), (rFtab)*sin(ai1-filAngle2+a1-segOffset)];
                
                % this is the outer part of the arc
                [arcOut arcLen] = arc(arc2Start, -arc2Start, firstFilPoint2);
                set(arcOut, 'Color', traceColor); 
                traceLength = traceLength + arcLen;
            end
            
        end % (end final - outisde arc)
    end
    
    % This part generates the phase markings in SILKSCREEN layer
    % Set TOP or BOTTOM SILKSCREEN here.
    if (Lcount == 1)
        traceColor = topPhaseColor;
        % SET LAYER FOR TOP SILKSCREEN
        if (fileOutputEnable)
            lineout = ['layer 21;']
            fprintf(fidOut, '%s\n', lineout);
        end
    else
        traceColor = bottomPhaseColor;
        % SET LAYER FOR BOTTOM SILKSCREEN
        if (fileOutputEnable)
            lineout = ['layer 22;']
            fprintf(fidOut, '%s\n', lineout);
        end
    end
    
    % SET SILKSCREEN LINE WIDTH
    % and make magnet circles
    if (fileOutputEnable)
        lineout = ['change width 0.016;']
        fprintf(fidOut, '%s\n', lineout);
        %
        %lineout = ['circle (0 0) (0 ' num2str(r1, '%0.4f') ');'];
        lineout = sprintf('circle (0 0) (0 %0.4f);', r1);
        fprintf(fidOut, '%s\n', lineout);
        %
        %lineout = ['circle (0 0) (0 ' num2str(r2, '%0.4f') ');'];
        lineout = sprintf('circle (0 0) (0 %0.4f);', r2);
        fprintf(fidOut, '%s\n', lineout);
    end
    
    % Make phaselines
    for i = 1:2*m
        if (traceColor ~= transparent)
        	kicadToggleFlag= 0;
        	if (fileKicadEnable)
        		fileKicadEnable= 0;
        		kicadToggleFlag= 1;
        	end	
            [phaseLine  lineLen] = lineL([r1*cos(mainOffset + (i-1)*pi/m) r2*cos(mainOffset + (i-1)*pi/m)], [r1*sin(mainOffset + (i-1)*pi/m)  r2*sin(mainOffset + (i-1)*pi/m)]);
            set(phaseLine, 'Color', traceColor);
            if (kicadToggleFlag)
            	fileKicadEnable = 1;
        	end
        end
    end
    
    % increment mainOffset for phase difference 
    % between TOP and BOTTOM coils
    % This is a 'quarter cycle' phase difference between top and bottom
    mainOffset = mainOffset + pi/2/m;
end

% This section puts in 4 holes to be used as optical paths for synchronous
% operation.  This is OPTIONAL
for i = 1:4
    encPhi = (i-1)* 2*pi/m / 4 + (i-1)*(2*pi/m) ;
    encCirc1 = circle(renc*cos(encPhi), renc*sin(encPhi), rencHole1);
    set(encCirc1, 'Color', magnetColor);
    if (fileOutputEnable)
        lineout = sprintf('via (%0.4f %0.4f);', renc*cos(encPhi), renc*sin(encPhi));
        fprintf(fidOut, '%s\n', lineout);
    end
end
    
% Add center hole last and Close coil file
if (fileOutputEnable)
    lineout = sprintf('via (%0.4f %0.4f);', 0, 0);
    fprintf(fidOut, '%s\n', lineout);
    fclose(fidOut);
end

if (fileKicadEnable)
	line1= ['$EndMODULE  CoilTest2'];
	line2= ['$EndLIBRARY'];
	fprintf(fidOut, '%s\n', line1);
	fprintf(fidOut, '%s\n', line2);
    fclose(fidOut);
end

% This part calculates estimated resistance and motor constant
% NOTE: the traceLength used in resistance calculation is the total
% length of everything plotted!  To get an accurate number make sure you
% have plotted only the trace(s) you want to estimate!

% copper resistivity at 20C
CuRho = 1.68E-08; % units of Ohm-meters

% It is assumed traceLength and trace width are in inches.
% Since we want to do an actual physical calculation the first thing
% we need to do is convert lengths to meters...
traceThicknessM = traceThickness * 2.54/100;
traceWidthM = traceWidth * 2.54/100;
traceLengthM = traceLength * 2.54/100;

disp('traceResistance estimates the sum of ALL TRACES PLOTTED!');
disp('Make sure you are only plotting what you want to estimate.');
traceResistance = CuRho * traceLengthM / traceThicknessM / traceWidthM

% motor performance factor
% This is actually an attempt at calculating the real motor constant, but
% so many assumptions are made that we call it instead the 'motor
% performance factor'.  It is meant to be used in comparing different
% geometries and less as an accurate quantitative prediction.  Interpret it
% as peak Volts-per-speed (units of volt*second) or 
% peak Torque-per-current (units of newton*meter/amp).  
% Remember that volt*seconds and newton*meter/amps are the same thing.  
% The performance factor assumes the magnets are wedge-shaped with size 
% roughly equal to the circular magnets actually used.  It will somewhat
% overestimate the actual motor constant.
% 
% All this really is, is finding the force on a current carrying wire in a
% uniform magnetic field.  F = I*L <cross> B
% where F is force, I is current, L is length, B is magnetic field,
% see
% http://en.wikipedia.org/wiki/Laplace_force#Force_on_a_current-carrying_wi
% re.
% Then we account for moment arm, count up contribution from all 
% the "wires" under all the magnets, and since we want motor constant,
% divide by current.  So we have something like:
% Km = F * r_moment / I = r_moment * L * B * n_turns * n_magnets

% radius to center of magnets
rMagcen = (r1+r2)/2;

% mag alpha is half the angle subtended by a magnet
magAlpha = asin(rMag/rMagcen);
turnsPerMag = 2 * floor(2*magAlpha / a1); % this accounts for top AND bottom phases
turnLengthM = 2 * rMag * 2.54/100; % length of a 'turn' under a wedge-magnet in meters
B = 0.4; % a uniform magnetic field is assumed, and based on earlier work we pick 0.4 Tesla for B in the airgap

% note the number of magnets is 2*m.  This is motor constant in units of
% N*m/amp or V*s.  Also compute in ozf*in/amp to compare with commonly reported numbers for, e.g. RC servos  
% The (pi/4) factor is just a naiive correction from 'square' to 'circle'
% to close approximate trace lengths under the circular magnet vs. a
% wedge-shaped magnet.
k_M = 2*m * (rMagcen*2.54/100) * B * turnLengthM * turnsPerMag * (pi/4);
disp('motor performance factor - warning - see caveats in comments about how much you can trust this number')
k_M_OzInPerAmp = 141.612 * k_M 
traceSpacing = a1*r1

% --------------------
% Now make script file for magnet plate
if (fileOutputEnable)
    fileOut2 = [outputpath 'magnetplate_' titleFrag '.scr'];
    fidOut2 = fopen(fileOut2,'w'); % open output file
    % setup
    lineout = ['BRD:']
    fprintf(fidOut2, '%s\n', lineout);
    lineout = ['change drill 0.255;'];
    fprintf(fidOut2, '%s\n', lineout);
    lineout = ['change shape round;'];
    fprintf(fidOut2, '%s\n', lineout);
end

% plot the magnet circles on the MATLAB plot
% and place holes for the magents in the EAGLE script
for i = 1:2*m
    if (magnetColor ~= transparent)
        magcirc = circle(rMagcen*cos((i-1)*pi/m), rMagcen*sin((i-1)*pi/m), rMag);
        set(magcirc, 'Color', magnetColor);
        if (mod(i,2) ~= 0)
            encCirc2 = circle(renc*cos((i-1)*pi/m), renc*sin((i-1)*pi/m), rencHole2);
            set(encCirc2, 'Color', magnetColor);
        end
    end
    if (fileOutputEnable)
        lineout = sprintf('via (%0.4f %0.4f);', rMagcen*cos((i-1)*pi/m), rMagcen*sin((i-1)*pi/m));
        fprintf(fidOut2, '%s\n', lineout);
        if (mod(i,2) ~= 0)
            lineout = sprintf('via (%0.4f %0.4f);', renc*cos((i-1)*pi/m), renc*sin((i-1)*pi/m));
            fprintf(fidOut2, '%s\n', lineout);
        end
    end
end

% place holes for output shaft in EAGLE script
if (fileOutputEnable)
    lineout = ['change drill 0.03937;'];
    fprintf(fidOut2, '%s\n', lineout);
end
for i = 1:nshaft
    if (shaftColor ~= transparent)
        shaftcirc = circle( rshaft*cos((i-1)*2*pi/nshaft), rshaft*sin((i-1)*2*pi/nshaft), .03937/2);
        set(shaftcirc, 'Color', shaftColor);
    end
    if (fileOutputEnable)
        lineout = sprintf('via (%0.4f %0.4f);', rshaft*cos((i-1)*2*pi/nshaft), rshaft*sin((i-1)*2*pi/nshaft));
        fprintf(fidOut2, '%s\n', lineout);
    end
end

% axis equal for nice looking plot
axis equal

% close magnet plate file
if (fileOutputEnable)
    fclose(fidOut2)
end

	