use </usr/share/openscad/MCAD/regular_shapes.scad>;
use </usr/share/openscad/MCAD/shapes.scad>;
// (c) 2012 christophercaswell@gmail.com
// licensed under the Creative Commons - GNU LGPL license.
// http://www.gnu.org/licenses/lgpl-2.1.html
// 
// this is the initial feature set for hand-wound solder-free 3D-PCB library
// extreme beta, at this point, the circuit demo has passed an initial functional test
// on my Up! printer (www.pp3dp.com) -- testing on the MakerBot Replicator pending
// 2012-03-08

min_hole_radius = 1.3; // minimum radius for a well-articulated hole (threading) 
error_tolerance = 1.05;
contact_width = 7; // radius of battery sphere holder
peg_height = 5; // standard height for circuit
board_thickness = 2.0;

/* ------------ END CONSTANT DECLARATION ------------- */

sample_circuit_demo ();
//switch_toggle_to_print() ;



/* ------------ TEST MODULES DEFINED BELOW ------------- */

module sample_circuit_demo () {
    // here is a simple parallel circuit in one board, showing the capabilities 
    // of all major components (battery holder, peg winding, momentary switch,
    // SPST slide switch, and LED holder
    // you'll also need to print the slide switch 'toggle' separately
    
    board_length = 110; 
    board_width = 70;
    
    union() {
	    base_board(board_length,board_width);
	    translate([board_length / 2, board_width *3/4, board_thickness]) battery_holder_watch_stack();
	    peg(70,35, 5);
	    peg(10,board_width* 3/4, 5);
	    translate([board_length - 10, board_width/2, 0]) rotate([0,0,90]) LED_holder_vertical();
	    translate([65,15,0]) rotate([0,0,-90]) SPST_slide_switch(10,10,5);
	    translate([20,20,0]) rotate([0,0,-90]) push_button();
    }
}

module switch_toggle_to_print() {
	rotate([180,0,-90]) SPST_slide_switch_toggle(10,10,5);
}



/* ------------ WORK MODULES DEFINED BELOW ------------- */

module base_board(length, width) {
	difference() {
	translate([0,0,-board_thickness]) cube([length, width, board_thickness*2]);
	translate([length*.05/2,width*.15/2,-board_thickness*2]) cube([length*.95, width*.85, board_thickness*2]);
	}
}

module battery_holder_AA() {
	length = 50;
	width = 20;
	height = 10;
	offset=-1.6; // this moves the bettery springs tighter or looser; the more negative the tighter to hold the batteries

	difference() {
		translate([0,0,height/2]) cube([length, width, height], center=true);
		scale(error_tolerance*[1,1,1]) translate([0,0,height]) AA_battery();
	}

	difference() {
		union() {
			// one spring on either side of the battery
			translate([-length/4-offset, 0,height/2]) rotate([0,0,-90]) battery_spring_sphere();
			translate([length/4+offset, 0,height/2]) rotate([0,0,90])  battery_spring_sphere();
		
			// the cubes holding it up, begging to the board
			translate([-length-offset, 0,6])  cube([2.5,10,12], center=true);
			translate([length+offset, 0,6]) cube([2.5,10,12], center=true);

			// and sloped support
			translate([-length-offset, 0,]) rotate([0,90,90]) rightTriangle(height * 1.4, 13, 10);
			translate([length+offset, 0,]) rotate([0,90,270]) rightTriangle(height * 1.4, 13, 10);

		}

		union() {
			// holes for the thread
			translate([0,0,height * .85])  rotate([90,90,90]) cylinder(h=length*5, r=min_hole_radius, center=true);
			translate([0,0,height * 1.15])  rotate([90,90,90]) cylinder(h=length*5, r=min_hole_radius, center=true);

			// mark positive and negative
			translate([-length - offset/2,0,height + contact_width]) cube([5,2,2], center=true); // neg
			translate([length + offset/2,0,height + contact_width]) cube([5,2,2], center=true); // pos
			translate([length + offset/2,0,height + contact_width]) cube([2,5,2], center=true);
		}
	}
}

module battery_holder_watch_stack() {
	length = 15.5;
	width = 20;
	height = 10;
	offset=-1.6; // this moves the bettery springs tighter or looser; the more negative the tighter to hold the batteries

	difference() {
		translate([0,0,height/2]) cube([length, width, height], center=true);
		scale(error_tolerance*[1,1,1]) translate([0,0,height]) watch_battery_3x_stack();
	}

	difference() {
		union() {
			// one spring on either side of the battery
			translate([-length/4-offset, 0,height/2]) rotate([0,0,-90]) battery_spring_sphere();
			translate([length/4+offset, 0,height/2]) rotate([0,0,90])  battery_spring_sphere();
		
			// the cubes holding it up, begging to the board
			translate([-length-offset, 0,6])  cube([2.5,10,12], center=true);
			translate([length+offset, 0,6]) cube([2.5,10,12], center=true);

			// and sloped support
			translate([-length-offset, 0,]) rotate([0,90,90]) rightTriangle(height * 1.4, 13, 10);
			translate([length+offset, 0,]) rotate([0,90,270]) rightTriangle(height * 1.4, 13, 10);

		}

		union() {
			// holes for the thread
			translate([0,0,height * .85])  rotate([90,90,90]) cylinder(h=length*5, r=min_hole_radius, center=true);
			translate([0,0,height * 1.15])  rotate([90,90,90]) cylinder(h=length*5, r=min_hole_radius, center=true);

			// mark positive and negative
			translate([-length - offset/2,0,height + contact_width]) cube([5,2,2], center=true); // neg
			translate([length + offset/2,0,height + contact_width]) cube([5,2,2], center=true); // pos
			translate([length + offset/2,0,height + contact_width]) cube([2,5,2], center=true);
		}
	}
}

module battery_spring_sphere() {
	$fs = 0.2;
	offset = -12;
	wind_gap = 0.6;
	translate([0,-contact_width*1.5,contact_width*0.75])
	difference() {
		translate([0,-contact_width*0.25,0]) sphere(r=contact_width,center=true);
		translate([0,-contact_width*2,0]) cube(contact_width*2*[1,1,1], center=true); // chop off the back
		translate([0,-contact_width*0.25,0]) torus2(contact_width, wind_gap);
	}
}

module LED_holder_vertical() { 
	$fs = 0.2;
	LED_diameter = 5;
	wall_thickness = 3;
	difference(){
		union() {
			resistor_holder();
			translate([0,-LED_diameter/4,8- wall_thickness/4]) cylinder(h=wall_thickness, r=LED_diameter, center=true);
		}
		translate([0,-LED_diameter/4,0]) cylinder(h=30, r=LED_diameter/2, center=true);
		translate([0,-LED_diameter,0]) cylinder(h=30, r=LED_diameter/2, center=true);
	}
}

// This is a standard peg for winding thread around, or winding up as an end point
module peg(x, y, peg_height) {
	$fs = 0.2;
	peg_radius_thin = 2.2;
	peg_radius_thick = 4;
	peg_gap = 1.3; // gap between peg, for winding thread
	slot_width=0.4;
	slot_depth=4;
	num_slots=4;

	translate([x,y,board_thickness + peg_height/2 + peg_gap/4 - peg_gap/2]) // position on top of the board
	difference() {
	union() {
	 	translate([0,0,0]) cylinder(h=peg_height, r2=peg_radius_thick*1, r1=peg_radius_thick*1.13,center=true); // thick trunk of peg
		translate([0,0,peg_height/2 + peg_gap/2]) cylinder(h=peg_gap, r=peg_radius_thin ,center=true); // thin trunk of peg, for wrapping
		translate([0,0,peg_height/2 + peg_gap]) cylinder(h=peg_gap, r=peg_radius_thick*1.15, center=true); // flat part leading to slant
		translate([0,0,peg_height/2 + peg_gap*2]) cylinder(h=peg_gap, r1=peg_radius_thick*1.15, r2=peg_radius_thin, center=true); // slanted top
	}
	union() { // minus slots on the side
		for( i= [0:num_slots] ) {
			rotate(i*360/num_slots, [0,0,1])
			translate([0,peg_radius_thick,0]) 
			union() {
			cube([slot_width,slot_depth,20],center=true);
			translate([0,0,peg_height/2+peg_gap*2+slot_width]) cube([slot_width,peg_radius_thick*2,slot_width*3],center=true);
			}
		}

	}
	}
}

module push_button() {
	button_distance = 2;
	resistor_length = 6;
	union() {
		peg(-resistor_length*2, -resistor_length*1.5, peg_height - button_distance/2); // holder pegs
		peg(resistor_length*2,  -resistor_length*1.5,  peg_height + button_distance/2);  // one and two[ 67.20, 0.00, 4.20 ]
		peg(-resistor_length*2, resistor_length*1.5,  peg_height + button_distance/2); // holder pegs
		peg(resistor_length*2,  resistor_length*1.5,  peg_height - button_distance/2);  // red and blue
	}
}

module resistor_holder() {
    resistor_length = 6;
	union() {
		peg(-resistor_length*1.6, -resistor_length, peg_height); // holder pegs
		peg(resistor_length*1.6,  -resistor_length, peg_height);  // one and two[ 67.20, 0.00, 4.20 ]
		peg(-resistor_length*1.6, resistor_length, peg_height); // holder pegs
		peg(resistor_length*1.6,  resistor_length, peg_height);  // red and blue

		resistor_holder_clip(); // main clip holding the resistor down
		translate([-resistor_length*3.5,0,0]) rotate([0,0,180]) scale([1.25,1,0.65]) resistor_holder_clip(); // secondary clips holding down the leds
		translate([resistor_length*3.5,0,0]) rotate([0,0,180]) scale([1.25,1,0.65]) resistor_holder_clip(); // one and two
	}
}

module resistor_holder_clip() { 
	wall_thickness = 3;
	clip_width = 9;
	clip_height = 8;
	clip_depth = 5;
	translate([0,0,board_thickness/2])
	union() {
		translate([0, clip_depth/2, clip_height - wall_thickness/2]) cube([clip_width, clip_depth, wall_thickness], center=true); // horizontal component
		translate([0, clip_depth, clip_height/2]) cube([clip_width, clip_depth/4, clip_height], center=true); // vertical component
		translate([0,clip_depth*9/8,0]) rotate([0,90,0]) rightTriangle(clip_height, clip_depth/2, clip_width); // supporting right triangle
	}
}

module SPST_slide_switch() {
	$fs = 0.2;
	switch_width = 8;
	peg_radius_thick = 4;
	button_height=2;
	switch_length = 24;
	gap = 1; // between walls /// NOTE parametric fail, don't change this unless you want trouble
	snap_distance = 3.2; // full actuated distance of switch
	snap_tolerance = 1.08;
	off = -snap_distance/2;
	on = snap_distance/2;
	contact_offset = -3.5;

	translate([0,0,switch_width/2 + board_thickness])
	union() {
		//translate([0,on,0]) SPST_slide_switch_toggle();

		// holder
		difference() {
			union() {
				translate([-switch_width - gap*2,0,-gap*2 + gap/2]) cube([switch_width/4,  switch_length*1.25, switch_width - gap*3], center=true); // vertical start
				translate([-switch_width,0,- switch_width/8 + gap]) cube([switch_width/2 + gap,  switch_length*1.25, switch_width /4], center=true); // horizontal hold
				translate([-switch_width - gap*3,0,-switch_width/2]) rotate([90,0,0]) rightTriangle(3,switch_width - gap*3,switch_length*1.25); // support
				// and the mirror:
				mirror([1,0,0]) {
				translate([-switch_width - gap*2,0,-gap*2 + gap/2]) cube([switch_width/4,  switch_length*1.25, switch_width - gap*3], center=true); // vertical start
				translate([-switch_width,0,- switch_width/8 + gap]) cube([switch_width/2 + gap,  switch_length*1.25, switch_width /4], center=true); // horizontal hold
				translate([-switch_width - gap*3,0,-switch_width/2]) rotate([90,0,0]) rightTriangle(3,switch_width - gap*3,switch_length*1.25); // support
				}

				// contact point
				translate( [0,contact_offset,0]) union() {
				translate([0,switch_length ,-switch_width/2-board_thickness]) peg(0,0,4);
				difference() {
					translate([0,switch_length - peg_radius_thick/2,0]) cylinder(r=switch_width * 4/12,h=switch_width * 8/12, center=true); // contact point
					translate([0,switch_length - peg_radius_thick/2,0])  torus2(switch_width*4/12, 0.2); // torus for threading contact
				}
				}

				// extra lead peg, for slack
				translate([0,-switch_length*1.5 ,-switch_width/2-board_thickness]) peg(0,0,4);

			}
			union() {
				translate([switch_width/2,snap_distance/2,0]) cylinder(r=switch_width * .28 * snap_tolerance,h=button_height*2,center=true); // first hole (on)
				translate([switch_width/2,-snap_distance/2,0]) cylinder(r=switch_width * .28 * snap_tolerance,h=button_height*2,center=true); // second hole (off)
				mirror([1,0,0]) {
				translate([switch_width/2,snap_distance/2,0]) cylinder(r=switch_width * .28* snap_tolerance,h=button_height*2,center=true); // first hole (on)
				translate([switch_width/2,-snap_distance/2,0]) cylinder(r=switch_width * .28 * snap_tolerance,h=button_height*2,center=true); // second hole (off)
				}
			}
		}
	}
}

module SPST_slide_switch_toggle() {
	$fs = 0.2;
	switch_width = 8;
	button_height=1.5;
	switch_length = 24;
	gap = 1; // between walls
	snap_tolerance = 1.15;
	num_rows = 0.5; // off by one, both of them (3 means 4)
	num_cols = 0.5; // rows and columns for thumb texture
	step_row = switch_width / (num_rows) * 2;
	step_col = switch_width / (num_cols) * 2;
	min_step = min (step_row,step_col);
	spacing = 0.8; // percent buffer between button pegs

	difference() {
		union() {
			cube([switch_width,  switch_length, switch_width], center=true); // main throw
			translate([0,0,switch_width/2 - switch_width/8]) cube([switch_width*3,  switch_length, switch_width/4], center=true); // slider slot upper
			translate([0,0,-switch_width/2 + switch_width/8]) cube([switch_width*2.15,  switch_length, switch_width * 0.29], center=true); // slider slot lower
			for ( x = [-num_rows/2:1:num_rows/2] ) {
				for ( y = [-num_cols/2:1:num_cols/2]) {
					if(num_cols < 1) { // special acse for 1 --- ugly ugly special case
						translate([0,0,switch_width/2]) cylinder(r=min_step / 3 * spacing,h=button_height,center=true); // thumb pegs
					}
					else {
						translate([x * step_row,y * step_col,switch_width/2]) cylinder(r=min_step / 2 * spacing,h=button_height,center=true); // thumb pegs
					}
				}
			}
			translate([switch_width*1.66,0,switch_width/2 - switch_width/2]) rotate([0,58,0]) cube([switch_width,  switch_length, switch_width/4], center=true); // angled slots
			translate([-switch_width*1.66,0,switch_width/2 - switch_width/2]) rotate([0,-58,0]) cube([switch_width,  switch_length, switch_width/4], center=true); // angled slots
		}
		union() {
			//translate([0,switch_width*1.5 - switch_width/4,0])  cylinder(h=0.85,r=switch_width*7/12,center=true); // cylinder for threading contact
			translate([0,0,-switch_width * 3/12]) cube([switch_width/4,  switch_length*2, switch_width*3/4], center=true); // bottom slot for threading 
			translate([0,switch_length * 6/12,-switch_width * 3/12]) cube([switch_width*2.5,  switch_width, switch_width*3/4], center=true); // other slot for threading
		}
	}

	difference() {
		translate([0,switch_length/2,switch_width*1/8]) cylinder(r=switch_width * 4/12, h=switch_width*3/4, center=true); // contact point
		translate([0,switch_length/2,0])  torus2(switch_width*4/12, 0.2); // torus for threading contact
		translate([0,switch_length/2,0])  rotate([90,0,0]) cylinder(h=15, r=min_hole_radius*.75, center=true);
	}
	// round catches
	translate([switch_width/2,0,0]) cylinder(r=switch_width * .28,h=switch_width/2,center=true); // peg
	translate([-switch_width/2,0,0]) cylinder(r=switch_width * .28,h=switch_width/2,center=true); // peg
	

}



/* ------------ HARDWARE MODULES DEFINED BELOW ------------- */

module AA_battery() {
	$fs = 0.2;
	battery_length = 50;
	battery_width = 14;
	rotate([90,0,0])
	rotate([0,270,0])   // reorient to horizontal
	union() {
		cylinder(h=battery_length-1, r=battery_width/2, center=true); // primary cylinder of battery
		translate([0,0,-battery_length/2]) cylinder(h=1, r=2.5, center=true); // positive led
	}
}

module watch_battery_3x_stack() { // quickly step up to 4.5v ...
	$fs = 0.2;
	battery_length = 15.5;
	battery_width = 11.5;
	rotate([90,0,0])
	rotate([0,270,0])   // reorient to horizontal
	union() {
		cylinder(h=battery_length, r=battery_width/2, center=true); // primary cylinder of battery
		translate([0,0,-battery_length/2]) cylinder(h=1, r=2.5, center=true); // positive led
	}
}



/* ------------ HELPER FUNCTIONS ------------- */

function distance(A,B) = sqrt( (A[0] - B[0])*(A[0] - B[0]) + (A[1] - B[1])*(A[1] - B[1]) );

function midpoint(A,B) = [(A[0]+B[0])/2, (A[1]+B[1])/2];



